
/***********************************************

  FOSSIL.CPP         By Jim Storch
                        3317 Esquarre Ct
  Turbo C++ 3.0         Woodbridge, Va 22193
                        BBS (703) 730-0542

  This is a simple-to-use Fossil Class for
  writing serial programs that interface
  with a fossil driver. It has been
  tested with BNU and X00. I wrote this a
  while back to add to a project and ended
  up never using it. It does work however,
  and I'm releasing it to the Public Domain.

  There are more fossil functions, but these
  were the only ones that interested me at the
  time. As you can seen, I used the asm keyword
  for the interrupt calls, but thats easy to
  replace with int86(), geninterrupt(), etc.

  Hope these help someone. Drop me a line if
  they do. Complaints may be forwarded to > NUL.

  **********************************************/


#include<iostream.h>    // For example main only

#include<dos.h>
#include<string.h>
#include<bios.h>

#define FOSSIL 0x14
class Fossil{
                int     comport;
		int	active;
		int	SetBaud(unsigned int baud);
		int	IsReceiveDataReady();
		int	IsTransmitBufferRoom();
		int	IsTransmitBufferEmpty();
		int	InitializeDriver();
		void	DeinitializeDriver();
		int	TransmitCharNoWait(unsigned char c);
		int	ReceiveCharWithWait();
		void	EnableFlowControl();
	public:
		enum	acks{FALSE=0,TRUE=1,NO=0,YES=1,ERROR=-1,OKAY=1};
			Fossil();
			~Fossil();
		int 	Detect(int port);
		int 	Open(int port, unsigned int baud);
		int 	Open(int port);
		int 	TransmitChar(char c);
		int 	ReceiveChar(char &c);
		void	PurgeOutputBuffer();
		int 	FlushOutputBuffer();
		void 	PurgeInputBuffer();
		int	IsCarrierDetect();
		void	LowerDTR();
		void	RaiseDTR();
        };

Fossil::Fossil()
	{
/*****************************************************************
 You may notice I delibrately avoided a constructor that activates
 the fossil. For my purposes I'd rather test for the fossil using
 Detect() then call Open(). This gives me the chance to switch to
 my own serial routines if Detect() fails.
 ******************************************************************/
	comport=0;
	active=FALSE;
	}

Fossil::~Fossil()
	{
	if(active==TRUE)
		{
		FlushOutputBuffer();
		DeinitializeDriver();
		}
	}

int	Fossil::Open(int port, unsigned int baud)
	{
	comport=port;
	if(ERROR==InitializeDriver())
		return(ERROR);
	active=TRUE;
	SetBaud(baud);
	EnableFlowControl();
	return(OKAY);
	}

int	Fossil::Open(int port)
	{
/*********************************************
 This OPEN() does not change the port speed.
 In some cases this can make life easier, ie
 a door where the port is already hot. You may
 have to experiment.
 *********************************************/
	comport=port;
	if(ERROR==InitializeDriver())
		return(ERROR);
	active=TRUE;
	EnableFlowControl();
	return(OKAY);
	}

int Fossil::TransmitChar(char c)
	{
/********************************************************
 This call will try to get the char into the fossil's
 transmit buffer for approx 5 secs. If nothing has been
 sent by then, return -1. The calling routine should test
 for this and be prepared to terminate.
 ********************************************************/
	int	count=0;
	while(1)
		{
		if(IsTransmitBufferRoom())
			{
			TransmitCharNoWait(c);
			return(OKAY);
			}
		if(count>500)
			return(ERROR);
		delay(10);
		count++;
		}
	}

int Fossil::ReceiveChar(char &c)
	{
/***************************************************************
 This member function return a 1 if a char is avail and 0 if not.
 The ARGUMENT is set to the received char's value
 ****************************************************************/
	if(IsReceiveDataReady())
		{
		c=ReceiveCharWithWait();
		return(TRUE);
		}
	return(FALSE);
	}


int Fossil::ReceiveCharWithWait()
	{
        _AH=0x02;
	_DX=comport;
	asm INT FOSSIL
	return(_AX);
	}

int Fossil::TransmitCharNoWait(unsigned char c)
	{
	_AH=0x0b;
	_DX=comport;
	_AL=c;
	asm INT FOSSIL
	if(_AX==FALSE)
		return(ERROR);
	else
		return(OKAY);
	}

void Fossil::EnableFlowControl()
	{
	_AH=0x0f;
	_AL=0xf3;
	_DX=comport;
	asm INT FOSSIL
	}

void Fossil::PurgeOutputBuffer()
	{
	_AH=0x09;
	_DX=comport;
	asm INT FOSSIL
	}

void Fossil::PurgeInputBuffer()
	{
	_AH=0x0a;
	_DX=comport;
	asm INT FOSSIL
	}

int Fossil::FlushOutputBuffer()
	{
/*********************************************
 This is not the fossil Flush-Output-Buffer.
 The built in one can hang under flow control.
 I prefer a breakout counter.
 *********************************************/
        int count=0;
	while(1)
		{
		if(IsTransmitBufferEmpty())
			return(OKAY);
		if(count>1000)
			return(ERROR);
		delay(10);
		count++;
		}
	}

void Fossil::RaiseDTR()
	{
	_AH=0x06;
	_DX=comport;
	_AL=TRUE;
	asm INT FOSSIL
	}

void Fossil::LowerDTR()
	{
	_AH=0x06;
	_DX=comport;
	_AL=FALSE;
	asm INT FOSSIL
	}

int Fossil::InitializeDriver()
	{
	_AH=0x04;
	_DX=comport;
	_BX=0;
	asm INT FOSSIL
	if(_AX==0x1954)
		return(OKAY);
	else
		return(ERROR);
	}

void Fossil::DeinitializeDriver()
	{
	_AH=0x05;
	_DX=comport;
	asm INT FOSSIL
	}

int	Fossil::Detect(int port)
	{
/****************************************************************
 This check looks for the word 1954 (hex) at offset 6 into
 the INT 14h handler. This call FAILS under Desqview. DV is
 rather hostile to serial programming. I presume that DV collects
 all the interrupts, so INT 14h points to the DV chain. This is
 pretty annoying, since I'd rather establish that the fossil is
 indeed there prior to making ANY calls. The other method to testing
 for the fossil is to call InitializeDriver() and see if it returns
 1954h in AX. Unfortunately, this is the PS/2 Bios Set Com parms!
 ****************************************************************/
	unsigned int seg;
	unsigned int ofs;
	unsigned int fos;
	ofs=peek(0,FOSSIL*4);
	seg=peek(0,(FOSSIL*4)+2);
	fos=peek(seg,ofs+6);
/*****************************
 Okay, we got off easy, return
 *****************************/
	if(fos==0x1954)
		return(TRUE);
 /***************************************************
  Nope, now we have to call Init and see what he says
  ***************************************************/
	_AH=0x04;
	_DX=port;
	_BX=0;
	asm INT FOSSIL
	if(_AX==0x1954)
		{
		/*******************************
		 Yes, must be running DV.
		 Now call Deinit just for safety
		 *******************************/
		_AH=0x05;
		_DX=port;
		asm INT FOSSIL
		return(TRUE);
		}
	return(FALSE);
	}

int	Fossil::SetBaud(unsigned int baud)
	{
	unsigned char	b=(2<<5);  //start 300
	if (baud>300) 	b=(3<<5);  //then 600     Ugly, but will handle
	if (baud>600) 	b=(4<<5);  //then 1200    weird numbers.
	if (baud>1200) 	b=(5<<5);  //then 2400
	if (baud>2400) 	b=(6<<5);  //then 4800
	if (baud>4800) 	b=(7<<5);  //then 9600
	if (baud>9600)	b=(0<<5);  //then 19200   why shift, eh?
	if (baud>19200)	b=(1<<5);  //stop 38400
        b|=3;                      //set bits 0&1 for N-8-1
	_AH=0;
	_AL=b;
	_DX=comport;
	asm INT FOSSIL
	return(_AX);
	}

int Fossil::IsCarrierDetect()
	{
	_AH=0x03;
	_DX=comport;
	asm INT FOSSIL
	if(_AH&0x80)
		return(TRUE);
	else
		return(FALSE);
	}

int Fossil::IsReceiveDataReady()
	{
	_AH=0x03;
	_DX=comport;
	asm INT FOSSIL
	if(_AH&0x1)
		return(TRUE);
	else
		return(FALSE);
	}

int Fossil::IsTransmitBufferRoom()
	{
	_AH=0x03;
	_DX=comport;
	asm INT FOSSIL
	if(_AH&0x20)
		return(TRUE);
	else
		return(FALSE);
	}

int Fossil::IsTransmitBufferEmpty()
	{
	_AH=0x03;
	_DX=comport;
	asm INT FOSSIL
	if(_AH&0x40)
		return(TRUE);
	else
		return(FALSE);
	}


/*********************************
 Here's a simple example using the
 fossil class. I was able to send
 dialing commands to the modem and
 log on a BBS!
 *********************************/

void main()
	{
	Fossil f;
	char r=0,s;

        cout << "\nTest of the Fossil Class. [ESC] to exit.\n";

        if (f.Detect(0)==Fossil::FALSE)
		{
		cout << "\nNo Fossil Installed!\n";
		return;
		}

	if(f.Open(0,19200)==Fossil::ERROR)
		{
		cout << "\nError Opening Fossil!\n";
		return;
		}
	else
		cout <<"\nFossil successfully Opened\n";

	do {
		if(f.ReceiveChar(r))
			cout << r;

		if(bioskey(1))
			{
			s=bioskey(0)&0xff;
			if(f.TransmitChar(s)==Fossil::ERROR)
				{
				cout << "\nFossil Transmit Error!\n";
				return;
				}
			}
                } while(s!=27);    // ESC to Exit

        cout << "\nDone!";
	}



