/*
	flexlist.hpp
	11-28-90
	Homogeneous-heterogeneous
	hybrid stack-queue-list-array generic class.
	C++ versions 1 & 2

	Copyright 1990
	John W. Small
	All rights reserved

	PSW / Power SoftWare
	P.O. Box 10072
	McLean, Virginia 22102 8072
	(703) 759-3838

*/

#ifndef FLEXLIST_CPP
#define FLEXLIST_CPP

//  Define CPP1 for C++ version 1 compatibility
// #define CPP1

#ifdef	CPP1
#define protected  public
#endif

//  Some compilers have ANSI C's standard library available
//  Define ANSI_C_STD_LIB to use this library
// #define ANSI_C_STD_LIB
#ifdef	ANSI_C_STD_LIB
#include <stddef.h>	/*  size_t    */
#include <limits.h>	/*  UINT_MAX  */
#include <string.h>	/*  strcmp(), memcpy(), memset()  */
#else
typedef	unsigned size_t;
#define UINT_MAX  -1
extern int strcmp(const char *s1, const char *s2);
extern void *memcpy(void *dest, const void *src, size_t n);
extern void *memset(void *s, int c, size_t n);
#endif

// FLvariantData specifies that the size of the data
// stored within the FlexNode is to be determined by
// the variant FlexNode virtual functions found in the
// FlexList class.  See the first FlexList constructor.
#define FLvariantData  0
// In the FlexList class the virtual functions are coded
// to handle C strings.  To construct a FlexList to
// hold variant length FlexNodes housing C strings
// code: FlexList s(FLstrings);
#define FLstrings  FLvariantData

// FLmaxMaxNodes specifies the maximum number of FlexNodes
// allowed in any FlexList.
#define FLmaxMaxNodes  UINT_MAX

class	FlexList;
typedef	FlexList *FlexL;
#define FlexL0  ((FlexL)0)
class	FlexNode;
typedef	FlexNode *FlexN;
#define FlexN0  ((FlexN)0)

class FlexNode  {
	FlexN  next, prev;
	friend FlexList;
public:
	char data[1];
};

// FLcomparE() typecasts a compare function pointer to
// the type required by FlexList::setCompare() and
// FlexList::sort().
#define FLcomparE(compare)  ((int (*)(const void *D1, \
		const void *D2)) compare)
// FLcompare0 is the NULL compare function pointer.
#define FLcompare0  FLcomparE(0)


class FlexList {
	FlexN  front, current, rear;
	unsigned  curNum, nodes, maxNodes;
	size_t sizeofNodeData, sizeofNode;
	int sorted;
	int (*compare)(const void *D1, const void *D2);
protected:
	// Variant FlexNode virtual functions:
	// these functions are setup  in the FlexList base
	// class to handle C strings.  A FlexNode in the
	// FlexList will only be as big as necessary to 
	// accomodate the string stored within (see first
	// FlexList constructor).  Derive a new class from
	// FlexList and redefine these virtual functions to
	// accomodate your specific variant data.
	virtual FlexN FNnew(const void *D);
	virtual int   FNwrite(void *ND, const void *D);
	virtual int   FNread(const void *ND, void *D);
	virtual int   FNdestruct(void *ND, void *D);
public:
	// FlexList constructors:
	// sizeofNodeData specifies the size of data to be
	// stored within the nodes of the FlexList.  If
	// sizeofNodeData is equal to FLvariantData, i.e. 0,
	// then the variant FlexNode virtual functions above
	// are used to accomodate variant length data.
	FlexList(size_t sizeofNodeData, unsigned maxNodes
		= FLmaxMaxNodes);
	//  Unpack array into FlexList
	FlexList(unsigned sizeofCell, unsigned cells,
		const void *array);

	// FlexList destructors:
	int   clear();
	virtual ~FlexList() { clear(); }

	// FlexList header functions:
	void *frontD() { return front? front->data : 0; }
	void *currentD() 
		{ return current? current->data : 0; }
	void *rearD() { return rear? rear->data : 0; }
	unsigned CurNum() { return curNum; }
	unsigned Nodes() { return nodes; }
	unsigned MaxNodes() { return maxNodes; }
	int   setMaxNodes(unsigned maxNodes = FLmaxMaxNodes)
		{ return ((maxNodes >= nodes)? 
		this->maxNodes = maxNodes, 1 : 0); }
	unsigned notFull() { return (maxNodes - nodes); }
	unsigned SizeofNodeData() { return sizeofNodeData; }
	int   isSorted() { return sorted; }
	int   unSort()  { sorted = 0; return 1; }
	int (*Compare())(const void *D1, const void *D2) 
		{ return compare; }
	void  setCompare(int (*compare)
		(const void *D1, const void *D2))
		{ this->compare = compare; sorted = 0; }
	unsigned isFixed() { return sizeofNodeData; }
	unsigned isVariant() { return !sizeofNodeData; }

	// FlexList stack and queue functions:
	void *pushN(FlexN N);
	void *pushD(const void *D = 0);
	FlexN popN();
	int   popD(void *D = 0);
	void *topD(void *D = 0);
	void *insQN(FlexN N);
	void *insQD(const void *D = 0);

	// FlexList list functions:
	void *mkcur(unsigned n = 0);
	void *insN(FlexN N);
	void *insD(const void *D = 0);
	void *insSortN(FlexN N);
	void *insSortD(const void *D); 
	FlexN delN();
	int   delD(void *D = 0);
	void *nextD(void *D = 0);
	void *prevD(void *D = 0);

	// FlexList search/sort functions:
	// (see also insSortN()/insSortD() list functions)
	void *findFirstD(const void *D);
	void *findNextD(const void *D);
	void *findLastD(const void *D);
	void *findPrevD(const void *D);
	int   sort(int (*compare)
		(const void *D1, const void *D2) = 0);

	// FlexList array functions:
	// See also compaction functions
	int   storeD(const void *D, unsigned n = 0);
	int   recallD(void *D, unsigned n = 0);

	// FlexList compaction functions:
	// (see also FlexList constructors)
	void *pack();
	void **packPtrs();
};


/*  
	FlexList implementation constants -
	change as required by target machine.
*/


// Segmented machine architectures prevent the maximum
// segment size from being allocated with new.  I've assumed
// a loss of 16 bytes to be the worst case for all machines.
// FLnewAlignLoss is used in macros below.
#define FLnewAlignLoss  16

// FLmaxSizeofNodeData is used in FlexList constructors to
// specify the maximum sizeof(data) that can be stored in
// a FlexNode.
#define FLmaxSizeofNodeData  \
	((size_t)(-(long)sizeof(FlexNode) \
	-FLnewAlignLoss))

// FLmaxSizeofArray is used in pack() and packPtrs().
// Many compilers truncate long to unsigned on calls to new!
#define FLmaxSizeofArray ((long)(size_t)-FLnewAlignLoss)
// If your machine and compiler can handle full long
// allocations then you can redefine FLmaxSizeofArray as:
// #define FLmaxSizeofArray sizeofArray
// This allows full range positive long values.


#endif
