/*	file name fftc.c
*	program name fftc() ... double version of FFT 
*
*	Written by Jim Pisano 
*	P.O. Box 3134
*	Charlottesville, VA 22903
*
*	Compiles under DeSmet C compiler ver. 2.3
*	This code is standard C and should be okay for most compilers.
*
*	Perform fast fourier transform on data according to E. Brigham in his
*  book Fast Fourier Transforms.

*  n_pts ... Number of real data points
*  nu ...... logarithm in base 2 of n_pts e.g. nu = 5 if n_pts = 32
*  inv ..... FALSE for forward transform, TRUE for inverse transform
*  real_data pointer to array of real data points
*  imag_data pointer to array of imaginary data points

*/

#include	<stdio.h>
#include	<math.h>
#define		PI		3.1419527

fft( real_data, imag_data, n_pts, nu, inv )
double *real_data, *imag_data;
int n_pts, nu, inv;
{
	int n2, j, j1, l, i, ib, k, k1, k2;
	double sgn;
	double arg, nu1;
	double c, s;
	double tr, ti;		/* temporary real & imaginary vars */

	n2 = n_pts / 2;
	nu1 = nu - 1.0;
	k = 0;
	
	sgn = inv ? -1.0 : 1.0;		/* sign change for inverse transform */
	for( l = 0; l != nu; l++ )
	{
		do
		{
/*
* calculate real & imaginary components of transform using Brigham's algorithm
*/ 
			for( i = 0; i != n2; i++ )
			{	
				j = k / (int) pow( 2.0, nu1 );
				ib = bit_swap( j, nu );
				arg = 2.0 * PI * (double) ib / (double) n_pts;
				c = cos( arg );
				s = sgn * sin( arg );
				k1 = k;
				k2 = k1 + n2;
				tr = *(real_data+k2) * c + *(imag_data+k2) * s;
				ti = *(imag_data+k2) * c - *(real_data+k2) * s;
				*(real_data+k2) = *(real_data+k1) - tr;
				*(imag_data+k2) = *(imag_data+k1) - ti;
				*(real_data+k1) = *(real_data+k1) + tr;
				*(imag_data+k1) = *(imag_data+k1) + ti;
				k++;
			}
			k +=  n2;
		} while( k < n_pts - 1);
		k = 0;
		nu1 = nu1 - 1.0;
		n2 /= 2;
	}	
	for( k = 0; k != n_pts; k++ )
	{
		ib = bit_swap( k, nu );
		if( ib > k)
		{
			swap( (real_data+k), (real_data+ib) );
			swap( (imag_data+k), (imag_data+ib) );
		}
	}
/* 
* for inverse transform, divide all data points by n_pts, the total number of
* data points.
*/
	if( inv )
		for( k = 0; k != n_pts; k++)
		{
			*(real_data+k) /= (double) n_pts;
			*(imag_data+k) /= (double) n_pts;
		} 
}
/*
* this routine swaps the bits in variable i according to Brigham's original
* FORTRAN version.  A more elegant method using bit operators could be develop-
* ed, but not today by me.
*/
bit_swap( i, nu )
int i, nu;
{
	int ib, i1, i2;

	ib = 0;

	for( i1 = 0; i1 != nu; i1++ )
	{
		i2  = i / 2;
		ib = ib * 2 + i - 2 * i2;
		i   = i2;
	}
	return( ib );
}
swap( x1, x2 )			/* swap pointers x1 & y1 */
double *x1, *x2;
{
	double *temp_x;
 
		*temp_x = *x1;
		*x1 = *x2;
		*x2 = *temp_x;
}                                                                                                                                                                                                                                                                