//****************************************************************************
// File: palfade.c
//
// Purpose: palette manipulation routines and file routines
//
// Functions:   
//    VOID      FadePalette(HWND hWnd, int nStep, int nTotalSteps)
//    HBITMAP   BM2ToBitmap(HGLOBAL hBM2, HPALETTE hpal)
//    VOID      UsePalette(HDC hDC, BOOL bNoStatic)
//    HPALETTE  CreatePalArray(PALETTEENTRY *pe1, PALETTEENTRY *pe2, int nColors)
//    VOID      CalculateFadeStep(PALETTEENTRY *pe1, PALETTEENTRY *pe2, int nTotalSteps)
//    HGLOBAL   CreateBM2(HGLOBAL hDIB1, PALETTEENTRY *pe1, 
//                        HGLOBAL hDIB2, PALETTEENTRY *pe2, OPTIONS *pOptions)
//
// Development Team:
//            Jeff Saathoff
//
// Written by Microsoft Product Support Services, Windows Developer Support
// COPYRIGHT:
//
//   (C) Copyright Microsoft Corp. 1993.  All rights reserved.
//
//   You have a royalty-free right to use, modify, reproduce and
//   distribute the Sample Files (and/or any modified version) in
//   any way you find useful, provided that you agree that
//   Microsoft has no warranty obligations or liability for any
//   Sample Application Files which are modified.
//
//****************************************************************************

#include <windows.h>
#include <io.h>
#include "global.h"
#include "palfade.h"
#include "dib.h"
#include "file.h"
#include "options.h"
#include "cutcolor.h"

typedef struct
  {
  int red;
  int green;
  int blue;
  } COLORS;

// Variables local to this file
static int nColors;
static int nPrevStep;
static PALETTEENTRY pe256Color[256];
static COLORS n256Step[256];

//***********************************************************************
// Function: FadePalette
//
// Purpose: Animates the palette to the designated position.
//
// Parameters:
//    hWnd        == handle to window
//    nStep       == current step
//    nTotalSteps == last possible step
//
// Returns: No Return
//
// Comments:
//
// History: Date      Author        Reason
//          4/15/92   JMS           Created
//***********************************************************************

VOID FadePalette(HWND hWnd, int nStep, int nTotalSteps)
{
  HDC hDC;
  HPALETTE hOldPal;
  int i, nDiff, nStart, nEnd;
    
  if (!hPal)
    return;

  if (nStep == nPrevStep)   // Nothing to do here
    return;

  // Calculate the new colors and save current step
  nDiff = nStep - nPrevStep;

  if (nColors == 15)
    {
    nStart = 0;             // for 15 colors, we animate
    nEnd = 225;             // colors 0 through 224 in the array
    }
  else
    {
    nStart = 1;             // for 16 colors, we animate
    nEnd = 255;             // colors 1 through 254
    }

  for (i=nStart; i<nEnd; i++)
    {
    pe256Color[i].peRed   = (BYTE) ((int) pe256Color[i].peRed
                                        + nDiff*n256Step[i].red);
    pe256Color[i].peGreen = (BYTE) ((int) pe256Color[i].peGreen
                                        + nDiff*n256Step[i].green);
    pe256Color[i].peBlue  = (BYTE) ((int) pe256Color[i].peBlue
                                        + nDiff*n256Step[i].blue);
    }
  nPrevStep = nStep;

  // Do the palette animation
  hDC = GetDC(hWnd);
  hOldPal = SelectPalette(hDC, hPal, FALSE);
	AnimatePalette(hPal, nStart, nEnd-nStart, pe256Color+nStart);

  // Special case the end points
  if (nStep == nTotalSteps)
    {
    int j,k=0;

    for (i=0; i<nColors; i++)
      for (j=0; j<nColors; j++, k++)
        pe256Color[k] = pe16Color2[j];

    AnimatePalette(hPal, nStart, nEnd-nStart, pe256Color+nStart);
    }
  else if (nStep == 0)
    {
    int j,k=0;

    for (i=0; i<nColors; i++)
      for (j=0; j<nColors; j++, k++)
        pe256Color[k] = pe16Color1[i];

    AnimatePalette(hPal, nStart, nEnd-nStart, pe256Color+nStart);
    }

  SelectPalette(hDC, hOldPal, TRUE);
  ReleaseDC(hWnd, hDC);
}

//***********************************************************************
// Function: BM2ToBitmap
//
// Purpose: Creates a DDB from a "dual" DIB (BM2)
//
// Parameters:
//    hBM2  == handle to the DIB
//    hpal  == handle to the corresponding palette (from CreatePalArray)
//
// Returns: handle to the bitmap or NULL if error
//
// Comments:
//
// History: Date      Author        Reason
//          4/15/92   JMS           Created
//***********************************************************************

HBITMAP BM2ToBitmap(HGLOBAL hBM2, HPALETTE hpal)
{
  LPBITMAPINFOHEADER  lpbi;
  HPALETTE            hpalOld;
  HBITMAP             hbm;
  HDC                 hdc;

  if (!(hpal && hBM2))
    return NULL;
  
  hdc = GetDC(NULL);

  lpbi = (LPBITMAPINFOHEADER) GlobalLock(hBM2);

  if (lpbi->biClrImportant != 15)
    UsePalette(hdc, TRUE);  // take over system color entries

  UnrealizeObject(hpal);
  hpalOld = SelectPalette(hdc, hpal, FALSE);
  RealizePalette(hdc);

  hbm = CreateDIBitmap( hdc,
                        lpbi,
                        CBM_INIT,
                        FindDIBBits((LPSTR) lpbi),
                        (LPBITMAPINFO) lpbi,
                        DIB_PAL_COLORS );

  if (hpalOld)
    SelectPalette(hdc, hpalOld, TRUE);

  if (!hbm && lpbi->biClrImportant != 15)
    UsePalette(hdc, FALSE); // restore system colors
 
  ReleaseDC(NULL, hdc);
  GlobalUnlock(hBM2);

  return hbm;
}

//***********************************************************************
// Function: UsePalette
//
// Purpose: Takes over or relinquishes the static area of the system palette
//
// Parameters:
//    hDC       == handle to DC to use
//    bNoStatic == flag indicating action to take
//
// Returns: No return
//
// Comments:
//  UsePalette does the following:
//  
//  1. If bNoStatic is TRUE, sets the system palette use to allow using 254
//  colors in the system palette, eliminating 18 of the 20 reserved colors.
//  Also, saves the system colors (for caption bars, window borders, etc.)
//  and changes them to black and white.
//
//  2. If bNoStatic is FALSE, sets the system palette use to restore the
//  18 reserved colors, so you can only use 236 colors. Also, restores
//  the system colors to their original values.
//
//  Borrowed from another sample, but forgot which.
//
// History: Date      Author        Reason
//
//***********************************************************************

#define RGBBlack    0x00000000L
#define RGBWhite    0x00FFFFFFL

VOID UsePalette(HDC hDC, BOOL bNoStatic)
{
  // Flag for whether we've initialized dwSysColors yet or not.
  static BOOL  fHaveSysColors = FALSE;

  // Array for holding old system colors
  static DWORD dwSysColors[21];

  // Black and white values for when we change the system colors
  static DWORD dwBWSysColors[21]
              = { RGBWhite,   // scrollbar
                  RGBWhite,   // background
                  RGBWhite,   // active caption
                  RGBWhite,   // inactive caption
                  RGBWhite,   // menu
                  RGBWhite,   // window
                  RGBBlack,   // window frame
                  RGBBlack,   // menu text
                  RGBBlack,   // window text
                  RGBBlack,   // caption text
                  RGBWhite,   // active border
                  RGBBlack,   // inactive border
                  RGBWhite,   // app workspace
                  RGBBlack,   // highlight
                  RGBWhite,   // highlight text
                  RGBWhite,   // button face
                  RGBBlack,   // button shadow
                  RGBBlack,   // gray text
                  RGBBlack,   // button text

                  RGBBlack,   // inactive caption text
                  RGBWhite }; // button highlight


  // Index array for setting the system colors
  static int iSysColorNdx[21] = { 0,1,2,3,4,5,6,7,8,9,10,11,
                                  12,13,14,15,16,17,18,19,20 };

  // If there's no palette, there's no reason to mess with system colors 
  if (!hPal)
    return;

  // If this is the first time we've been called, we need to 
  // get the current system colors.

  if (!fHaveSysColors)
    {
    int i;

    for(i=0; i<21; i++)
      dwSysColors[i] = GetSysColor(i);

    fHaveSysColors = TRUE;
    }

  // If we are taking over the palette, we want to change the
  // system colors first, so the appearance of flashing is
  // minimized. If we are releasing use of the whole palette,
  // we will do it at the end.
    
  if(bNoStatic)
    {
    // Set the system colors to black and white
    SetSysColors(21, iSysColorNdx ,dwBWSysColors);

    // Broadcast WM_SYSCOLORCHANGE
    SendMessage((HWND)0xFFFF, WM_SYSCOLORCHANGE, 0, 0L);
    }

  SetSystemPaletteUse(hDC, bNoStatic ? SYSPAL_NOSTATIC : SYSPAL_STATIC);

  // If we are releasing use of the whole palette, we change the
  // system colors back at the end to minimize the appearance of
  // flashing from the palette changes.

  if(!bNoStatic)
    {
    // Set the system colors to original colors
    SetSysColors(21, iSysColorNdx ,dwSysColors);

    // Broadcast WM_SYSCOLORCHANGE
    SendMessage((HWND)0xFFFF, WM_SYSCOLORCHANGE, 0, 0L);
    }
}

//***********************************************************************
// Function: CreatePalArray
//
// Purpose: Creates the 256-color palette from the 2 16-color palettes
//
// Parameters:
//    pe1   == pointer to first array of PALETTENTRY's
//    pe2   == pointer to second array of PALETTENTRY's
//    nClrs == number of colors to use (15 or 16)
//
// Returns: handle to the palette or NULL if error
//
// Comments:
//
// History: Date      Author        Reason
//          4/15/92   JMS           Created
//***********************************************************************

HPALETTE CreatePalArray(PALETTEENTRY *pe1, PALETTEENTRY *pe2, int nClrs)
{
  HPALETTE hPalette;
  NPLOGPALETTE pPal;
  int i,j,k;

  // Save number of colors for CalculateFadeStep and FadePalette

  if (nClrs != 15)
    nColors = 16;
  else 
    nColors = 15;
    
  // Allocate memory for 225/256 color palette.

  pPal = (NPLOGPALETTE) LocalAlloc(LPTR,
            2*sizeof(WORD) + nColors*nColors*sizeof(PALETTEENTRY));

  if (!pPal)
    return NULL;

  pPal->palVersion    = 0x300;  // for both 3.0 and 3.1
  pPal->palNumEntries = nColors*nColors;

  // Initialize the color palette as an array with each row
  // being a single color from the first 15/16 color palette.

  k = 0;
  for (i=0; i<nColors; i++)
    for (j=0; j<nColors; j++, k++)
      pe256Color[k] = pPal->palPalEntry[k] = pe1[i];

  // If we are using 16x16 colors, we won't reserve the first
  // color (the darkest) or the last color (the lightest) since
  // we can only animate 254 colors.  These 2 colors will map
  // to black and white, respectively.  (LoadBMP takes care of
  // putting the darkest color first and the lightest color last.)

  if (nColors != 15)
    {
    pe256Color[0].peFlags   = pPal->palPalEntry[0].peFlags   = 0;
    pe256Color[255].peFlags = pPal->palPalEntry[255].peFlags = 0;
    }

  hPalette = CreatePalette(pPal);

  LocalFree((HLOCAL) pPal);

  return hPalette;
}

//***********************************************************************
// Function: CalculateFadeStep
//
// Purpose: Calculates a table of increments for use in FadePalette
//
// Parameters:
//    pe1         == pointer to first array of PALETTENTRY's
//    pe2         == pointer to second array of PALETTENTRY's
//    nTotalSteps == number of animation steps
//
// Returns: No return
//
// Comments:
//
// History: Date      Author        Reason
//          4/15/92   JMS           Created
//***********************************************************************

VOID CalculateFadeStep( PALETTEENTRY *pe1, PALETTEENTRY *pe2, int nTotalSteps)
{
  int i,j,k=0;

  nPrevStep = 0;

  for (i = 0; i < nColors; i++)
    for(j = 0; j < nColors; j++, k++)
	  {
	  n256Step[k].red   = (int) ((double) (pe2[j].peRed   - pe1[i].peRed  )/nTotalSteps);
	  n256Step[k].green = (int) ((double) (pe2[j].peGreen - pe1[i].peGreen)/nTotalSteps);
	  n256Step[k].blue  = (int) ((double) (pe2[j].peBlue  - pe1[i].peBlue )/nTotalSteps);
	  }

  return;
}

//***********************************************************************
// Function: CreateBM2
//
// Purpose: Creates a "dual" DIB from two 16-color DIB's
//
// Parameters:
//    phDIB1    == pointer to handle of first DIB
//    pe1       == pointer to first color table
//    phDIB2    == pointer to handle of second DIB
//    pe2       == pointer to second color table
//    pOptions  == pointer to OPTIONS structure
//
// Returns: Global handle to new DIB or NULL if error.
//
// Comments:
//  Creates an 8-bpp DIB from scratch, including the bits (can be slow).
//
// History: Date      Author        Reason
//          4/15/92   JMS           Created
//***********************************************************************

HGLOBAL CreateBM2(HGLOBAL *phDIB1, PALETTEENTRY *pe1,
                  HGLOBAL *phDIB2, PALETTEENTRY *pe2, OPTIONS *pOptions)
{
  LPBITMAPINFOHEADER  lpbi, lpbi1, lpbi2;
  HGLOBAL hDIBMem;
  HPBYTE hpBits, hpBits1, hpBits2;
  HPBYTE hp, hp1, hp2;
  LPWORD lpPal1, lpPal2;
  DWORD dwWidth, dwHeight;
  DWORD dwScanWidth8, dwScanWidth4;
  DWORD i,j;

  lpbi1 = (LPBITMAPINFOHEADER) GlobalLock(*phDIB1);
  lpbi2 = (LPBITMAPINFOHEADER) GlobalLock(*phDIB2);

  // Find width and height of new BM2
  switch (LOBYTE(pOptions->wFlags))
    {
    case IDB_LARGER:
    case IDB_SMALLER:
    default:
      if (LOBYTE(pOptions->wFlags) == IDB_SMALLER)
        {
        // Use the minimum dimensions from the 2 DIBs.

        dwWidth  = min(lpbi1->biWidth,  lpbi2->biWidth);
        dwHeight = min(lpbi1->biHeight, lpbi2->biHeight);
        }
      else
        {
        // Use the maximum dimensions from the 2 DIBs

        dwWidth  = max(lpbi1->biWidth,  lpbi2->biWidth);
        dwHeight = max(lpbi1->biHeight, lpbi2->biHeight);
        }
      break;

    case IDB_SIZE1:
      // Use the size of DIB 1
      dwWidth  = lpbi1->biWidth;
      dwHeight = lpbi1->biHeight;
      break;

    case IDB_SIZE2:
      // Use the size of DIB 2
      dwWidth  = lpbi2->biWidth;
      dwHeight = lpbi2->biHeight;
      break;

    case IDB_CUSTOM:
      // Use user defined size
      dwWidth  = pOptions->nWidth;
      dwHeight = pOptions->nHeight;
      break;
    }

  // Adjust DIBs to new size

  GlobalUnlock(*phDIB1);
  GlobalUnlock(*phDIB2);

  if (!SizeDIB(phDIB1, pe1, dwWidth, dwHeight, pOptions->wFlags))
    return NULL;

  if (!SizeDIB(phDIB2, pe2, dwWidth, dwHeight, pOptions->wFlags))
    return NULL;

  lpbi1 = (LPBITMAPINFOHEADER) GlobalLock(*phDIB1);
  lpbi2 = (LPBITMAPINFOHEADER) GlobalLock(*phDIB2);

  // Save scan widths for later use
  dwScanWidth8  = WIDTHBYTES(dwWidth<<3);           // 8 bits per pixel
  dwScanWidth4  = WIDTHBYTES(dwWidth<<2);           // 4 bpp

  // allocate memory for new 8-bit DIB
  hDIBMem = GlobalAlloc(GHND, sizeof(BITMAPINFOHEADER)
      + 256*sizeof(RGBQUAD) + dwScanWidth8*dwHeight);

  if (!hDIBMem)
    {
    GlobalUnlock(*phDIB1);
    GlobalUnlock(*phDIB2);
    return NULL;
    }

  lpbi = (LPBITMAPINFOHEADER) GlobalLock(hDIBMem);

  // set up BITMAPINFOHEADER for 8-bit DIB
  InitBitmapInfoHeader (lpbi, dwWidth, dwHeight, 8);
  lpbi->biCompression = BI_RGB;
  lpbi->biClrUsed = 256;
  lpbi->biClrImportant = (pOptions->wFlags&IDB_15COLORS ? 15 : 16);

  // Load the color table with DIB_PAL_COLORS indices
  lpPal1 = (LPWORD) (lpbi + 1);
  for (i=0; i<256; i++, lpPal1++)
    *lpPal1 = (WORD) i;

  // get pointers to DIB_PAL_COLORS indices for 16-color DIBs

  lpPal1 = (LPWORD) (lpbi1 + 1);
  lpPal2 = (LPWORD) (lpbi2 + 1);


  // If we're using 15x15 colors instead of 16x16 then we need 
  // to eliminate one of the colors in each of the 4-bpp DIBs.
  // If CutColor fails, just map the last color to the next to 
  // last color.

  if (lpbi->biClrImportant == 15)
    {
    HCURSOR hcurSave = SetCursor(LoadCursor(NULL, IDC_ARROW));

    if (!CutColor(*phDIB1, pe1))
      lpPal1[15] = lpPal1[14];
    if (!CutColor(*phDIB2, pe2))
      lpPal2[15] = lpPal2[14];

    SetCursor(hcurSave);
    }

  // find the beginning of the bits

  hp  = hpBits  = FindDIBBits((LPSTR) lpbi);
  hp1 = hpBits1 = FindDIBBits((LPSTR) lpbi1);
  hp2 = hpBits2 = FindDIBBits((LPSTR) lpbi2);

  // Next set the bits.  This is the heart of the entire process.
  // Basically, we take the bits from the 4-bpp DIBs, put them
  // together, and put the result into the 8-bpp DIB.  The pixel
  // values from the 4-bpp DIBs (which are offsets into the color
  // tables) are translated using the DIB_PAL_COLORS indices stored
  // in the respective color tables.  If we were always going to
  // use all 16x16 colors, the "* lpbi->biClrImportant) +" could be
  // replaced with "<< 4) |" to improve speed.  The method used
  // below works for any number of colors from 2 to 16 (we 
  // use either 15 or 16).

  for (j=0; j<dwHeight; j++)          // loop first by scanline
    {
    for (i=0; i<dwWidth; i++)         // loop through pixels on scanline
      {
      if (i & 0x01L)      // odd pixels... increment hp1 and hp2

        *hp++ = LOBYTE( (*(lpPal1 + (*hp1++ & 0x0f)) * lpbi->biClrImportant)
            + *(lpPal2 + (*hp2++ & 0x0f)) );

      else                // even pixels... DON'T increment hp1 and hp2

        *hp++ = LOBYTE( (*(lpPal1 + ((*hp1 & 0xf0) >> 4)) * lpbi->biClrImportant)
            + *(lpPal2 + ((*hp2 & 0xf0) >> 4)) );
      }
    hp  = (hpBits  += dwScanWidth8);  // go to beginning of next scan line
    hp1 = (hpBits1 += dwScanWidth4);
    hp2 = (hpBits2 += dwScanWidth4);
    }

  // Clean up

  GlobalUnlock(hDIBMem);
  GlobalUnlock(*phDIB1);
  GlobalUnlock(*phDIB2);

  // Finally, create the color array and return.

  if(!(hPal = CreatePalArray(pe1, pe2, (pOptions->wFlags&IDB_15COLORS ? 15 : 16))))
    return NULL;

  return hDIBMem;
}
