//****************************************************************************
// File: cutcolor.c
//
// Purpose : Called by CreateBM2() to remove a color from a 16 color DIB.
//
// Functions:
//    CutColor()        - puts up the dialog
//    CutColorDlgProc() - dialog function
//    PatRect()         - draws a rectangular frame using PatBlt()
//
// Development Team:
//            Jeff Saathoff
//
// Written by Microsoft Product Support Services, Windows Developer Support
// COPYRIGHT:
//
//   (C) Copyright Microsoft Corp. 1993.  All rights reserved.
//
//   You have a royalty-free right to use, modify, reproduce and
//   distribute the Sample Files (and/or any modified version) in
//   any way you find useful, provided that you agree that
//   Microsoft has no warranty obligations or liability for any
//   Sample Application Files which are modified.
//
//****************************************************************************

#include <windows.h>
#include "global.h"
#include "cutcolor.h"
#include "dib.h"

// Constants used for sizing the dialog
#define MAXPICWIDTH   560
#define MINPICWIDTH   180
#define MAXPICHEIGHT  420
#define MINPICHEIGHT  180

#define SPACE 10
#define FRAME 2

// Data type passed to the Cut Color dialog procedure
typedef struct
  {
  HGLOBAL hdib;
  LPPALETTEENTRY pe;
  } DIBPAL, FAR *LPDIBPAL;

// Local function prototype
void PatRect(HDC hdc, RECT *prc, int nWidth, DWORD dwRop);

//***********************************************************************
// Function: CutColor
//
// Purpose: Puts up the Cut Color dialog to remove a color from 
//          a 16-color DIB.
//
// Parameters:
//    hdib  == handle to DIB in CF_DIB format
//    pe    == pointer to color table
//
// Returns: the value returned by DialogBoxParam
//
// Comments:
//
// History: Date      Author        Reason
//          8/30/92   JMS           Created
//***********************************************************************

BOOL CutColor(HGLOBAL hdib, LPPALETTEENTRY pe)
{
  DIBPAL dp;
  DLGPROC lpDlg;
  BOOL bRet;

  dp.hdib = hdib;
  dp.pe   = pe;

  lpDlg = (DLGPROC) MakeProcInstance((FARPROC)CutColorDlgProc, ghInst);
  bRet = (BOOL) DialogBoxParam(ghInst, "CUTCOLOR", ghWnd, lpDlg, (LPARAM)(LPDIBPAL)&dp);
  FreeProcInstance ((FARPROC) lpDlg);

  return bRet;
}

//***********************************************************************
// Function: CutColorDlgProc
//
// Purpose: Dialog procedure for the Cut Color dialog
//
// Parameters: standard window procedure parameters
//
// Returns: TRUE if the message was processed, FALSE otherwise
//
// Comments:
//
// History: Date      Author        Reason
//          8/30/92   JMS           Created
//***********************************************************************

BOOL FAR PASCAL CutColorDlgProc (HWND hDlg, UINT msg, WPARAM wParam, LPARAM lParam)
{
  static HGLOBAL hdib;
  static LPPALETTEENTRY pe;
  static HBITMAP hbm;
  static HPALETTE hpalAnimate, hpalNoAnimate;
  static int nFrom, nTo;

  switch (msg)
    {
    case WM_INITDIALOG:
      {
      LPBITMAPINFOHEADER lpbi;
      PLOGPALETTE ppal;
      HPALETTE hpalSave;
      HBITMAP hbmSave;
      HDC hdc, hdcMem;
      HWND hwnd;
      RECT rc;
      int i,j, nPicWidth, nPicHeight;

      // save handle to dib and pointer to palette entries
      hdib = ((LPDIBPAL) lParam)->hdib;
      pe   = ((LPDIBPAL) lParam)->pe;

      // Initialize the controls.

      for (i=0; i<16; i++)
        {
        SendDlgItemMessage(hDlg, IDC_FROM, CB_ADDSTRING, 0, (LPARAM) PALETTEINDEX(i));
        SendDlgItemMessage(hDlg, IDC_TO,   CB_ADDSTRING, 0, (LPARAM) PALETTEINDEX(i));
        }

      // Set the initial selections.
      // If the palette happens to match the standard 16 VGA colors,
      // this will map dark red to black.
      nFrom = 1;
      nTo = 0;
      SendDlgItemMessage(hDlg, IDC_FROM, CB_SETCURSEL, nFrom, 0L);
      SendDlgItemMessage(hDlg, IDC_TO,   CB_SETCURSEL, nTo,   0L);

      // Setup the dialog based on the size of the bitmap...

      lpbi = (LPBITMAPINFOHEADER) GlobalLock(hdib);

      nPicWidth  = max(min((int) lpbi->biWidth, MAXPICWIDTH),  MINPICWIDTH);
      nPicHeight = max(min((int) lpbi->biWidth, MAXPICHEIGHT), MINPICHEIGHT);

      hwnd = GetDlgItem(hDlg, IDC_FROM);
      GetWindowRect(hwnd, &rc);
      ScreenToClient(hDlg, (LPPOINT) &rc);
      ScreenToClient(hDlg, ((LPPOINT) &rc)+1);

      i = 3*SPACE + 2*FRAME;

      SetWindowPos( hDlg, NULL, 0, 0,
                    nPicWidth + rc.right-rc.left + i + 2*GetSystemMetrics(SM_CXDLGFRAME),
                    nPicHeight + i + GetSystemMetrics(SM_CYCAPTION),
                    SWP_NOMOVE|SWP_NOZORDER|SWP_NOREDRAW|SWP_NOACTIVATE );

      i -= SPACE;
      rc.bottom = i + rc.bottom - rc.top;
      rc.top = i;
      i += nPicWidth;
      MoveWindow(hwnd, i, rc.top, rc.right-rc.left, rc.bottom-rc.top, FALSE);
      hwnd = GetDlgItem(hDlg, IDC_REPLACE);
      MoveWindow(hwnd, i, rc.top-2*SPACE, rc.right-rc.left, 2*SPACE, FALSE);
      hwnd = GetDlgItem(hDlg, IDC_WITH);
      MoveWindow(hwnd, i, rc.bottom+SPACE, rc.right-rc.left, 2*SPACE, FALSE);
      hwnd = GetDlgItem(hDlg, IDC_TO);
      MoveWindow(hwnd, i, rc.bottom+3*SPACE, rc.right-rc.left, rc.bottom-rc.top, FALSE);

      i -= SPACE;
      GetClientRect(hDlg, &rc);
      i += rc.right;
      j = rc.bottom;
      hwnd = GetDlgItem(hDlg, IDCANCEL);
      GetClientRect(hwnd, &rc);
      MoveWindow(hwnd, (i-rc.right)>>1,
        j-2*(SPACE+rc.bottom), rc.right, rc.bottom, FALSE);
      hwnd = GetDlgItem(hDlg, IDOK);
      MoveWindow(hwnd, (i-rc.right)>>1,
        j-SPACE-rc.bottom, rc.right, rc.bottom, FALSE);

      // Create the palettes.

      hpalAnimate = hpalNoAnimate = NULL;
      ppal = (PLOGPALETTE) LocalAlloc(LPTR, 2*sizeof(WORD) + 16*sizeof(PALETTEENTRY));

      if (ppal)
        {
        ppal->palVersion = 0x300;
        ppal->palNumEntries = 16;

        for (i=0; i<16; i++)
          {
          ppal->palPalEntry[i] = pe[i];
          ppal->palPalEntry[i].peFlags = 0;
          }

        hpalNoAnimate = CreatePalette(ppal);

        for (i=0; i<16; i++)
          ppal->palPalEntry[i].peFlags = PC_RESERVED;

        hpalAnimate = CreatePalette(ppal);

        LocalFree((HLOCAL) ppal);
        }

      if (!hpalAnimate)
        hpalAnimate = GetStockObject(DEFAULT_PALETTE);

      // Create the bitmap.

      hdc = GetDC(hDlg);
      hdcMem = CreateCompatibleDC(hdc);

      hbm = CreateCompatibleBitmap(hdc, nPicWidth, nPicHeight);
      hbmSave = SelectObject(hdcMem, hbm);

      ReleaseDC(hDlg, hdc);

      hpalSave = SelectPalette(hdcMem, hpalAnimate, FALSE);
      RealizePalette(hdcMem);
      AnimatePalette(hpalAnimate, nFrom, 1, &pe[nTo]);

      StretchDIBits(hdcMem,
                    0, 0, nPicWidth, nPicHeight,
                    0, 0, (int) lpbi->biWidth, (int) lpbi->biHeight,
                    FindDIBBits((LPSTR) lpbi),
                    (LPBITMAPINFO) lpbi,
                    DIB_PAL_COLORS,
                    SRCCOPY);

      if (hpalSave)
        SelectPalette(hdcMem, hpalSave, TRUE);
      if (hbmSave)
        SelectObject(hdcMem, hbmSave);
      DeleteDC(hdcMem);

      GlobalUnlock(hdib);

      return TRUE;
      }

    case WM_PAINT:
      {
      PAINTSTRUCT ps;
      HPALETTE hpalSave;
      HBITMAP hbmSave;
      BITMAP bm;
      RECT rc;
      HDC hdcMem;

      BeginPaint(hDlg, &ps);

      if (!(hbm && hpalAnimate))
        {
        EndPaint(hDlg, &ps);
        return TRUE;
        }

      // We'll need the width and height of the bitmap.

      GetObject(hbm, sizeof(bm), &bm);

      // Draw frame around bitmap.

      rc.top = SPACE;
      rc.left = SPACE;
      rc.right = rc.left+bm.bmWidth+2*FRAME;
      rc.bottom = rc.top+bm.bmHeight+2*FRAME;

      PatRect(ps.hdc, &rc, FRAME, BLACKNESS);

      InflateRect(&rc, -FRAME, -FRAME);

      // Set up a memory DC and draw the bitmap.

      hdcMem = CreateCompatibleDC(ps.hdc);

      SelectPalette(hdcMem, hpalAnimate, TRUE);
      hpalSave = SelectPalette(ps.hdc, hpalAnimate, TRUE);
      RealizePalette(ps.hdc);

      hbmSave = SelectObject(hdcMem, hbm);

      BitBlt(ps.hdc, rc.left, rc.top, bm.bmWidth, bm.bmHeight,
        hdcMem, 0, 0, SRCCOPY);

      // Clean up.

      if (hpalSave)
        SelectPalette(ps.hdc, hpalSave, TRUE);

      if (hbmSave)
        SelectObject(hdcMem, hbmSave);

      DeleteDC(hdcMem);
      EndPaint(hDlg, &ps);
      return TRUE;
      }

    case WM_MEASUREITEM:
      if (wParam != -1)
        ((LPMEASUREITEMSTRUCT) lParam)->itemHeight = 2*SPACE;
      return TRUE;

    case WM_DRAWITEM:
      {
      RECT rc;
      HBRUSH hbr=NULL, hbrSave=NULL;
      HPALETTE hpalSave=NULL;
      LPDRAWITEMSTRUCT lpdi = (LPDRAWITEMSTRUCT) lParam;

      if (!hpalNoAnimate || lpdi->itemAction == ODA_FOCUS)
        return TRUE;

      hpalSave = SelectPalette(lpdi->hDC, hpalNoAnimate, TRUE);
      RealizePalette(lpdi->hDC);

      if (lpdi->itemAction&ODA_DRAWENTIRE)
        {
        hbr = CreateSolidBrush((COLORREF) lpdi->itemData);
        hbrSave  = SelectObject(lpdi->hDC, hbr);

        rc = lpdi->rcItem;

        InflateRect(&rc, -FRAME, -FRAME);
        PatRect(lpdi->hDC, &rc, FRAME, BLACKNESS);

        InflateRect(&rc, -FRAME, -FRAME);
        PatBlt(lpdi->hDC, rc.left, rc.top, rc.right-rc.left, rc.bottom-rc.top, PATCOPY);

        SelectObject(lpdi->hDC, hbrSave);
        DeleteObject(hbr);
        }

      if (lpdi->itemAction&ODA_SELECT)
        {
        COLORREF cr;

        if (lpdi->itemState&ODS_SELECTED)
          {
          cr = GetSysColor(COLOR_HIGHLIGHT);
          if (GetNearestColor(lpdi->hDC, cr) == GetNearestColor(lpdi->hDC, (COLORREF) lpdi->itemData))
            cr = GetSysColor(COLOR_HIGHLIGHTTEXT);
          }
        else
          cr = (COLORREF) lpdi->itemData;

        hbr = CreateSolidBrush(cr);
        hbrSave  = SelectObject(lpdi->hDC, hbr);

        rc = lpdi->rcItem;
        InflateRect(&rc, -2*FRAME, -2*FRAME);

        PatRect(lpdi->hDC, &rc, FRAME, PATCOPY);

        SelectObject(lpdi->hDC, hbrSave);
        DeleteObject(hbr);
        }

      if (hpalSave)
        SelectPalette(lpdi->hDC, hpalSave, TRUE);

      return TRUE;
      }

    case WM_PALETTECHANGED:
      if ((HWND) wParam == hDlg)
        break;
      // else FALL THROUGH!!!

    case WM_QUERYNEWPALETTE:
      {
      HPALETTE hpalSave;
      HDC hdc = GetDC(hDlg);

      if (hpalAnimate)
        {
        // Want this one in the foreground.
        hpalSave = SelectPalette(hdc, hpalAnimate, FALSE);
        RealizePalette(hdc);
        SelectPalette(hdc, hpalSave, TRUE);
        }

      if (hpalNoAnimate)
        {
        // Force this one to the background.
        hpalSave = SelectPalette(hdc, hpalNoAnimate, TRUE);
        RealizePalette(hdc);
        SelectPalette(hdc, hpalSave, TRUE);
        }

      ReleaseDC(hDlg, hdc);
      return TRUE;
      }

    case WM_DESTROY:
      if (hpalAnimate)
        DeleteObject(hpalAnimate);
      if (hpalNoAnimate)
        DeleteObject(hpalNoAnimate);
      if (hbm)
        DeleteObject(hbm);
      return FALSE;

    case WM_COMMAND:
      switch (wParam)
        {
        case IDCANCEL:
          EndDialog(hDlg, FALSE);
          return TRUE;

        case IDOK:
          if (nFrom == nTo)
            {
            MessageBox(hDlg, "Cannot replace one color with the same color!", NULL, MB_ICONSTOP | MB_OK);
            SetFocus(GetDlgItem(hDlg, IDC_TO));
            }
          else
            {
            LPWORD lpw;
            int i;
            PALETTEENTRY peTmp;
            LPBITMAPINFOHEADER lpbi = (LPBITMAPINFOHEADER) GlobalLock(hdib);

            lpw = (LPWORD) (lpbi + 1);

            // Find index of old color and re-map it to the new color.

            for (i=0; i<16; i++)
              if (lpw[i] == (WORD) nFrom)
                {
                lpw[i] = nTo;
                break;
                }

            // Save the color that's being removed (and its index).
            // This currently isn't used for anything, but it might
            // be useful sometime, so we'll save it at the end of
            // of the PALETTEENTRY's.
              
            peTmp = pe[nFrom];
            peTmp.peFlags = (BYTE) i;

            // Remove color from color table

            for (i=nFrom; i<15; i++)
              pe[i] = pe[i+1];

            // Put removed color and index at the end just in case 
            // it's needed later.

            pe[15] = peTmp;

            // Adjust DIB_PAL_COLORS indices now that the color is removed.

            for (i=0; i<16; i++, lpw++)
              if (*lpw > (WORD) nFrom)
                (*lpw)--;

            GlobalUnlock(hdib);
            EndDialog(hDlg, TRUE);
            }
          return TRUE;


        // If the user has changed the color selections, update the 
        // palette so the changes are visible in the bitmap.

        case IDC_FROM:
          if (hpalAnimate && HIWORD(lParam) == CBN_SELCHANGE)
            {
            // Get new selection
            int n = (int) SendMessage((HWND) LOWORD(lParam), CB_GETCURSEL, 0, 0L);

            if (n == nFrom)
              return TRUE;    // Nothing to do.

            // Restore previous color.
            AnimatePalette(hpalAnimate, nFrom, 1, &pe[nFrom]);

            // Save the new selection.
            nFrom = n;

            // Prevent nFrom and nTo from being the same.  If they're 
            // the same, change nTo to something else.
            if (nFrom == nTo)
              {
              nTo = (++nTo)%16;
              // This will cause a CBN_SELCHANGE for IDC_TO, so
              // we'll do the AnimatePalette during that message.
              return (BOOL) SendDlgItemMessage(hDlg, IDC_TO, CB_SETCURSEL, nTo, 0L);
              }

            // Change colors to reflect the new selection.
            AnimatePalette(hpalAnimate, nFrom, 1, &pe[nTo]);
            }
          return TRUE;

        case IDC_TO:
          if (hpalAnimate && HIWORD(lParam) == CBN_SELCHANGE)
            {
            // Get new selection
            int n = (int) SendMessage((HWND) LOWORD(lParam), CB_GETCURSEL, 0, 0L);

            if (n == nTo)
              return TRUE;    // Nothing to do.

            // Prevent nFrom and nTo from being the same.  If they're 
            // the same, change nTo to something else.
            if (n == nFrom)
              {
              n = (++n)%16;
              // This will cause another CBN_SELCHANGE with n!=nFrom, so
              // we'll do the AnimatePalette during that message.
              return (BOOL) SendMessage((HWND) LOWORD(lParam), CB_SETCURSEL, n, 0L);
              }

            // Save the new selection.
            nTo = n;

            // Change colors to reflect the new selection.
            AnimatePalette(hpalAnimate, nFrom, 1, &pe[nTo]);
            }
          return TRUE;
    	  }
      break;
    }
  return FALSE;
}

//***********************************************************************
// Function: PatRect
//
// Purpose: Draws a frame around a RECT using PatBlt.
//
// Parameters:
//    hdc     == handle to device context
//    prc     == pointer to RECT
//    nWidth  == width of frame
//    dwRop   == ROP to use
//
// Returns: No return
//
// Comments:
//
// History: Date      Author        Reason
//          8/30/92   JMS           Created
//***********************************************************************

void PatRect(HDC hdc, RECT *prc, int nWidth, DWORD dwRop)
{
  PatBlt(hdc, prc->left, prc->top, prc->right-prc->left, nWidth, dwRop);
  PatBlt(hdc, prc->right-nWidth, prc->top, nWidth, prc->bottom-prc->top, dwRop);
  PatBlt(hdc, prc->left, prc->top, nWidth, prc->bottom-prc->top, dwRop);
  PatBlt(hdc, prc->left, prc->bottom-nWidth, prc->right-prc->left, nWidth, dwRop);
}
