/************************************************************************
**
** @(#)respwind.cpp	01/05/94	Chris Ahlstrom
**
** C++ version, requires Turbo Vision by Borland
**
**	This module interfaces C++ with the standard FACE response
** light code (in FACE.LIB), and with code for handling various kinds
** of button devices.
**
**	This module has functions very similar to those of
** response.cpp, but this module only handles mouse events for
** a response box implemented in a window.
**
**	This modules calls routines from the ResponseInterior class to
** implement the lights.  See respintr.cpp for the types of lights
** implemented.
**
**	The lights are mapped onto a byte.  Whereever a bit is set,
** the corresponding "light" gets turned on.  Some convenient labels
** are added, too, to represent combinations of lights.
**
*************************************************************************/

#define RESPWIND_cpp

#include <stdio.h>
#include <conio.h>
#include <ctype.h>

#include "mkbutton.h"		// ResponseButton creation class
#include "rbuttons.h"		// ResponseButton class
#include "respwind.h"		// ResponseDevice class
#include "respvga.h"		// declares/defines virtualBox



/************************************************************************
** ResponseDevice
**
**	Can't make copies and such until we write the copy constructor
** and assignment operator.
**
*************************************************************************/

ResponseDevice::ResponseDevice
(
    ResponseType device,
    const TRect& bounds,
    const char *atitle,
    short anumber
) :
    TWindow	(bounds, atitle, anumber),
    TWindowInit	(ResponseDevice::initFrame),
    buttons	(makeResponseButton(device))
{
    TRect wbounds = getExtent();		// size of window
    wbounds.grow(-1, -1);			// fit it in window
    subInterior = new ResponseInterior		// make an interior
    (
	virtualBox, wbounds
    );
    if (subInterior)
	insert(subInterior);			// put it in window
}


/************************************************************************
** ~ResponseDevice
*************************************************************************/

ResponseDevice::~ResponseDevice()
{
    // In Turbo Vision, the interior of the window (pointed to by
    // subInterior), will automatically be deleted.  I doubt this
    // will hold true using Object Windows... check.

    if (buttons)
	delete buttons;
}


/************************************************************************
** getPalette
*************************************************************************/

TPalette&
ResponseDevice::getPalette() const
{
    static TPalette palette(cpRespPalette, sizeof(cpRespPalette)-1);
    return palette;
}


/************************************************************************
** startScreen
*************************************************************************/

ResponseSet
ResponseDevice::startScreen()
{
    ResponseSet status = RESPONSE_NORMAL;

    subInterior->light(RESP_ALL_ON);			 // show first display
    if (buttons && (buttons->waitKey() == BOTH_BUTTONS)) // wait for subject
	status = RESPONSE_QUIT;
    subInterior->light(RESP_ALL_OFF);			 // turn off all lights

    return status;
}


/************************************************************************
** Encapsulating routines:
**
**	ResponseDevice::readyLight
**	ResponseDevice::responseOff
**	ResponseDevice::intervalLight
**	ResponseDevice::answerLight
**	ResponseDevice::responseFeedback
**
*************************************************************************/

void
ResponseDevice::readyLight ()
{
    subInterior->light(RESP_READY);	// Ready light
}


void
ResponseDevice::responseOff ()
{
    subInterior->light(RESP_ALL_OFF);
}


void
ResponseDevice::intervalLight
(
    ushort interval
)
{
    subInterior->light(interval << RESP_INTV_BITS);
}


void
ResponseDevice::answerLight ()
{
    subInterior->light(RESP_ANSWER);
}


int
ResponseDevice::responseFeedback
(
    int stimulus
)
{
    int tkey, mask;

    subInterior->light(RESP_ANSWER);		// light answer light
    if (buttons)
	tkey = buttons->waitKey();		// wait for subject
    subInterior->light(RESP_ALL_OFF);		// turn off answer light

    if (tkey != BOTH_BUTTONS)			// exit button(s)?
    {
	mask = lightMask2AFC(stimulus, tkey);
	subInterior->light(mask);		// light feedback light
    }
    return tkey;
}


void
ResponseDevice::pause ()
{
    if (buttons)
	(void) buttons->waitKey();		// wait for subject
}


/************************************************************************
** lightMask2AFC
**
**	Makes it easy to determine which lights get
** turned on during a 2AFC task.  It accepts a stimulus code and a
** response code, and generates the correct mask for turning on the
** proper combination of lights.  Input values for stimulus and
** response must be either 1 or 2.
**
*************************************************************************/

int
ResponseDevice::lightMask2AFC
(
    int stimulus,				// range: 1 or 2
    int response				// range: 1 or 2
)
{
    int mask;

    if
    (
	(stimulus < 1) || (stimulus > 2) ||
	(response < 1) || (response > 2)
    )
    {
	mask = RESP_FAILURE;
    }
    else
    {
	mask = 1 << ((stimulus-1) + (INTERVAL_1-1));
	mask = mask | (1 << ((response-1) + (RESPONSE_1-1)));
    }
    return mask;
}

