/************************************************************************
**
** @(#)tv_confg.cpp	12/09/93	Chris Ahlstrom
**
**	C++
**
**	Displays selectable directories and files related to a
** BasicConfiguration, and allows one to be picked.
**
*************************************************************************/

#define TV_CONFG_cpp

#include <stdio.h>
#include <string.h>

#include "tv_confg.h"		// SimpleConfiguration class
#include "tv_confg.err"		// general configuration-file error codes




/************************************************************************
** SimpleConfiguration constructor
**
*************************************************************************/

const char * const
SimpleConfiguration::MAGIC = "SimpleConfiguration  ::12345678";

SimpleConfiguration::SimpleConfiguration		// constructor
(
    TDeskTop *desk,			// desktop to write onto
    char *filename,			// default filename for config file
    BasicConfiguration *cfglist,	// a list of data areas
    char *magicaddition			// magic string, if applicable
) :
    configList	(cfglist),
    errorCode	(0)
{
    magic = new char[CFG_MAGIC_STRING_LENGTH];
    if (magic)
	strcpy(magic, SimpleConfiguration::MAGIC);	// get a copy

    if (magic != 0 && magicaddition != 0 && *magicaddition != '\0')
    {
	char *dest = &magic[CFG_OFFSET];	// offset of personal magic
	int count = CFG_PERSONAL_MAGIC;		// size of personal magic
	char ch;

	while ((ch = *magicaddition++) != '\0')
	{
	    if (count-- == 0)
		break;				// ran out of personal magic
	    else
		*dest++ = ch;
	}
    }

    char wildcard[16];

    if (filename != 0 && *filename != '\0')
    {
	strcpy(wildcard, filename);	// try whatever the caller gave
    }
    else
    {
	strcpy(wildcard, "*.CFG");	// nothing, try default wildcard
    }

    configFile = new FilePicker(desk, wildcard);

    if (configFile->fileActive())		// we have a full name
    {
	errorCode = read();
    }
}


/************************************************************************
** SimpleConfiguration destructor
**
*************************************************************************/

SimpleConfiguration::~SimpleConfiguration ()		// constructor
{
    if (magic)
	delete [] magic;
    if (configFile)
	delete configFile;
}


/************************************************************************
*************************************************************************/

const char *
SimpleConfiguration::errorMsg ()
{
    if (errorCode < 0 || errorCode >= (int) ERR_CFG_MAX)
	return "Error code out of SimpleConfig range";
    else
	if (errorCode)
	    return simpleConfigErrors[errorCode];
	else
	    return (char *) 0;
}


/************************************************************************
** read()
**
**	Reads a configuration into memory.  Not much error-checking
** is done.
**
**	If successful, a 0 is returned, otherwise an integer
** corresponding to the ConfigurationError type.
**
*************************************************************************/

int
SimpleConfiguration::read
(
    ForceNameFlag newname
)
{
    int errcode = ERR_CFG_NONE;
    char magicbuffer[CFG_MAGIC_STRING_LENGTH];

    if (configFile)
    {
	if (configList)
	{
	    char *fname;
	    FILE *fptr;

	    if (newname)
		configFile->inactivateFile();

	    fname = configFile->fileSpec();
	    if (configFile->fileAction() == FILE_CANCELLED)
	    {
		errorCode = errcode;
		return errcode;
	    }

	    if ((fptr = fopen(fname, "rb")) != NULL)
	    {
		size_t items;		// number of items successfully written
		long size = 0L;		// accumulation of items
		size_t needsize = 0;	// the necessary size
		BasicConfiguration *cfg = configList;

		items = fread
		(
		    magicbuffer, (size_t) CFG_MAGIC_STRING_LENGTH,
		    (size_t) 1, fptr
		);
		if (items < (size_t) 1)
		{
		    errcode = ERR_CFG_BAD_SECTION_READ;
		}
		else if (strcmp(magicbuffer, magic) != 0)
		{
		    errcode = ERR_CFG_WRONG_KIND_OF_FILE;
		}
		if (errcode)
		{
		    errorCode = errcode;
		    return errcode;
		}
		while (cfg->configuration != 0)
		{
		    needsize = cfg->configurationSize;
		    items = fread
		    (
			cfg->configuration, (size_t) needsize, (size_t) 1, fptr
		    );

		    if (items < (size_t) 1)
		    {
			errcode = ERR_CFG_BAD_SECTION_READ;
			break;
		    }
		    size += items;
		    cfg++;		// next device in config list
		}
		(void) fread(&size, sizeof(long), (size_t) 1, fptr);	// check
		if (fclose(fptr) == EOF)
		    errcode = (int) ERR_CFG_FILE_CLOSE;
	    }
	    else
		errcode = (int) ERR_CFG_FILE_OPEN;
	}
	else
	    errcode = (int) ERR_CFG_NO_CONFIG_LIST;
    }
    else
	errcode = (int) ERR_CFG_CANT_ASSOCIATE_A_FILE;

    errorCode = errcode;
    return errcode;
}


/************************************************************************
** write()
**
**	Writes a configuration into memory.  Not much error-checking
** is done.
**
**	If successful, a 0 is returned, otherwise an integer
** corresponding to the ConfigurationError type.
**
*************************************************************************/

int
SimpleConfiguration::write
(
    ForceNameFlag newname
)
{
    int errcode = ERR_CFG_NONE;

    if (configFile)
    {
	if (configList)
	{
	    char *fname;
	    FILE *fptr;

	    if (newname)
		configFile->inactivateFile();

	    fname = configFile->fileSpec();
	    if (configFile->fileAction() == FILE_CANCELLED)
	    {
		errorCode = errcode;
		return errcode;
	    }

	    if ((fptr = fopen(fname, "wb")) != NULL)
	    {
		size_t items;		// number of items successfully written
		size_t size = 0;	// accumulation of items
		long needsize = 0L;	// the necessary size
		BasicConfiguration *cfg = configList;

		items = fwrite
		(
		    magic, (size_t) CFG_MAGIC_STRING_LENGTH, (size_t) 1, fptr
		);
		if (items < (size_t) 1)
		{
		    errorCode = ERR_CFG_BAD_SECTION_READ;
		    return errorCode;
		}

		while (cfg->configuration != 0)
		{
		    needsize = cfg->configurationSize;
		    items = fwrite
		    (
			cfg->configuration, (size_t) needsize, (size_t) 1, fptr
		    );
		    if (items < (size_t) 1)
		    {
			errcode = ERR_CFG_BAD_SECTION_WRITE;
			break;
		    }
		    size += items;
		    cfg++;		// next device in config list
		}
		(void) fwrite(&size, sizeof(long), (size_t) 1, fptr);	// check
		if (fclose(fptr) == EOF)
		    errcode = (int) ERR_CFG_FILE_CLOSE;
	    }
	    else
		errcode = (int) ERR_CFG_FILE_OPEN;
	}
	else
	    errcode = (int) ERR_CFG_NO_CONFIG_LIST;
    }
    else
	errcode = (int) ERR_CFG_CANT_ASSOCIATE_A_FILE;

    errorCode = errcode;
    return errcode;
}

