/************************************************************************
**
** @(#)tv_menu.h	11/23/93	Chris Ahlstrom
**
**  ------------------------
**  73340.26!compuserve.com
**  ------------------------
**
**	C++ version of discmenu.h for FACE_TV applications.  This header
** file was simplified by removing any need for color codes and line codes.
** Let TurboVision handle that stuff, too.
**
**	We complicated it by making setups for two different types
** of arrays for describing dialog boxes.  In one, the array describes
** the location, label, type of data, and format of data display
** for each data item in the dialog box.  Optionally, an extra parameter
** lets the value be re-mapped for display.
**
**	In the other, each data item has it's own location and label,
** but the type of data, format of the display, and mapping are all
** specified by the data-type, which points into an array of data-types
** the program will support.
**
**	Ouch!  Too complex!  But read FACE_BOX.MEN for a demonstration
** of how this stuff can be used.
**
*************************************************************************/

#if !defined(TV_MENU_h)				// { TV_MENU_h
#define TV_MENU_h

#define Uses_TPoint
#include <objects.h>		// Turbo Vision header for C++ declarations

#include "bios_vid.h"		// PC video constructs for Ahlstrom programs
#include "datatype.h"		// the DataType enumeration type


/************************************************************************
** Main Dialog Box for FACE_TV, C++ code
**
**	The structure defined herein are adapted from my old SPAM
** code (see DISCMENU.H or the DISCMENU library), and translated, as
** much as possible, into Turbo Vision.
**
**	As in DISCMENU, the programmer creates a large data structure
** describing the labels and the data fields.  In the present version,
** however, Turbo Vision takes over the managment.  Let's recapitulate
** only what the programmer needs to know.
**
*************************************************************************
**
**	Here is a TV dialog box that illustrates some intended features:
**
**
**   size and
**   location---
**	of	|
**     box	v
**		 =[*]=======Main text string============
**		|					|
**		| Name:  Chris Ahlstrom  MI: J		|
**		|					|
**		| Birthdate: 07/07/57			|
**		|					|
**		| Age: 34				|
**		|					|
**	 ------>| Income: $10000			|
**	|	|			 		|
**	|	|	  [  OK  ]     [Cancel]		|
**	|	|			 		|
**	|	 =======================================
**	|		     ^
**	|		     |
**	|____ Field label,   |_________	Data location,
**	      label length,		length,
**	      label location,		type,
**					and min/max values
**
**
**	Not shown is the main text string, which was used to write all
** text that has nothing to do with fields.  Now, however, the programmer
** incorporates such non-functional strings into the dialog box routine.
** A lot more flexibility, from Turbo Vision.
**
**	The colons are not included automatically.  The label/field
** locations are relative to the box's upper left corner (ignoring the
** row and column taken up by the border, just as the
** box is located relative to the upper left corner of the screen.
** The data box location is relative to the label (even if the label is
** null!)
**
**	Also not shown is the source for initialization of the
** field data-strings, which is the fields themselves.
**
**	To figure out the structure, look for the MainBox
** structure below.  As you follow it down to ever simpler data
** structure, read the comments.
**
**	The structures and defines below should be self-explanatory!
**
** Styles of lines for Boxes:
**
**	There's no control over the style of boxes, except by over-
** riding the dialog-box routines of Turbo-Vision.
** 
*************************************************************************/

/************************************************************************
** Range
**
**	Each Range value describes the limits of parameters for a given
** DataType.  Normally, two ranges are needed:  one is the range of
** values that the device requires, and the other is the in units the
** user likes.  The code converts between the two types of units.
**
*************************************************************************/

typedef struct
{
    float minimum;
    float maximum;

} Range;

#define RNG_STRING	    -1		// use range strings
#define RNG_MAP		    -2		// use range-mapping function
#define NULL_STRING	   NULL		// just for readability
#define NULL_STRINGS_LIST  NULL		// just for readability
#define NULL_FORMAT	   NULL		// just for readability


/************************************************************************
** ------------------
** ExtendedField
** ------------------
**
**	Used with Extended.
**
**	This type is useful in extending OLD PROGRAMS.  It is similar
** to MappedFieldType below, but only contains extra information
** that the old TextData structures did not have.
**
**	See FACE_BOX.H and FACE_BOX.MEN for examples of how it is used.
**
**	Usually, the programmer will add a new type which is an enum,
** and use that, casting it to a plain old ExtendedField where necessary.
**
**	Each value of ExtendedField is a bit like an index... it tells
** what kind of data-handling is to be used.  The programmer will
** provide an array of ExtendedDescriptors, into which the
** ExtendedField will be an index.
**
**	ExtendedField is used to add an infinite set of much more flexible
** datatypes than those provided in TINPUT.H; see FACE_BOX.MEN for
** a fairly well-documented demonstration.
**
** -----------------------
** ExtendedDescriptor
** -----------------------
**
**	This type is useful in extending OLD PROGRAMS.  It is similar
** to MappedFieldDescriptor below, but only contains extra information
** that the old TextData structures did not have.
**
**	See FACE_BOX.H and FACE_BOX.MEN for examples of how it is used.
**
**	This structure encapsulates the properties of any kind of
** parameter byte.
**
**	Normally, a list of fields will have a simple DataType.
** But, if the DataType is equal to Extended, this signals that
** the programmer wants to manipulate the data before and after
** editing (e.g. to display numbers in a way suitable for human
** consumption, as opposed to machine consumption.)
**
**	In this case, doDialog() will look further, into one of
** the structures below, to see what work to do.
**
**	The way this is set up, the caller can check to be sure
** that "type" is equal to the type used to reach this type. Ha ha.
**
*************************************************************************/

#define BOGUS_EXTENDED_FIELD	((ExtendedField) 32767)

typedef int ExtendedField;	// any enum will do as a match for int

typedef struct
{
    ExtendedField type;		// what kind of byte is it?
    DataType datatype;		// how should we show it onscreen?
    Range user;			// numeric range in user's format

} ExtendedDescriptor;


/************************************************************************
** ------------------
** MappedFieldType
** ------------------
**
**	This type is similar to the above, but helps generate shorter
** dialog-box field lists, at the expense of having to define a
** separate array to determine what to do with the data.
**
**	And having another whole function to deal with it.
**
** -----------------------
** MappedFieldDescriptor
** -----------------------
**
**	This describes everything about a data item, but in a compact
** format.  It achieves this by moving all information about the
** format and display of the data (except for locations in the
** dialog box) into an array element describing a "type" of data.
**
*************************************************************************/

#define BOGUS_MAPPED_FIELD_TYPE	((MappedFieldType) 32767)

typedef int MappedFieldType;	// you can use any enum to match this

typedef struct
{
    MappedFieldType type;	// what kind of byte is it?
    char *dtmplate;		// template string
    DataType datatype;		// how should we show it onscreen?
    Range code;			// the actual numeric range
    Range user;			// the range for data mapping
    char *format;		// the format to use
    char **strings;		// if applicable, as list of values

} MappedFieldDescriptor;


/***********************************************************************
** CtrlFunction
**
**	A pointer to a function which will alter the functioning of
** another routine.  This function is a link between the mouse code
** which outputs a numeric value to use, and the user's code which
** takes that value and uses it to set parameters on devices, live!
**
**	The input is the parameter (as picked by the current change
** in position of the mouse).  The output is an integer error code
** (currently unused).
**
************************************************************************/

typedef int (*CtrlFunction)(double);		// pointer to function


/***********************************************************************
** MapType
**
**	Lists descriptive names for the type of mappings we currently
** support.
**
************************************************************************/

typedef enum
{
    LINEAR	= 0,
    EXPONENTIAL

} MapType;


/***********************************************************************
** MouseMap
**
**	Yet another structure to pass data to the dodialog() routine.
**
************************************************************************/

typedef struct
{
    MapType mapType;		// the type of mapping function to use
    int maxMickey;		// maximum value of mouse "position"
    int maxKey;			// maximum value of key "position"
    CtrlFunction ctrlFunction;	// the control function to execute

} MouseMap;


/************************************************************************
** TextBox
**
**	TextBox defines everything that has nothing to do with data.
** The only reason it still remains in this file is that it might be
** useful in the future to alter, at run-time, the way Turbo Vision
** displays the dialog box.
**
**	string defines the main text string.  This string must be
** assembled in a fairly tedious manner, so that each word of it appears
** at the proper place in the menu.
**
**	labelloc is the location of string.  (This is different from
** the old SPAM version of discmenu.h.)
**
**	location locates the upper-left corner of the box on the video
** screen.
**
**	size indicates the extent of the box.  There is no error checking
** until you run the program and see how crappy the box looks.
**
**	linetype determines the characters used to draw the border.
** See the LineStyle definitions elsewhere in this module.  But they
** don't exist anymore in the Turbo Vision version.
**
**	Note that the C++ version eliminates any specification of
** colors.  If different colors are wanted, we'll have to derive to
** objects using different palettes, as recommended in the Turbo Vision
** manual.
**
*************************************************************************/

typedef struct
{
    char *title;		// desired title banner for this box
    char *string;		// string containing extraneous text in box
    TPoint location;		// location of upper left corner of box
    TPoint size;		// size of box in number of chars and rows
    TPoint labelloc;		// global offset to apply to labels

} TextBox;


/************************************************************************
** TextLabel
**
**	This is a label to associate with a data field.  Labels can
** occupy more than one line of text; using "\n" in the labels causes
** cursor movement to the left side of the box defining the label.
**
**	There is no checking whether the label goes outside its box,
** though.
**
**	string defines the label; use \n if desired.
**
**	PC-graphics characters can be used, since no mathematics is done
** on the "char" type used.
**
**	location determines where the label goes; it can even
** go outside the main box.
**
**	There is no size parameter... the routine banner_size()
** [boxtools.cpp] determines the 2-dimensional size of the label.
**
*************************************************************************/

typedef struct
{
    char *string;		// the string to use for the label
    TPoint location;		// location relative to upper left box corner

} TextLabel;			// describes each feature of a label


/************************************************************************
** TextData
**
**	This structure contains everything you ever need to know about
** a data item.  I'm not sure yet how all this will fit in with Turbo
** vision, however.  Hence, I haven't altered any of the descriptions
** below.
**
**	dtmplate	The ASCII representation of the data item, as
**			it should look on the screen.  This item
**	determines the size of the little box the data is shown in.
**	PC-graphics characters can be used, since no mathematics is done
**	on the "char" type used.  But, for numeric types, such characters
**	have no meaning.
**
**	location	This and size define the attributes of the box
**			containing the data.  The location is relative
**	to the start of the corresponding TextLabel.  Thus, it is easy
**	to move both label and data at the same time.  The data can be
**	located anywhere relative to the label.  It is best somewhere
**	near the label.  The size is determined by using banner_size()
**	[boxtools.cpp] to determine the size of the template.
**
**	datatype	Determines whether the data is numeric, text,
**			or some kind of button-box.  See the DataType
**	enumeration defined elsewhere in this module.
**
**	format		Tells the program how to convert a value into
**			the string to be displayed.  The format
**	specifier is a string of the kind that formats output in printf()
**	and sprintf().  The length part of the format specifier *must*
**	be equal to the actual length of the dtmplate, not including
**	the null.
**
**	code		The minimum and maximum fields of this Range
**			structure are floating-point versions of the
**	ranges of values allowed for that field.  They apply only to
**	numeric types.
**
**	spec		This is a pointer to a union.  Each of the
**			structures unioned together are related to
**	at least one DataType. See the description of TextDataSpec below.
**
**
** TextDataSpec
**
**	This is a union that can be one of the following:
**
**	1.  A pointer to a list of strings for
**
**		Radiobox

**		Checkbox
**		Radiobyte
**		Checkbyte
**
**	    To handle Turbo Vision's check-boxes and radio-button
**	    lists, we provide pointer to an array of strings to
**	    use to label each of the selectable items.  It is called
**	    "strings", not to be confused with "string".  The array
**	    pointed to must be terminated with a NULL pointer.
**	    See FACE_BOX.MEN for an example.
**
**	2.  A integer (ExtendedField) that serves as an index for
**
**		Extended
**
**	    Right now, dtype is still part of TextData, so code
**	    won't yet have to be changed.
**
**	3.  A pointer to a mouse/keystroke mapping structure (MouseMap)
**	    used to specify the mapping between mouse movement and
**	    numeric control, and perhaps a control function to be called
**	    after every mouse movement.  Used only with
**
**		FloatControl
**
**	4.  For all other DataTypes, this union ("spec") is unused.
**
*************************************************************************/

typedef union
{
    char **strings;		// labels of selectable buttons
    ExtendedField dtype;	// if type == Extended, use this
    MouseMap *ctrlmap;		// parameters of mouse/key control

} TextDataSpec;

typedef struct
{
    char *dtmplate;		// template string and character buffer
    TPoint location;		// location relative to beginning of label
    DataType datatype;		// kind of data (string, number, etc.
    char *format;		// format of data (Unix printf format)
    Range code;			// minimum and maximum values
    TextDataSpec spec;		// was "char **strings"

} TextData;			// describes each feature of a data item


/************************************************************************
**
** -----------
** MenuField
** -----------
**
**	This structure combines the label and data, for easy reference.
** The programmer should define an array of MenuFields, and
** then set the "fields" (note the plural spelling) value to the
** address of this array.
**
** -----------
** MappedField
** -----------
**
**	This structure combines the label and data, for easy reference.
**
**	This is a label to associate with a data field.  Labels can
** occupy more than one line of text; using "\n" in the labels causes
** cursor movement to the left side of the box defining the label.
** There is no checking whether the label goes outside its box, and
** the label location is relative to the dialog box.
**
**	string defines the label; use \n if desired.
**
**	PC-graphics characters can be used, since no mathematics is done
** on the "char" type used.
**
**	location and size determine where the labels go; they can even
** go outside the main box.
**
**	The programmer must make sure that the length of the string
** (ignoring the null) is equal to (size.x * size.y).
**
**	"type" points to everything you ever need to know about
** a data item that is not related to the final display of the data
** item.
**
**	location and size define the attributes of the box
** containing the data.  The location is relative to the start of
** the corresponding label.  Thus, it is easy to move both
** label and data at the same time.  The data can be located anywhere
** relative to the label.  Of course, it is best to have it somewhere
** nearby the label.
**
** ------------
** Summary
** ------------
**
**	A good way to think of it is as follows:
**
**	A MenuField lets you describe different ways of displaying
** any one kind of data.  If the MenuField is an Extended datatype,
** then you can also specify a user-range to which the data item will
** be transformed for viewing.
**
**	A MappedField, however, hooks into a set of new datatypes,
** and each of these datatypes can only be displayed one way... the
** display format and re-mapping is tied to the datatype.
**
*************************************************************************/

typedef struct
{
    TextLabel label;		// a structure describing the label
    TextData data;		// a structure describing the data field

} MenuField;			// a menu field consists of two structures


typedef struct
{
    char *label_string;		// the string to use for the label
    TPoint label_location;	// location relative to upper left box corner
    TPoint data_location;	// location relative to label location
    MappedFieldType dtype;	// type of data (determines a lot of things!)

} MappedField;


/************************************************************************
** ---------
** MainBox
** ---------
**
**	The main part is the TextBox, which defines everything that is
** not concerned with data.
**
**	Again, at this time I'm not sure how all this will fit in with
** Turbo Vision.
**
**	fieldcount is the number of data fields defined.
**
**	fields is the array of MenuField structures that describes
** all data fields.  Whereas SoftCode generates individual calls
** for each data field (at least in the Pascal version I saw),
** We simply examine each array element to determine what it should
** do.
**
** ---------
** MappedBox
** ---------
**
**	Needed for handling MappedFields instead of MenuFields, this
** structure also does away with some of the unnecessary complications
** of MainBox.
**
**	fieldcount is the number of data fields defined.
**
**	fields is the array of MappedField structures that describes
** everything about each data field.
**
*************************************************************************/

typedef struct
{
    TextBox box;		// structure defining the box
    int fieldcount;		// the number of fields defined
    MenuField *fields;		// pointer to array of field structures

} MainBox;


typedef struct
{
    TextBox box;		// structure defining the box
    int fieldcount;		// the number of fields defined
    MappedField *fields;	// pointer to array of field structures

} MappedBox;


#endif							// } TV_MENU_h
