/************************************************************************
**
** @(#)tinput.cpp	06/09/93	Chris Ahlstrom
**
** "Variations on TInputLine in C++", Op. 2.12
**
**	This module provides objects that inherit capabilities
** from TInputLine, but then extends them to handle objects other
** than strings.
**
**	Also see tinpmous.cpp for more derived objects of amazing power!
**
*************************************************************************/

#define TINPUT_cpp

#include <math.h>		// for the floor() function
#include "tinput.h"
#include "tinpextd.h"

__link(RByteString)		// ready them all for streaming
__link(RInputByte)
__link(RInputInteger)
__link(RInputFloat)
__link(RInputLong)
__link(RByteRadioButtons)
__link(RByteCheckBoxes)


/************************************************************************
** TByteString
**
**	Similar to TInputLine.  However, TInputLine requires at least
** one null on the end of the string.  TByteString gets around that
** by copying the string to a data buffer.
**
**	Therefore, the length parameter provided to TByteString
** must not include the byte for the null character.
**
**	We do not have to override much here.
**
**	First, the size of our data is really cmaxLen; through the
** internecine circumlocutions of calls, we find (via the debugger)
** that maxLen ends up being the size we want, which doesn't include
** any bytes for the null (this is good).  So, we return maxLen instead
** of maxLen+1 as the dataSize().
**
*************************************************************************/

const char * const TByteString::name = "TByteString";


TByteString::TByteString
(
    const TRect& bounds,	// rectangle for the Byte's input-line
    int cmaxLen			// the maximum length to use for buffer
)
    : TInputLine(bounds, cmaxLen)
{
}


ushort
TByteString::dataSize()
{
    return maxLen;		// instead of the default, maxLen+1
}


void
TByteString::getData
(
    void *rec
)
{
    strncpy((char *)rec, data, maxLen);
}


void
TByteString::setData
(
    void *rec
)
{
    strnset(data, '\0', maxLen);		// clear the data buffer
    strncpy(data, (char *)rec, maxLen);		// copy string to data
    data[maxLen] = EOS;				// guarantee terminating null
    selectAll(True);				// select the whole field
}


TStreamable *
TByteString::build ()
{
    return new TByteString(streamableInit);
}


TStreamableClass
RByteString
(
    TByteString::name,
    TByteString::build,
    __DELTA(TByteString)
);


/************************************************************************
** TInputByte
**
**	Accepts unsigned byte inputs, but only accepts them in numeric
** format.  Also accepts hexadecimal numbers if expressed in the
** format "0x00".
**
*************************************************************************/

const char * const TInputByte::name = "TInputByte";

TInputByte::TInputByte
(
    const TRect& bounds,		// rectangle for Byte's input-line
    int bmaxLen,			// the length of the buffer
    const Range& code,			// range of internal representation
    char *formatstring,			// desired display format
    int mapped,				// should remapping be performed?
    const Range& user			// range of user's representation
) :
    TExtended(bounds, bmaxLen, code, formatstring, mapped, user)
{
    format  = formatstring;
}


ushort
TInputByte::dataSize()
{
    return sizeof(unsigned char);
}


double
TInputByte::getValue ()
{
    return (double) Value;
}


void
TInputByte::putValue
(
    double value
)
{
    value = floor(value + 0.5);		// round up number (cast can't do it)
    Value = (unsigned char) value;
}


void
TInputByte::updateValue ()
{
    setData(&Value);
}


void
TInputByte::getData
(
    void *rec
)
{
    char buffer[intBufSize];
    long temp;

    temp = strtol(data, NULL, 0);	// convert based only on input
    if (temp > 255 || temp < 0)
	temp = 255;
    Value = (char) temp;
    sprintf(buffer, (format == NULL) ? "%2x" : format, (int) Value);
    strnset(data, '\0', maxLen);
    strncpy(data, buffer, maxLen);
    data[maxLen] = EOS;
    drawView();				// do we really need this???
    mapToCode();			// perhaps convert the Value
    memcpy(rec, &Value, sizeof(Value));
}


void
TInputByte::setData
(
    void *rec
)
{
    char buffer[intBufSize];		// temporary buffer

    memcpy(&Value, rec, sizeof(Value));	// copy byte to value
    mapToUser();			// perhaps convert the Value
    sprintf(buffer, (format == NULL) ? "%2x" : format, (int) Value);
    strnset(data, '\0', maxLen);	// clear the data buffer
    strncpy(data, buffer, maxLen);	// copy converted number to buffer
    data[maxLen] = EOS;			// guarantee the terminating null
    selectAll(True);			// select the whole field
}


Boolean
TInputByte::valid
(
    ushort command
)
{
    long value;
    Boolean ok;
    char msg[80];
    ostrstream os(msg, 80);

    ok = True;
    if ((command != cmCancel) && (command != cmValid))
    {
	if (strlen(data) == 0)
	    strcpy(data, "0");

	value = (long) ((char) strtol(data, NULL, 0));
	if (value < userMin || value > userMax)
	{
	    select();
	    os << "Number outside range " << userMin
		<< ".." << userMax << ends;
	    messageBox(os.str(), mfError + mfOKButton);
	    selectAll(True);
	    ok = False;
	}
    }
    if (ok)
	return TInputLine::valid(command);
    else
	return False;
}


void
TInputByte::write
(
    opstream& os
)
{
    TInputLine::write(os);
    os << userMin;
    os << userMax;
    
}


void *
TInputByte::read
(
    ipstream& is
)
{
    TInputLine::read(is);
    is >> userMin;
    is >> userMax;
    return this;
}


TStreamable *
TInputByte::build ()
{
    return new TInputByte(streamableInit);
}


TStreamableClass
RInputByte
(
    TInputByte::name,
    TInputByte::build,
    __DELTA(TInputByte)
);


/************************************************************************
** TInputSignedByte
**
**	Accepts signed byte inputs, but only accepts them in numeric
** format.  Also accepts hexadecimal numbers if expressed in the
** format "0x00".
**
*************************************************************************/

const char * const TInputSignedByte::name = "TInputSignedByte";

TInputSignedByte::TInputSignedByte
(
    const TRect& bounds,		// rectangle for Byte's input-line
    int bmaxLen,			// the length of the buffer
    const Range& code,			// range of internal representation
    char *formatstring,			// desired display format
    int mapped,				// should remapping be performed?
    const Range& user			// range of user's representation
) :
    TInputByte(bounds, bmaxLen, code, formatstring, mapped, user)
{
    format  = formatstring;
}


double
TInputSignedByte::getValue ()
{
    return (double) Value;
}


void
TInputSignedByte::putValue
(
    double value
)
{
    value = floor(value + 0.5);		// round up number (cast can't do it)
    Value = (signed char) value;
}


void
TInputSignedByte::updateValue ()
{
    setData(&Value);
}


void
TInputSignedByte::getData
(
    void *rec
)
{
    char buffer[intBufSize];
    long temp;

    temp = strtol(data, NULL, 0);	// convert based only on input
    Value = (char) temp;
    sprintf(buffer, (format == NULL) ? "%2x" : format, (int) Value);
    strnset(data, '\0', maxLen);
    strncpy(data, buffer, maxLen);
    data[maxLen] = EOS;
    drawView();				// do we really need this???
    mapToCode();			// perhaps convert the Value
    memcpy(rec, &Value, sizeof(Value));
}


void
TInputSignedByte::setData
(
    void *rec
)
{
    char buffer[intBufSize];		// temporary buffer

    memcpy(&Value, rec, sizeof(Value));	// copy byte to value
    mapToUser();			// perhaps convert the Value
    sprintf(buffer, (format == NULL) ? "%2x" : format, (int) Value);
    strnset(data, '\0', maxLen);	// clear the data buffer
    strncpy(data, buffer, maxLen);	// copy converted number to buffer
    data[maxLen] = EOS;			// guarantee the terminating null
    selectAll(True);			// select the whole field
}


TStreamable *
TInputSignedByte::build ()
{
    return new TInputSignedByte(streamableInit);
}


TStreamableClass
RInputSignedByte
(
    TInputSignedByte::name,
    TInputSignedByte::build,
    __DELTA(TInputSignedByte)
);


/************************************************************************
** TInputInteger
**
**	Accepts signed integer inputs.  Also accepts hexadecimal numbers
** if expressed in the format "0x000..."
**
*************************************************************************/

const char * const TInputInteger::name = "TInputInteger";

TInputInteger::TInputInteger
(
    const TRect& bounds,		// rectangle for integer's input-line
    int imaxLen,			// maximum length to use for buffer
    const Range& code,			// range of internal representation
    char *formatstring,			// desired display format
    int mapped,				// should remapping be performed?
    const Range& user			// range of user's representation
) :
    TExtended(bounds, imaxLen, code, formatstring, mapped, user)
{
    format  = formatstring;
}


ushort
TInputInteger::dataSize()
{
    return sizeof(int);
}


double
TInputInteger::getValue ()
{
    return (double) Value;
}

void
TInputInteger::putValue
(
    double value
)
{
    value = floor(value + 0.5);		// round up number (cast can't do it)
    Value = (int) value;
}


void
TInputInteger::updateValue ()
{
    setData(&Value);
}


void
TInputInteger::getData
(
    void *rec
)
{
    char buffer[intBufSize];
    long temp;

    temp = strtol(data, NULL, 0);	// convert based only on input
    if (temp > 65536L || temp < -32768L)
	temp = 65536L;
    Value = (int) temp;
    sprintf(buffer, (format == NULL) ? "%d" : format, Value);
    strnset(data, '\0', maxLen);
    strncpy(data, buffer, maxLen);
    data[maxLen] = EOS;
    drawView();				// do we really need this???
    mapToCode();			// perhaps convert the Value
    memcpy(rec, &Value, sizeof(Value));
}


void
TInputInteger::setData
(
    void *rec
)
{
    char buffer[intBufSize];		// temporary buffer

    memcpy(&Value, rec, sizeof(Value));	// copy integer to value
    mapToUser();			// perhaps convert the Value
    sprintf(buffer, (format == NULL) ? "%d" : format, Value);
    strnset(data, '\0', maxLen);	// clear the data buffer
    strncpy(data, buffer, maxLen);	// copy converted number to buffer
    data[maxLen] = EOS;			// guarantee the terminating null
    selectAll(True);			// select the whole field
}


Boolean
TInputInteger::valid
(
    ushort command
)
{
    int value;
    Boolean ok;
    char msg[80];
    ostrstream os(msg, 80);

    ok = True;
    if ((command != cmCancel) && (command != cmValid))
    {
	if (strlen(data) == 0)
	    strcpy(data, "0");

	value = (int) strtol(data, NULL, 0);
	if (value < userMin || value > userMax)
	{
	    select();
	    os << "Number outside range " << userMin
	       << ".." << userMax << ends;
	    messageBox(os.str(), mfError + mfOKButton);
	    selectAll(True);
	    ok = False;
	}
    }
    if (ok)
	return TInputLine::valid(command);
    else
	return False;
}


void
TInputInteger::write
(
    opstream& os
)
{
    TInputLine::write(os);
    os << userMin;
    os << userMax;
    
}

void *
TInputInteger::read
(
    ipstream& is
)
{
    TInputLine::read(is);
    is >> userMin;
    is >> userMax;
    return this;
}

TStreamable *
TInputInteger::build ()
{
    return new TInputInteger(streamableInit);
}


TStreamableClass
RInputInteger
(
    TInputInteger::name,
    TInputInteger::build,
    __DELTA(TInputInteger)
);


/************************************************************************
** TInputFloat
*************************************************************************/

const char * const TInputFloat::name = "TInputFloat";

TInputFloat::TInputFloat
(
    const TRect& bounds,		// rectangle for Float's input-line
    int fmaxLen,			// maximum length to use for buffer
    const Range& code,			// range of internal representation
    char *formatstring,			// desired display format
    int mapped,				// should remapping be performed?
    const Range& user			// range of user's representation
) :
    TExtended(bounds, fmaxLen, code, formatstring, mapped, user)
{
    format  = formatstring;
}


ushort
TInputFloat::dataSize()
{
    return sizeof(double);
}


double
TInputFloat::getValue ()
{
    return Value;
}


void
TInputFloat::putValue
(
    double value
)
{
    Value = value;
}


void
TInputFloat::updateValue ()
{
    setData(&Value);
}


void
TInputFloat::getData
(
    void *rec
)
{
    char buffer[floatBufSize];

    Value = atof(data);
    sprintf(buffer, (format == NULL) ? "%g" : format, Value);
    strnset(data, '\0', maxLen);
    strncpy(data, buffer, maxLen);
    data[maxLen] = EOS;
    drawView();				// do we really need this???
    mapToCode();			// perhaps convert the Value
    memcpy(rec, &Value, sizeof(Value));
}


void
TInputFloat::setData
(
    void *rec
)
{
    char buffer[floatBufSize];

    memcpy(&Value, rec, sizeof(Value));
    mapToUser();			// perhaps convert the Value
    sprintf(buffer, (format == NULL) ? "%g" : format, Value);
    strnset(data, '\0', maxLen);
    strncpy(data, buffer, maxLen);
    data[maxLen] = EOS;
    selectAll(True);
}


Boolean
TInputFloat::valid
(
    ushort command
)
{
    double value;
    Boolean ok;
    char msg[80];
    ostrstream os(msg, 80);

    ok = True;
    if ((command != cmCancel) && (command != cmValid))
    {
	if (strlen(data) == 0)
	    strcpy(data, "0");
	value = atof(data);
	if (value < userMin || value > userMax)
	{
	    select();
	    os << "Number outside range " << userMin
	       << ".." << userMax << ends;
	    messageBox(os.str(), mfError + mfOKButton);
	    selectAll(True);
	    ok = False;
	}
    }
    if (ok)
	return TInputLine::valid(command);
    else
	return False;
}


void
TInputFloat::write
(
    opstream& os
)
{
    TInputLine::write(os);
    os << userMin;
    os << userMax;
}

void *
TInputFloat::read
(
    ipstream& is
)
{
    TInputLine::read(is);
    is >> userMin;
    is >> userMax;
    return this;
}

TStreamable *
TInputFloat::build ()
{
    return new TInputFloat(streamableInit);
}


TStreamableClass
RInputFloat
(
    TInputFloat::name,
    TInputFloat::build,
    __DELTA(TInputFloat)
);


/*************************************************************************
** >> and << operators for doubles
**
**	These operators make up for some deficiencies in Turbo Vision's
** library (TOBJSTRM.CPP).
**
**	Later, if a newer version of TV doesn't fix it, I shall make
** a separate module for it.
**
**************************************************************************/

ipstream& operator >> ( ipstream& ps, double &d )
{
    ps.readBytes(&d, sizeof(d));
    return ps;
}

opstream& operator << ( opstream& ps, double d )
{
    ps.writeBytes(&d, sizeof(d));
    return ps;
}



/************************************************************************
** TInputLong
**
**	This code is only reformatted and very slightly modified from
** BC\TVISION\DEMOS\FIELDS.CPP.
**
*************************************************************************/

const char * const TInputLong::name = "TInputLong";

TInputLong::TInputLong
(
    const TRect& bounds,		// rectangle for Long display
    int lmaxLen,			// length of buffer
    const Range& code,			// range of internal representation
    char *formatstring,			// desired display format
    int mapped,				// should remapping be performed?
    const Range& user			// range of user's representation
) :
    TExtended(bounds, lmaxLen, code, formatstring, mapped, user)
{
    format  = formatstring;
}


ushort
TInputLong::dataSize ()
{
    return sizeof(long);
}


double
TInputLong::getValue ()
{
    return (double) Value;
}


void
TInputLong::updateValue ()
{
    setData(&Value);
}


void
TInputLong::putValue
(
    double value
)
{
    value = floor(value + 0.5);		// round up number (cast can't do it)
    Value = (long) value;
}


void
TInputLong::getData
(
    void *rec
)
{
    char buffer[longBufSize];

    Value = strtol(data, NULL, 0);	// convert based only on input
    sprintf(buffer, (format == NULL) ? "%l" : format, Value);
    strnset(data, '\0', maxLen);
    strncpy(data, buffer, maxLen);
    data[maxLen] = EOS;
    drawView();				// do we really need this???
    mapToCode();			// perhaps convert the Value
    memcpy(rec, &Value, sizeof(Value));
}


void
TInputLong::setData
(
    void *rec
)
{
    char buffer[longBufSize];		// temporary buffer

    memcpy(&Value, rec, sizeof(Value));	// copy integer to value
    mapToUser();			// perhaps convert the Value
    sprintf(buffer, (format == NULL) ? "%l" : format, Value);
    strnset(data, '\0', maxLen);	// clear the data buffer
    strncpy(data, buffer, maxLen);	// copy converted number to buffer
    data[maxLen] = EOS;			// guarantee the terminating null
    selectAll(True);			// select the whole field
}


Boolean
TInputLong::valid
(
    ushort command
)
{
    long value;
    Boolean ok;
    char msg[80];
    ostrstream os(msg, 80);

    ok = True;
    if ((command != cmCancel) && (command != cmValid))
    {
	if (strlen(data) == 0)
	    strcpy(data,"0");

	value = strtol(data, NULL, 0);
	if (value < userMin || value > userMax)
	{
	    select();
	    os << "Number outside range " << userMin
	       << ".." << userMax << ends;
	    messageBox(os.str(), mfError + mfOKButton);
	    selectAll(True);
	    ok = False;
	}
    }
    if (ok)
	return TInputLine::valid(command);
    else
	return False;
}

void
TInputLong::write
(
    opstream& os
)
{
    TInputLine::write(os);
    os << userMin;
    os << userMax;

}

void *
TInputLong::read
(
    ipstream& is
)
{
    TInputLine::read(is);
    is >> userMin;
    is >> userMax;
    return this;
}

TStreamable *
TInputLong::build ()
{
    return new TInputLong(streamableInit);
}


TStreamableClass
RInputLong
(
    TInputLong::name,
    TInputLong::build,
    __DELTA(TInputLong)
);




/************************************************************************
** TByteRadioButtons
**
**	Since the only difference between this class and its base,
** TRadioButtons, is the size of the data item, we don't have many
** functions to change.
**
*************************************************************************/

const char * const TByteRadioButtons::name = "TByteRadioButtons";


ushort
TByteRadioButtons::dataSize ()
{
    return sizeof(bytevalue);
}


void
TByteRadioButtons::getData
(
    void *rec
)
{
    *(unsigned char *)rec = bytevalue;
    drawView();				// do we really need this???
}


void
TByteRadioButtons::setData
(
    void *rec
)
{
    value = *(unsigned char *)rec;
    sel = value;
    bytevalue= (unsigned char) value;
    drawView();				// do we really need this???
}


Boolean
TByteRadioButtons::mark
(
    int item
)
{
    return Boolean(item == value);
}


void
TByteRadioButtons::press
(
    int item
)
{
    value = item;
    bytevalue = (unsigned char) item;
}


void
TByteRadioButtons::movedTo
(
    int item
)
{
    value = item;
    bytevalue = (unsigned char) item;
}


TStreamable *
TByteRadioButtons::build ()
{
    return new TByteRadioButtons(streamableInit);
}


TStreamableClass
RByteRadioButtons
(
    TByteRadioButtons::name,
    TByteRadioButtons::build,
    __DELTA(TByteRadioButtons)
);


/************************************************************************
** TByteCheckBoxes
**
**	Since the only difference between this class and its base,
** TRadioButtons, is the size of the data item, we don't have many
** functions to change.
**
*************************************************************************/

const char * const TByteCheckBoxes::name = "TByteCheckBoxes";


ushort
TByteCheckBoxes::dataSize ()
{
    return sizeof(bytevalue);
}


void
TByteCheckBoxes::getData
(
    void *rec
)
{
    *(unsigned char *)rec = bytevalue;
    drawView();				// do we really need this???
}


void
TByteCheckBoxes::setData
(
    void *rec
)
{
    value = *(unsigned char *)rec;
    sel = value;
    bytevalue= (unsigned char) value;
    drawView();				// do we really need this???
}


Boolean
TByteCheckBoxes::mark
(
    int item
)
{
    return Boolean((value & (1 << item)) != 0);
}


void
TByteCheckBoxes::press
(
    int item
)
{
    value = value ^ (1 << item);
    bytevalue = (unsigned char) value;
}


TStreamable *
TByteCheckBoxes::build ()
{
    return new TByteCheckBoxes(streamableInit);
}


#if JUNK
void
TByteCheckBoxes::write
(
    opstream& os
)
{
    TView::write(os);
    os << bytevalue << sel << strings;
}


void *
TByteCheckBoxes::read
(
    ipstream& is
)
{
    TView::read(is);
    is >> bytevalue >> sel >> strings;
    setCursor(2, 0);
    showCursor();
    return this;
}
#endif

TStreamableClass
RByteCheckBoxes
(
    TByteCheckBoxes::name,
    TByteCheckBoxes::build,
    __DELTA(TByteCheckBoxes)
);


/************************************************************************
** Others, in tv_menu.h, that aren't done yet:
**
** TInputHex
** TInputDate
** TInputMoney
**
*************************************************************************/
