/************************************************************************
**
** @(#)grafxscr.h	10/08/93	Chris Ahlstrom
**
**	The Screen structure describes everything about the status
** of a text screen.
**
**	This module is self-contained -- it does not refer to any
** other structures or data types.  Hence, any program can use it,
** without the need for a seemingly endless list of secondary
** include files.
**
**	See bios_vid.h for more information.
**
*************************************************************************/


#ifndef GRAFXSCR_h					/* { GRAFXSCR_h	*/
#define GRAFXSCR_h

#include <graphics.h>

#include "bios_vid.h"


/************************************************************************
** -----------------------------------------------------
** Fundamental Data Types that differ between compilers
** -----------------------------------------------------
**
** ---------------
** GraphicsDriver
** ---------------
**
**	Basically, this is a code number related to the kind of video
** the routines are to handle.  For the Borland compiler, this type
** will be equivalent to the following enumeration:
**
**	DETECT		= 0	(requests autodetection)
**	CGA
**	MCGA
**	EGA
**	EGA64
**	EGAMONO
**	IBM8514
**	HERCMONO
**	ATT400
**	VGA
**	PC3270
**	CURRENT_DRIVER = -1
**
** For the Microsoft compiler, this type will be a simpler enumeration,
** as can be seen below.
**
** ---------------
** GraphicsDriver
** ---------------
**
**	Basically, this is a code number related to the kind of video
** mode to be applied to the current GraphicsDriver.  For the Borland
** compiler, this type will be equivalent to the following enumeration:
**
**	CGAC0		= 0	320x200 palette 0; 1 page
**	CGAC1		= 1	320x200 palette 1; 1 page
**	CGAC2		= 2	320x200 palette 2: 1 page
**	CGAC3		= 3	320x200 palette 3; 1 page
**	CGAHI		= 4	640x200 1 page
**	MCGAC0		= 0	320x200 palette 0; 1 page
**	MCGAC1		= 1	320x200 palette 1; 1 page
**	MCGAC2		= 2	320x200 palette 2; 1 page
**	MCGAC3		= 3	320x200 palette 3; 1 page
**	MCGAMED		= 4	640x200 1 page
**	MCGAHI		= 5	640x480 1 page
**	EGALO		= 0	640x200 16 color 4 pages
**	EGAHI		= 1	640x350 16 color 2 pages
**	EGA64LO		= 0	640x200 16 color 1 page
**	EGA64HI		= 1	640x350 4 color  1 page
**	EGAMONOHI	= 0	640x350 64K card, 1 page, 256K card, 4 pages
**	HERCMONOHI	= 0	720x348 2 pages
**	ATT400C0	= 0	320x200 palette 0; 1 page
**	ATT400C1	= 1	320x200 palette 1; 1 page
**	ATT400C2	= 2	320x200 palette 2; 1 page
**	ATT400C3	= 3	320x200 palette 3; 1 page
**	ATT400MED	= 4	640x200 1 page
**	ATT400HI	= 5	640x400 1 page
**	VGALO		= 0	640x200 16 color 4 pages
**	VGAMED		= 1	640x350 16 color 2 pages
**	VGAHI		= 2	640x480 16 color 1 page
**	PC3270HI	= 0	720x350 1 page
**	IBM8514LO	= 0	640x480 256 colors
**	IBM8514HI	= 1    1024x768 256 colors
**
** For the Microsoft compiler, this will be int (that might change!),
** and we will specifically define VGALO, VGAMED, and VGAHI,
** although only the first two of these have universally applicable
** values.
**
** ---------------
** GraphicsError
** ---------------
**
**	Basically, this is a code number related to the kind of video
** error status.  For the Borland compiler, this type will be
** equivalent to the following enumeration:
**
**	grOk			=   0,
**	grNoInitGraph		=  -1,
**	grNotDetected		=  -2,
**	grFileNotFound		=  -3,
**	grInvalidDriver		=  -4,
**	grNoLoadMem		=  -5,
**	grNoScanMem		=  -6,
**	grNoFloodMem		=  -7,
**	grFontNotFound		=  -8,
**	grNoFontMem		=  -9,
**	grInvalidMode		= -10,
**	grError			= -11,   // generic error
**	grIOerror		= -12,
**	grInvalidFont		= -13,
**	grInvalidFontNum	= -14,
**	grInvalidVersion	= -18
**
** For the Microsoft compiler, this will be int, with only grOk
** specifically defined.
**
*************************************************************************/

#ifdef __TURBOC__				/* { __TURBOC__		*/

typedef enum graphics_drivers GraphicsDriver;
typedef enum graphics_modes GraphicsMode;
typedef enum graphics_errors GraphicsError;
typedef struct palettetype Palette;

#else						/* }{ __TURBOC__	*/

#define ADAPTER					/* override bios_vid.h	*/
typedef enum					/* minimal drivers	*/
{
    DETECT		= 0,
    CGA			= 1,
    CGA_DISPLAY		= 1,
    EGA			= 2,
    EGA_DISPLAY		= 2,
    VGA			= 3,
    VGA_DISPLAY		= 3,
    SUPER_VGA_DISPLAY,
    UNKNOWN_DISPLAY

} GraphicsDriver;

#define VGALO		0			/* minimal graphic mode	*/
#define VGAMED		1
#define VGAHI		2
typedef int GraphicsMode;

#define grOk		0			/* minimal error code	*/
typedef int GraphicsError;

#define MAXCOLORS 15				/* same as Borland	*/
typedef struct palettetype
{
    unsigned char size;
    signed char colors[MAXCOLORS+1];
} Palette;

#endif						/* } __TURBOC__		*/


/************************************************************************
** --------------------------------------------------
** Fundamental Data Types the same between compilers
** --------------------------------------------------
**
** -------------
** AdapterSelect
** -------------
**
**	0 is the main adapter, and 1 is the alternate adapter.
** The library attempts to keep the handling of each adapter well
** separate.
**
** ------
** Screen
** ------
**
**	Attempts to hold all possible information about the current
** screen configuration.
**
*************************************************************************/

typedef enum
{
    ADAPTER_MAIN,
    ADAPTER_ALT,
    ADAPTER_INACTIVE

} AdapterSelect;

typedef struct
{
    AdapterSelect adapter;		/* active adapter for this one	*/
    GraphicsMode screen_type;		/* see enum in bios_vid.h	*/
    GraphicsDriver screen_driver;	/* type of driver card		*/
    int screen_mode;			/* current video mode		*/
    int screen_page;			/* current video page		*/
    int screen_attribute;		/* foreground DOS screen color	*/
    int screen_backattribute;		/* background DOS screen color	*/
    int screen_rows;			/* number of rows on the screen	*/
    int screen_columns;			/* number of columns on screen	*/
    int screen_gmax_y;			/* maximum y-axis pixel offset	*/
    int screen_gmax_x;			/* maximum x-axis pixel offset	*/
    int screen_rasters;			/* # of rasters lines supported	*/
    int screen_charsets;		/* # ofcharacter sets supported	*/
    int screen_chardefs;		/* # of character definitions	*/
    int cursor_row;			/* location of cursor, y	*/
    int cursor_column;			/* location of cursor, x	*/
    int cursor_start_line;		/* starting line of cursor icon	*/
    int cursor_end_line;		/* ending line of cursor "icon"	*/
    int screen_colors;			/* maximum colors specified	*/
    double screen_aspect;		/* aspect ratio			*/
    Palette palette;			/* 17-byte palette structure	*/
    int error_code;			/* last error for this screen	*/
    char *screen_driver_name;		/* name of the screen driver	*/

} Screen;


/************************************************************************
** Functions defined in GRAFXSCR.C
*************************************************************************/

extern GraphicsError graphicsScreen
(
    Screen *s
);
extern int setscreen
(
    int mode,
    int page,
    int attribute,
    int backattribute,
    int rows,
    int columns,
    int currow,
    int curcolumn,
    int curstart,
    int curend
);
extern Screen *getScreenStructure (void);
extern void freeScreenStructure
(
    Screen *s
);
extern void initScreens
(
    Screen *main,
    Screen *alt
);
extern int initBGI
(
    Screen *s,				/* all screen parameters	*/
    int alt_adapter,			/* adapter to use (0 or 1)	*/
    int adapter_type			/* kind of adapter		*/
);
extern int fontinit
(
    Screen *s
);
extern int changetextstyle
(
    Screen *s,
    int font,
    int direction,
    int charsize
);
extern void video_main (void);
extern void video_alternate (void);
extern void tomain (void);
extern void toalternate (void);
extern int init1
(
    Screen *s,
    char *basefont
);


/************************************************************************
** Stuff privy to GRAFXSCR.C
*************************************************************************/

#define	ERR_SCREEN_BAD							\
	    "Screen structure was not obtained by legitimate means.\n"	\
	    "Therefore, I cannot allow you access to the BGI.\n"	\
	    "You must call getScreenStructure() to get a Screen.\n"

#endif							/* } GRAFXSCR_h	*/
