/************************************************************************
**
** @(#)grafxlog.c	01/05/94	Chris Ahlstrom
**
**	A module using Borland video functions, but meant to isolate
** this "Borlandness" from caller programs.
**
**	Contains routines for setting and obtaining the current video
** context in one fell swoop.  These routines obtain everything that
** Borland functions allow you to obtain concerning the graphics
** screen.
**
**	All of these routines work with whatever the current
** video adapter is.
**
*************************************************************************/

#define GRAFXLOG_c

#include <alloc.h>
#include <conio.h>
#include <dos.h>
#include <graphics.h>
#include <stdio.h>
#include <string.h>

#include "bios_vid.h"
#include "grafxlog.h"

#define KBHIT	while (!kbhit() ) ; (void) getch()



/************************************************************************
** setGraphicsModes()
**
**	Sets the current graphics video mode values by loading from the
** Screen structure.  The following members of Screen object *s must
** be loaded first:
**
**	screen_page
**	screen_mode
**	cursor_row
**	cursor_column
**	cursor_start_line
**	cursor_end_line
**
*************************************************************************/

int
setGraphicsModes
(
    Screen *s
)
{
    union REGS regs;
    int type;
    int err = 0;

    /********************************************************************
    ** Set the page number, if in the proper range.
    *********************************************************************/

    if (s->screen_page >= 0 && s->screen_page < 8)
    {
	regs.h.ah = VIDEO_PAGE_SELECT;
	regs.h.al = s->screen_page;
	int86(VIDEO_BIOS, &regs, &regs);	/* set video page no.	*/
    }
    else
	err = 1;

    /********************************************************************
    ** Now set the mode of the page
    *********************************************************************/

    if (s->screen_mode > 80)		/* supports extended modes	*/
    {
	s->screen_mode += VID_TEXT_43_50; /* IS THIS RIGHT? NO...	*/
    }
    regs.h.ah = VIDEO_MODE;
    regs.h.al = s->screen_mode;
    int86(VIDEO_BIOS, &regs, &regs);	/* set cursor position		*/

    /********************************************************************
    ** Set the cursor type.
    *********************************************************************/

    type = s->cursor_start_line;	/* CH = starting (top) line	*/
    type <<= 8;				/* CL = ending (bottom) line	*/
    type += s->cursor_end_line;
    regs.h.ah = VIDEO_CURSOR_SIZE;
    regs.x.cx = type;			/* position of cursor bounds	*/
    int86(VIDEO_BIOS, &regs, &regs);	/* set cursor position		*/

    /********************************************************************
    ** Set the cursor position.  Use desired video page.
    *********************************************************************/

    regs.h.ah = VIDEO_CURSOR_POSITION;
    regs.h.bh = s->screen_page;
    regs.h.dh = s->cursor_row;
    regs.h.dl = s->cursor_column;
    int86(VIDEO_BIOS, &regs, &regs);	/* set cursor position		*/

    setcolor(s->screen_attribute);
    setbkcolor(s->screen_backattribute);

    /********************************************************************
    ** We cannot set the rest, so we dummy them for now.
    **
    ** dummy = s->screen_rows;
    **
    ********************************************************************/

    return err;
}


/************************************************************************
** getGraphicsModes()
**
**	Returns the current graphics video mode values by loading up the
** Screen structure.
**
**	The return value is the type of video screen (which is also
** returned as part of the structure).  The rest are returned as
** side-effects, in the Screen structure.
**
**	If the Screen structure pointer is NULL, then only the
** screen driver code is returned (for easy, breakable, screen-checking).
** See video.c for typical values returned for some video setups.
** Also see Borland's graphics.h file.
**
** IMPORTANT:
**
**	The screen_gmax_x and screen_gmax_y parameters will be 0 unless
** graphics-mode is first entered [e.g. by graphicsScreen()] before
** calling getGraphicsModes().
**
** WARNING:
**
**	1.  Still doesn't obtain s->screen_atribute
**	2.  Doesn't check s->adapter... just uses the current one
**
*************************************************************************/

GraphicsDriver
getGraphicsModes
(
    Screen *s
)
{
    union REGS regs;
    int graphdrvr = (int) DETECT;	/* start with auto-detection	*/
    int graphmode = (int) VGALO;	/* start with a "zero" value	*/
    int *gd = &graphdrvr;		/* detectgraph (if 0) or driver	*/
    int *gm = &graphmode;		/* if not zero, graphics mode	*/
    int xasp, yasp;			/* for reading aspect ratio	*/
    int errcode = DETECT;		/* here, zero is an error!	*/

    detectgraph(gd, gm);			/* find basic fact	*/
    if (graphresult() != grOk)			/* no graphics hardware	*/
    {
	return errcode;
    }

    if (s != NULL)
    {
	/****************************************************************
	** Allocate the following two structures, and make sure that
	** the SFT member of *vbs points to *sft.  Be very rigorous
	** about error-checking.
	*****************************************************************/

	Static_Function_Table *sft;
	Video_BIOS_State *vbs;

	sft = (Static_Function_Table *) malloc(sizeof(Static_Function_Table));
	if (sft == NULL)
	    return errcode;

	vbs = (Video_BIOS_State *) malloc(sizeof(Video_BIOS_State));
	if (vbs == NULL)
	{
	    free(sft);
	    return errcode;
	}
	else
	{
	    vbs->SFT = sft;
	}

	regs.h.ah = VIDEO_STATUS;
	int86(VIDEO_BIOS, &regs, &regs);
	s->screen_mode	  = (int) regs.h.al;	/* get the mode		*/
	s->screen_page	  = (int) regs.h.bh;	/* get the video page	*/
	s->screen_columns = (int) regs.h.ah;	/* width in characters	*/

	/****************************************************************
	** regs.h.bh from above is used implicitly as the screen page number.
	*****************************************************************/

	regs.h.ah = VIDEO_READ_CURSOR;
	int86(VIDEO_BIOS, &regs, &regs);	/* get cursor position	*/
	s->cursor_column	= regs.h.dl;
	s->cursor_row		= regs.h.dh;

	/****************************************************************
	** Instead of the following, we use the getvbs() routine below:
	**
	** s->cursor_start_line	= regs.h.cl;
	** s->cursor_end_line	= regs.h.ch;
	**
	*****************************************************************/

	(void) getvbs(vbs);			/* state of video BIOS	*/

	s->cursor_start_line	= vbs->cursor_start_line;
	s->cursor_end_line	= vbs->cursor_end_line;

	if (s->screen_columns != vbs->character_columns)
	{
	    printf("Screen column mismatch!\n");
	    return errcode;
	}
	s->screen_rows		= vbs->number_of_rows;
	s->screen_gmax_x	= getmaxx();	/*vbs->character_columns*/
	s->screen_gmax_y	= getmaxy();	/*vbs->number_of_rows	*/
	s->screen_attribute	= getcolor();
	s->screen_backattribute	= getbkcolor();

	s->screen_rasters	= sft->raster_lines_supported;
	s->screen_charsets	= sft->max_character_sets;
	s->screen_chardefs	= sft->max_chardef_tables;
	s->screen_driver	= (GraphicsDriver) graphdrvr;
	s->screen_type		= (GraphicsMode) graphmode;   /* vbs->mode*/

	getaspectratio(&xasp, &yasp);	/* read the hardware aspect	*/
	s->screen_aspect	= (double) xasp / (double) yasp;

	s->screen_colors	= getmaxcolor()+1; /* max # of colors	*/

	getpalette(&s->palette);	/* read the palette from board	*/

	free(vbs);
	free(sft);

    }
    return graphdrvr;
}


/************************************************************************
** getScreenModes()
**
**	Returns a subset of the variables returned by getGraphicsModes().
**
**	Frankly, the only reason for this routine is that, in text mode
** on this Dell 433/M machine, the call to Borland's getcolor()
** causes and immediate exit, returning an error code of 255.  This
** doesn't happen when a program first changes to a graphics mode.
**
**	Be sure to heed the relevant warnings in getGraphicsModes().
**
**	Because this is really meant for text-mode usage (although we do
** not enforce that restriction), all the graphics parameters are
** zero'd, unless they "apply" to text-mode.
**
**	Of course, we don't bother with color calls here.
**
*************************************************************************/

GraphicsDriver
getScreenModes
(
    Screen *s
)
{
    union REGS regs;
    int graphdrvr = (int) DETECT;	/* start with auto-detection	*/
    int graphmode = (int) VGALO;	/* start with a "zero" value	*/
    int *gd = &graphdrvr;		/* detectgraph (if 0) or driver	*/
    int *gm = &graphmode;		/* if not zero, graphics mode	*/
    int errcode = DETECT;		/* here, zero is an error!	*/

    detectgraph(gd, gm);			/* find basic fact	*/
    if (graphresult() != grOk)			/* no graphics hardware	*/
    {
	return errcode;
    }

    if (s != NULL)
    {
	/****************************************************************
	** Allocate the following two structures, and make sure that
	** the SFT member of *vbs points to *sft.  Be very rigorous
	** about error-checking.
	*****************************************************************/

	Static_Function_Table *sft;
	Video_BIOS_State *vbs;

	sft = (Static_Function_Table *) malloc(sizeof(Static_Function_Table));
	if (sft == NULL)
	    return errcode;

	vbs = (Video_BIOS_State *) malloc(sizeof(Video_BIOS_State));
	if (vbs == NULL)
	{
	    free(sft);
	    return errcode;
	}
	else
	{
	    vbs->SFT = sft;
	}

	regs.h.ah = VIDEO_STATUS;
	int86(VIDEO_BIOS, &regs, &regs);
	s->screen_mode	  = (int) regs.h.al;	/* get the mode		*/
	s->screen_page	  = (int) regs.h.bh;	/* get the video page	*/
	s->screen_columns = (int) regs.h.ah;	/* width in characters	*/


	/****************************************************************
	** Instead of the following, we use the getvbs() routine below:
	**
	** s->cursor_start_line	= regs.h.cl;
	** s->cursor_end_line	= regs.h.ch;
	**
	*****************************************************************/

	(void) getvbs(vbs);			/* state of video BIOS	*/

	s->cursor_start_line	= vbs->cursor_start_line;
	s->cursor_end_line	= vbs->cursor_end_line;

	if (s->screen_columns != vbs->character_columns)
	{
	    printf("Screen column mismatch!\n");
	    return errcode;
	}
	s->screen_rows		= vbs->number_of_rows;
	s->screen_gmax_x	= getmaxx();	/* pixel width		*/
	s->screen_gmax_y	= getmaxy();	/* pixel height		*/
	s->screen_attribute	= 0;		/* was 'getcolor()'	*/
	s->screen_backattribute	= 0;		/* was 'getbkcolor()'	*/

	s->screen_rasters	= 0;
	s->screen_charsets	= 0;
	s->screen_chardefs	= 0;
	s->screen_driver	= 0;
	s->screen_type		= 0;

	/****************************************************************
	** Not used here...
	**
	** getaspectratio(&xasp, &yasp);
	** s->screen_aspect	= (double) xasp / (double) yasp;
	** s->screen_colors	= getmaxcolor()+1;
	** getpalette(&s->palette);	// read the palette from board	//
	**
	*****************************************************************/

	s->screen_aspect	= 0;
	s->screen_colors	= 0;

	free(vbs);
	free(sft);

    }
    return graphdrvr;
}


/************************************************************************
** getScreenVBS()
**
**	A routine to get the video BIOS state into the appropriate
** elements of the Screen structure.
**
************************************************************************/

void
getScreenVBS
(
    Screen *s
)
{
    if (s != NULL)
    {
	Static_Function_Table *sft;
	Video_BIOS_State *vbs;

	sft = (Static_Function_Table *) malloc(sizeof(Static_Function_Table));
	if (sft == NULL)
	{
	    s->error_code = 1;
	    return;
	}

	vbs = (Video_BIOS_State *) malloc(sizeof(Video_BIOS_State));
	if (vbs == NULL)
	{
	    free(sft);
	    s->error_code = 2;
	    return;
	}
	else
	{
	    vbs->SFT = sft;
	}

	(void) getvbs(vbs);			/* state of video BIOS	*/

	s->cursor_start_line	= vbs->cursor_start_line;
	s->cursor_end_line	= vbs->cursor_end_line;
	s->screen_rows		= vbs->number_of_rows;
	s->screen_rasters	= sft->raster_lines_supported;
	s->screen_charsets	= sft->max_character_sets;
	s->screen_chardefs	= sft->max_chardef_tables;

	free(vbs);
	free(sft);
    }
}


/************************************************************************
** getvbs()
**
**	A routine to get the "video BIOS state".  See bios_vid.h for
** complete details.
**
** Input:
**
**	AH	27
**	BX	implementation type (must be 0)
**	ES:DI	address of 64-Byte buffer
**
** Output:
**
**	AL	= 0x1B
**	ES:DI	buffer updated with function and state information
**
************************************************************************/

int
getvbs
(
    Video_BIOS_State *vbs
)
{
    union REGS regs;
    struct SREGS sregs;

    regs.h.ah	= VIDEO_BIOS_STATE;
    regs.x.bx	= 0;
    sregs.es	= FP_SEG(vbs);
    regs.x.di	= FP_OFF(vbs);
    int86x(VIDEO_BIOS, &regs, &regs, &sregs);

    return (regs.h.al != 0x1B);	/* 0 if normal operation occurred	*/
}


/************************************************************************
** getvbsdump()
**
**	Allocate the following two structures, get them filled,
** and dump (to a file called "GRAFXVBS.TXT") the current state
** of the video adapter.
**
*************************************************************************/

void
getvbsdump (void)
{
    FILE *f;
    Static_Function_Table *sft;
    Video_BIOS_State *vbs;

    sft = (Static_Function_Table *) malloc(sizeof(Static_Function_Table));
    if (sft != NULL)
    {
	vbs = (Video_BIOS_State *) malloc(sizeof(Video_BIOS_State));
	if (vbs == NULL)
	{
	    free(sft);
	}
	else
	{
	    vbs->SFT = sft;
	    f = fopen("GRAFXVBS.TXT", "a");
	    if (f == NULL)
	    {
		free(sft);
		free(vbs);
	    }
	    else
	    {
		(void) getvbs(vbs);	/* state of the video BIOS	*/
		dumpvbs(f, vbs);
		fclose(f);
	    }
	}
	free(vbs);
	free(sft);
    }
}


/************************************************************************
** dumpvbs()
**
**	Does the actual dumping to a file.  Assumes everything's
** been setup ok.
**
*************************************************************************/

#define P	fprintf

void
dumpvbs
(
    FILE *f,
    Video_BIOS_State *vbs
)
{
    Static_Function_Table *sft;

    sft = vbs->SFT;

    P
    (
        f,
        "\n\n===============================================\n\n"
        "Video Bios Table\n"
        "----------------\n\n"
    );

    P(f, "Static_Func_Tbl *SFT	= %p\n", vbs->SFT);
    P(f, "Byte mode		= 0x%x\n", (unsigned)vbs->mode);
    P(f, "Word char_columns	= 0x%x\n", vbs->character_columns);
    P(f, "Word buff_disp_length	= 0x%x\n", vbs->buffer_displayed_length);
    P(f, "Word buff_start		= 0x%x\n", vbs->buffer_start);
    /****P(f, "Cursor_Coordinate cursor[8]= 0x%x\n", vbs->);****/
    P(f, "Byte cursor_end_line	= 0x%x\n", (unsigned)vbs->cursor_end_line);
    P(f, "Byte cursor_start	= 0x%x\n", (unsigned)vbs->cursor_start_line);
    P(f, "Byte active_vid_page	= 0x%x\n", (unsigned)vbs->active_video_page);
    P(f, "Word CRTC_IO_port	= 0x%x\n", vbs->CRTC_IO_port);
    P(f, "Byte crt_mode_set	= 0x%x\n", (unsigned)vbs->crt_mode_set);
    P(f, "Byte crt_palette	= 0x%x\n", (unsigned)vbs->crt_palette);
    P(f, "Byte number_of_rows	= 0x%x\n", (unsigned)vbs->number_of_rows);
    P(f, "Word points_in_character= 0x%x\n", vbs->points_in_character);
    P(f, "Byte active_display	= 0x%x\n", (unsigned)vbs->active_display);
    P(f, "Byte inactive_display	= 0x%x\n", (unsigned)vbs->inactive_display);
    P(f, "Word colors_in_display	= 0x%x\n", vbs->colors_in_display);
    P(f, "Byte pages_supported	= 0x%x\n", (unsigned)vbs->pages_supported);
    P(f, "Byte raster_lines	= 0x%x\n", (unsigned)vbs->raster_lines);
    P(f, "Byte alpha_table_0	= 0x%x\n", (unsigned)vbs->alpha_table_0);
    P(f, "Byte alpha_table_1	= 0x%x\n", (unsigned)vbs->alpha_table_1);
    P(f, "Byte misc_state		= 0x%x\n", (unsigned)vbs->misc_state);
    /****P(f, "Byte reserved[3]	= 0x%x\n", (unsigned)vbs->);****/
    P(f, "Byte video_RAM_size	= 0x%x\n", (unsigned)vbs->video_RAM_size);
    P(f, "Byte save_area_status	= 0x%x\n", (unsigned)vbs->save_area_status);

    P
    (
        f,
        "\nStatic Function Table\n"
        "---------------------\n\n"
    );
    P
    (
	f,
	"Byte vmodes_support[3]	= 0x%x %x %x\n",
	(unsigned) sft->vmodes_supported[0],
	(unsigned) sft->vmodes_supported[1],
	(unsigned) sft->vmodes_supported[2]
    );
    P
    (
	f,
	"Byte sft_reserved[4]	= 0x%x %x %x %x\n",
	(unsigned) sft->sft_reserved[0],
	(unsigned) sft->sft_reserved[1],
	(unsigned) sft->sft_reserved[2],
	(unsigned) sft->sft_reserved[3]
    );
    P(f, "Byte raster_lines_sup	= 0x%x\n",(unsigned)sft->raster_lines_supported);
    P(f, "Byte max_char_sets	= 0x%x\n", (unsigned)sft->max_character_sets);
    P(f, "Byte max_chardef_tbls	= 0x%x\n", (unsigned)sft->max_chardef_tables);
    P(f, "Byte misc_abilities	= 0x%x\n", (unsigned)sft->misc_abilities);
    P(f, "Byte misc_BIOS_abil	= 0x%x\n", (unsigned)sft->misc_BIOS_abilities);
    P
    (
	f,
	"Byte sft_reverved_2[2]	= 0x%x %x\n",
	(unsigned)sft->sft_reverved_2[0],
	(unsigned)sft->sft_reverved_2[1]
    );
    P(f, "Byte save_area_abil	= 0x%x\n", (unsigned)sft->save_area_abilities);
    P(f, "Byte sft_reserved_3	= 0x%x\n", (unsigned)sft->sft_reserved_3);
}


/************************************************************************
** showScreen
**
**	A procedure to display some video parameters; mainly a tool
** for debugging.
**
**	Note that we must bring graphics up in order to get the
** non-zero values of the screen dimensions in units of pixels.
**
*************************************************************************/

void
showScreen
(
    Screen *s
)
{
    if (s != NULL)
    {
	printf
	(
	    "\tVideo driver type\t: %3d (%s)\n"
	    "\tVideo driver mode\t: %3d\n"
	    "\tRaster lines supported\t: %3d\n"
	    "\tMaximum character sets\t: %3d\n"
	    "\tMaximum chardef tables\t: %3d\n",

	    s->screen_driver, s->screen_driver_name,
	    s->screen_type,
	    s->screen_rasters,
	    s->screen_charsets,
	    s->screen_chardefs
	);

	printf
	(
	    "\tVideo mode\t\t: %3d\n"
	    "\tVideo page\t\t: %3d\n"
	    "\tVideo columns\t\t: %3d\n"
	    "\tVideo rows\t\t: %3d\n"
	    "\tHorizontal pixels\t: %3d\n"
	    "\tVertical pixels\t\t: %3d\n",

	    s->screen_mode,
	    s->screen_page,
	    s->screen_columns,
	    s->screen_rows,
	    s->screen_gmax_x,
	    s->screen_gmax_y
	);

	printf
	(
	    "\tForeground color\t: %3d\n"
	    "\tBackground color\t: %3d\n"
	    "\tCursor start line\t: %3d\n"
	    "\tCursor end line\t\t: %3d\n",

	    s->screen_attribute,
	    s->screen_backattribute,
	    s->cursor_start_line,
	    s->cursor_end_line
	); 

	puts("Hit any key to exit...");
	KBHIT;
    }
    else
    {
	printf("Need a non-null Screen pointer.\n");
    }
}

/************************************************************************
** Typical results for various systems:
**
** Easy way:			   EGA	VGA  VGA
**				        (1)  (2)
**
**	Video mode		=    3    3    3
**	Video page		=    0    0    0
**	Video width		=   80   80   80
**
** Hard way:
**
**	Video driver type	:    3    9    9
**	Video driver mode	:    1    2    2
**	Raster lines supported	:    0    0    0
**	Maximum character sets	:    0    0    0
**	Maximum chardef tables	:    0    0    0

**	Video mode		:    3    3    3
**	Video page		:    0    0    0
**	Video columns		:    0   80   80
**	Video rows		:    0   25   50
**	Horizontal pixels	:  639  639  639
**	Vertical pixels		:  349  479  479
**	Cursor start line	:    0   13    6
**	Cursor end line		:    0   14    7
**
** Notes:
**
**	(1) 25-line mode;
**	(2) 50-line mode;
**
*************************************************************************/


/************************************************************************
** getScreenContext()
**
**	Similar in idea to getGraphicsModes(), but obtains the
** information solely via Borland BGI facilities.
**
**	The purpose of this routine is to store all necessary information
** about the current adapter before switching to another one.
**
*************************************************************************/

int
getScreenContext
(
    Screen *s
)
{
    int graphdrvr = (int) DETECT;	/* start with auto-detection	*/
    int graphmode = (int) VGALO;	/* start with a "zero" value	*/
    int *gd = &graphdrvr;		/* detectgraph (if 0) or driver	*/
    int *gm = &graphmode;		/* if not zero, graphics mode	*/
    int xasp, yasp;			/* for reading aspect ratio	*/
    int errcode = DETECT;		/* here, zero is an error!	*/

    detectgraph(gd, gm);		/* check graphics hardware	*/
    errcode = graphresult();		/* check graphics hardware code	*/

    if (s != NULL)
    {
	if (s->adapter != ADAPTER_INACTIVE)
	{
	    s->screen_type		= (GraphicsMode) getgraphmode();
	    s->screen_driver		= (GraphicsDriver) graphdrvr;
	    s->screen_mode		= getgrafxmode();
	    s->screen_page		= getgrafxpage();
	    s->screen_attribute		= getcolor();
	    s->screen_backattribute	= getbkcolor();
	    s->screen_columns		= getgrafxcolumns();
	    s->screen_gmax_x		= getmaxx();
	    s->screen_gmax_y		= getmaxy();
	    s->cursor_column		= getx();
	    s->cursor_row		= gety();
	    s->screen_colors		= getmaxcolor() + 1;

	    getaspectratio(&xasp, &yasp);
	    s->screen_aspect		= (double) xasp / (double) yasp;

	    getpalette(&s->palette);

	    s->error_code		= errcode;
	    s->screen_driver_name	= getdrivername();

	    getScreenVBS(s);			/* get all the rest	*/
	}
    }
    return s->error_code;
}


/************************************************************************
** setScreenContext()
**
**	Similar in idea to setGraphicsModes(), but sets the
** information solely via Borland BGI facilities.
**
**	The purpose of this routine is to recover all information
** about the new adapter after switching from another one.
**
*************************************************************************/

void
setScreenContext
(
    Screen *s
)
{
    int graphdrvr = (int) DETECT;	/* start with auto-detection	*/
    int graphmode = (int) VGALO;	/* start with a "zero" value	*/
    int *gd = &graphdrvr;		/* detectgraph (if 0) or driver	*/
    int *gm = &graphmode;		/* if not zero, graphics mode	*/
    int errcode = DETECT;		/* here, zero is an error!	*/

    detectgraph(gd, gm);		/* check graphics hardware	*/
    errcode = graphresult();		/* check graphics hardware code	*/

    if (s != NULL)
    {
	if (s->adapter != ADAPTER_INACTIVE)
	{
	    setgraphmode((int) s->screen_type);
	    s->screen_driver		= (GraphicsDriver) graphdrvr;
	    s->screen_mode		= getgrafxmode();
	    s->screen_page		= getgrafxpage();
	    setcolor(s->screen_attribute);
	    setbkcolor(s->screen_backattribute);
	    s->screen_columns		= getgrafxcolumns();
	    s->screen_gmax_x		= getmaxx();
	    s->screen_gmax_y		= getmaxy();
	    s->cursor_column		= getgrafxcursorcolumn(s->screen_page);
	    s->cursor_row		= getgrafxcursorrow(s->screen_page);
	    s->screen_colors		= getmaxcolor() + 1;

	    getpalette(&s->palette);

	    s->error_code		= errcode;
	    s->screen_driver_name	= getdrivername();

	    getScreenVBS(s);			/* get all the rest	*/
	}
    }
}


/************************************************************************
** getgrafxmode()
**
**	Not sure if this is different from Borland's getgraphmode(),
** but it get the mode directly from the BIOS.
**
** getgrafxpage()
**
**	Gets the page number directly from the hardware.
**
** getgrafxcolumns()
**
**	Gets the number of columns directly from the hardware.
**
*************************************************************************/

int
getgrafxmode (void)
{
    union REGS regs;

    regs.h.ah = VIDEO_STATUS;
    int86(VIDEO_BIOS, &regs, &regs);
    return (int) regs.h.al;
}

int
getgrafxpage (void)
{
    union REGS regs;

    regs.h.ah = VIDEO_STATUS;
    int86(VIDEO_BIOS, &regs, &regs);
    return (int) regs.h.bh;
}

int
getgrafxcolumns (void)
{
    union REGS regs;

    regs.h.ah = VIDEO_STATUS;
    int86(VIDEO_BIOS, &regs, &regs);
    return (int) regs.h.ah;
}

int
getgrafxcursorrow			/* similar to BGI's gety()	*/
(
    int page
)
{
    union REGS regs;

    regs.h.ah = VIDEO_READ_CURSOR;
    regs.h.bh = page;			/* get the video page	*/
    int86(VIDEO_BIOS, &regs, &regs);	/* get cursor position	*/
    return (int) regs.h.dh;
}

int
getgrafxcursorcolumn			/* similar to BGI's getx()	*/
(
    int page
)
{
    union REGS regs;

    regs.h.ah = VIDEO_READ_CURSOR;
    regs.h.bh = page;			/* get the video page	*/
    int86(VIDEO_BIOS, &regs, &regs);	/* get cursor position	*/
    return (int) regs.h.dl;
}



static char bgiPath[] = "C:\BC\BGI";	/* we'll use env variables later*/

int
initGraphics
(
    Screen *s				/* all screen parameters	*/
)
{
    int errcode = 0;			/* return code for errors	*/
    char *bgi_path = &bgiPath[0];	/* path to find BGI support	*/
    int *gm, *gd;			/* for initgraph()		*/
    int gdriver = DETECT;
    int gmode;

    gd = (int *) &gdriver;
    gm = (int *) &gmode;
    initgraph(gd, gm, bgi_path);
    s->error_code = graphresult();		/* result of initialize	*/
    if (s->error_code != grOk)		/* init error occurred	*/
    {
	printf
	(
	    "%%Graphics error: %s, current adapter\n",
	    grapherrormsg(s->error_code) 
	);
	errcode = 1;
    }
    return errcode;
}

void
closeGraphics
(
    Screen *s				/* all screen parameters	*/
)
{
    closegraph();
    s->adapter = ADAPTER_INACTIVE;	/* ???	*/
}
