/************************************************************************
**
** @(#)announce.cpp		05/26/93	Chris Ahlstrom
**
**	C++ version.
**
**	Implements a general and extensible set of error and informational
** messages.
**
*************************************************************************/

#define ANNOUNCE_cpp

#include <conio.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>

#include "tinput.h"	// "Variations on TInputLine in C++", Op. 3.10
#include "announce.h"	// generic error-handling on a small budget
#include "boxtools.h"	// helpful support tools for simple monologues


/************************************************************************
** UserMessages constructors
**
**	(Eventually) reads a file to obtain a list of messages; these
** messages are pointed to by errorMessages.
**
**	Actually, we have three versions of the constructor available
** (eventually):
**
**	1.  Creates a class with no messages available.  All callers
**	    of this class must provide their own ad hoc messages and
**	    use only the string version of the error handler.
**
**	2.  Creates a class assuming a list of messages has been created
**	    at compile-time as an array of strings.  The callers of this
**	    class can then use either version of errorHandler().
**
**	3.  Same as 2, but the messages are read in from a text-file.
**	    This version is not ready yet.
**
*************************************************************************/

UserMessages::UserMessages		// construct messages on the fly
(
    TDeskTop *desk			// the desktop for display
) :
    deskTop		(desk),
    Error		(ERR_NONE),
    userMessages	(NULL),
    userMessagesLength	(0),
    usedErrorFile	(0)
{
}


UserMessages::UserMessages		// construct messages from memory
(
    TDeskTop *desk,			// the desktop for display
    char **msg_buffer,			// pointer to array of messages
    unsigned length			// the number of pointers
) :
    deskTop		(desk),
    Error		(ERR_NONE),
    userMessages	(msg_buffer),
    userMessagesLength	(length),
    usedErrorFile	(0)
{
    if (msg_buffer == NULL)
    {
	Error = ERR_NULL_POINTER;
	deskTop = NULL;
    }
}


UserMessages::UserMessages		// construct messages from a file
(					// NOT READY YET
    TDeskTop *desk,
    char *msg_file_name
) :
    deskTop		(desk),
    Error		(ERR_NONE),
    usedErrorFile	(0)		// will set this to 1 if successful
{
    char **msg_buffer = NULL;
    unsigned length = 0;

    /*
    ** We will eventually implement messages in a data file.
    ** For now, though, we'll use only memory
    */

    userMessages		= msg_buffer;
    userMessagesLength		= length;

    if (msg_buffer == NULL)
    {
	Error = ERR_NULL_POINTER;
	deskTop = NULL;
    }
}


/************************************************************************
** UserMessages::errorHandler (2 versions)
**
**	This function interfaces between any kind of code, and
** Turbo Vision.  If Turbo Vision is available, its resources are used
** for showing error messages, otherwise straight text output is
** used.
**
** ERR_SHOW_NO_MESSAGE	Behavior needs no flagging to user
** ERR_NONE		Not an error message
** ERR_BAD_MESSAGE	Not to be used as an error code
**
**	Rather than provide one version, which uses a bogus parameter
** value (NULL) to indicate that a code is available, we provide
** two overloaded versions.  One accepts a message-code as a parameter
** (so the caller doesn't need to know the message contents), and the
** other uses a message pointers (so the caller provides the text
** of the message... a little dangerous or inflexible).
**
*************************************************************************/

static char *DEFAULT_BAD_MESSAGE =
    "%%Message support erroneous for this part of the program!";

void
UserMessages::errorHandler		// a message is provided
(
    char *message
)
{
    if (message == NULL)
	message = DEFAULT_BAD_MESSAGE;
    Error = ERR_USER;

    errorBox(message, 0U);
}


void
UserMessages::errorHandler		// a message code is provided
(
    int message
)
{
    char *messstr;

    ErrorCode messagecode = (ErrorCode) message;
    if (messagecode != ERR_SHOW_NO_MESSAGE)
    {
	if (messagecode < 0 || messagecode > userMessagesLength)
	    messstr = DEFAULT_BAD_MESSAGE;
	else
	    messstr = userMessages[messagecode-1];
	errorBox(messstr, 0U);
    }
    Error = messagecode;			// set this "global"
}


/************************************************************************
** UserMessages::errorBox
**
**	Displays a simple dialog box proclaiming an error condition.
** A single value may be optionally provided, in which case we
** hope the value and the message string (in the guise of a format
** string) match up.  If 0, the value is not used.
**
**	Called by errorHandler(), but can be called on its own, too.
**
*************************************************************************/

void
UserMessages::errorBox
(
    char *message,
    unsigned value
)
{
    if (message == NULL)
    {
	message = "%Programming error";
    }
    else
    {
	if (strlen(message) > ERR_STRING_SIZE-8)	// for safety
	{
	    message = "%Programming error";
	}
	else
	{
	    if (value != 0U)
	    {
		sprintf(errorString, message, value);
		message = errorString;
	    }
	}
    }
    messageBox(message, mfError | mfOKButton);
}


/************************************************************************
** UserMessages::confirmation
**
**	Simple way to get yes or no status.
**
*************************************************************************/

YesNoType
UserMessages::confirmation
(
    char *string			// short message inside the box
)
{
    ushort action;
    YesNoType code;

    code = NO;

    action = messageBox(string, mfWarning | mfOKCancel);
    if (action == cmOK)
	code = YES;
    return code;
}


/************************************************************************
** UserMessages::lastError()
**
**	Provides access to the private error code for the latest error.
**
*************************************************************************/

ErrorCode
UserMessages::lastError ()
{
    return Error;
}


/************************************************************************
** UserMessages::prompt
**
**	Simple way to get yes or no status.
**
*************************************************************************/

int
UserMessages::prompt
(
    char *string			// short message inside the box
)
{
    ushort action;
    int code;

    code = NO;

    action = messageBox(string, mfInformation|mfOKCancel);
    if (action == cmOK)
	code = YES;
    return code;
}


/************************************************************************
** UserMessages::msgHandler()
**
**	More robust and flexible way to present messages.
**
*************************************************************************/

void
UserMessages::msgHandler
(
    char *string
)
{
    if (string == NULL)
	string = "\n%%Programmer: null string passed to msgHandler\n";

    (void) messageBox(string, mfInformation|mfOKButton);
}


/************************************************************************
** UserMessages::bannerHandler
**
**	Handles banner strings by first determining the width (in
** characters) and height (in lines) of the banner, then generating
** the appropriate box.
**
**	The user must click on the OK button to continue.  There
** is only a return value of non-ERR_NONE if an internal problem
** occurred.
**
*************************************************************************/

ErrorCode
UserMessages::bannerHandler
(
    char *message
)
{
    if (message == NULL)
    {
	message = "%%Programming error";
    }

    ErrorCode error = ERR_NONE;
    TPoint banner;			// calculated size of text
    TRect box;				// calculated size of dialog

    banner	= banner_size(message);	// dimensions of the banner
    box		= banner_spec(banner);	// box banner fits into

    TDialog *pd = new TDialog(box, "Information");

    if (pd)				// did it work above?
    {
	TRect text;			// location and extent of text

	text.a.x = BOX_MARGIN;		// move to text boundary
	text.a.y = BOX_TOP;		// move to text line
	text.b.x = banner.x;		// width of the text
	text.b.y = banner.y;		// heigth of the text
	text.b  += text.a;		// lower right corner of text

	TView *b = new TStaticText(text, message);
	if (b)
	{
	    pd->insert(b);		// display the text

	    TRect button;

	    button = banner_OK(box);	// location of button in box
	    b = new TButton(button, "~O~K", cmOK, bfDefault);
	    if (b)
		pd->insert(b);		// display the OK button

	    ushort control;
	    control = deskTop->execView(pd);	// display the whole box

	    /*
	    ** Since there is no Cancel button, the following code
	    ** serves no purpose.  Oh, well!
	    */

	    if (control != cmCancel)
		error = ERR_NONE;
	    else
		error = ERR_USER_ABORT;
	}
	else
	{
	    TView *b = new TStaticText(TRect(4,4,16,5), "DIALOG ERROR");
	    if (b)
		pd->insert(b);
	    deskTop->execView(pd);
	    error = ERR_DIALOG;
	}
	deskTop->destroy(pd);		// remove the box now
    }
    return error;
}

