#ifndef EXPARRAY
#define EXPARRAY
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <dir.h>
#include <iostream.h>

/******************************************************************
	General Expandable Disk Array Class.
   Bill Honey, 1993

	Holds a certain ammount of the array in memory, whilst the rest
	is held on disc. If an element that is held on disc is requested
	the chunk in memory is swopped with the chunk on disc that holds
	that element.
   If an element beyond the end of the array is requested, the array
   expands itself to make that element valid and accessable.

   Implemented using templates.
   Holds the actual data elements (not pointers to them).

   comment out "cout" statments when using for real

******************************************************************/


template<class T> class DskArray
{
  	FILE * fp;     				// file handle for disc array.
	char pathname[128];			// file name of same.
	T * data;     					// the part of the array held in memory.
	int size; 	  					// the number of elements in the total (disc) array.
   int memsize;  					// the number of elements held in memory.
   int memstart;              // the index of the first element held in memory.
	int increaseSize;  			// how much to increase the array size by
										// if we run out of room.
	int counter;  					// a general purpose counter.
	void increase();				// function that increases the array size if we
										// try to access an element off the end of the array.
   void swop(int indx);       // swops the current block of data to
   									// disc and gets in a block that includes indx.
	int  npts;						// A variable keep internally for storing
										// the npts in use (for ease of usage only)
public:
   DskArray(int sz=10, int msz = 10, int incr=10); // constructor
   ~DskArray();												// destructor
	 T & operator [](int indx);    					   // [] operator
	void SetNpts(int inpts) { npts = inpts;};
   int Npts() { return npts;};
};

/******************************************************************
Constructor.
	1) Creates the disc file and fills it with empty elements.
   2) Creates the memory part of the array.
******************************************************************/
template<class T> DskArray<T>::DskArray(int sz=10, int msz = 10, int incr=10)
{
T * tmp;
	// Create a tmp file with a unique name of type Taaa.aaa.
	strcpy(pathname, "TXXXXXX");
   mktemp(pathname);						// Unique to Borland ?

   											// Open the tmp file.
	if ((fp = fopen(pathname,"w+b")) == NULL) errMsg("DskArray: can't open tmp file");

  	size = sz;								// Set the array size,
  	memsize = msz;  						// and the memory array size,
  	increaseSize = incr;					// and the increase size.
  	memstart = 0;							// set the memory starting index.

   npts = 0;
  	// write a load of empty T's to the file
	if (!(tmp = (T *) new T)) errMsg("DskArray Constructor: Out of memory ");

	fseek(fp,0L,SEEK_SET);
  	for (counter = 0; counter < size; counter++)
		fwrite(tmp, sizeof(T), 1, fp);
  	delete tmp;

  	// create the part that will reside in memory
  	if (!(data = (T *) new T[memsize])) errMsg("DskArray Constructor: Out of memory ");

}

/******************************************************************
Destructor
	1) Closes and delete the disv file
	2) Deletes the memory array
******************************************************************/
template<class T> DskArray<T>::~DskArray()
{
   fclose (fp);
   if (remove(pathname) == -1) errMsg("Can't close temp file");
   delete data;
}


/******************************************************************
Square brackets operator
	1) returns a reference to the data element requested.
   2) If necessary, increases the array size to
			make the requested element legal
	3) If necessary, swops data from disc to put that element in memory.
******************************************************************/
template <class T> T & DskArray<T>::operator [](int indx)
{

	if (indx < 0) errMsg ("DskArray: Indexing Error - index less than zero \n");

	// If the index is beyond the end of the array, increase the
	// (disc) array size.
	while (indx >= size) increase();

   // If index is on the disc, swop chunks.
   if (indx < memstart || indx >= (memstart + memsize))
   	swop(indx);

// return the current data element
return data[indx - memstart];
}
/******************************************************************
Swop function:
	1) Puts the current memory array onto disc.
	2) Retrieves from disc a section of the array that includes
		the currently requested index.

******************************************************************/
template <class T> void DskArray<T>::swop(int indx)
{
    cout << " swopping...\n"; // Comment out when using for real

	// Write the current block to disc.
   fseek(fp,(long) memstart * (long) sizeof(T),SEEK_SET);
	fwrite(data, sizeof(T), memsize, fp);

   // Check that the end of the requested block does not over run the file,
   // and that the block is not larger than the array.
   while ((indx + memsize) > size) indx --;
	if (indx < 0 || indx >= size)  errMsg("DskArray: Indexing Error\n");

   // Read the requested block
   fseek(fp,(long)indx * (long)sizeof(T),SEEK_SET);
   fread(data, sizeof(T), memsize, fp);

   // Set the starting index
   memstart = indx;
}
/******************************************************************
Increase function:
	   Increases the size of the disk array by adding more elements
		to the end of the file
******************************************************************/

void errMsg(char * msg)
{
	perror(msg);
   exit(1);
} 

template <class T> void DskArray<T>::increase()
{
T * tmp;

  	cout << "increasing array size\n";   // Comment when using for real

	if (!(tmp = (T *) new T)) errMsg("DskArray Increase : Out of memory ");

   // Write extra elements to the end of the file
  	fseek(fp,0L,SEEK_END);
  	for (counter = 0; counter < increaseSize; counter++)
	  fwrite(tmp, sizeof(T), 1, fp);

   // set the new array size
  	size += increaseSize;  // new size

  	delete tmp;
}

#endif