/* Application header file for user with 3D renderer */

/* Part of the REND386 package by Dave Stampe and Bernie Roehl */

/* Copyright 1992 by Dave Stampe and Bernie Roehl.
   May be freely used to write software for release into the public domain;
   all commercial endeavours MUST contact Bernie Roehl and Dave Stampe
   for permission to incorporate any part of this software into their
   products!
 */

struct Screeninfo {
	int xmin, ymin, xmax, ymax, xcent, ycent, colors, pages;
	};

#define SCREEN_FACTOR ((65536L*3)/4)  /* screen is physically 4:3 */
extern struct Screeninfo screeninfo;

typedef void OBJECT;
typedef void OBJLIST;

typedef void POLY;

typedef void SEGMENT;

#ifndef MATRIXDEF
typedef long MATRIX[4][3];  	  /* 3x3 rotate plus 3 translate para's */
			/* rotate matrix is <3.29>, translate is <32.0> */
#define MATRIXDEF 1
#endif

/* renderer viewpoint/screen control structure */
/* viewoint in X, Y, Z coords */
/* pan, tilt, roll in (float*65536) formats */
/* zoom is equiv. to magnification from 90 deg. FOV (also float*65536) */
/* aspect sets how much to magnify Y more than X to fix up displays */
/* light source point in world coordinates */
/* left, right, top, bottom set edges of screen */
/* hither sets closest point: keep >16 for best range of world coords */
/* yon sets max. distance: keep it 1<<26 if not used */
/* rest is renderer workspace */

typedef struct {
	/* VIEWPOINT */
	long ex, ey, ez;        /* <25.0>  location of eyepoint         */
	long pan, tilt, roll;   /* <16.16> viewing angles (deg) +/- 128 */
	long zoom;              /* <16.16> 1/tan(H FOV/2) 0.5 to 16     */
	long lx,ly,lz;          /* <25.0>  location of light source     */

	/* SCREEN DATA */
	long left,right;        /* <25.0> clipping planes */
	long top, bottom;
	long hither, yon;       /* <25.0> near and far clipping planes   */
	long aspect;            /* <16.16> x:y fixup factor (magnify Y by...) */

	/* RENDERING CONTROL */
	unsigned flags;         /* 16 bits of flags */

	/* ADVANCED SCREEN DATA */
	long x_offset, y_offset;   /* amount to move screen center in pixels */
	unsigned orientation;      /* used to mirror screen image */
#define NOFLIP 0x0000      /* bits used in orientation field */
#define XFLIP  0x0001
#define YFLIP  0x0002

	MATRIX eye_xform;

	char spare[200];        /* 100 bytes needed, plus 100 bytes spare */
	} VIEW;

/* View flags: */

#define WIREFRAME           0x0001
#define HIDE_HIGHLIGHTED    0x0002
#define HIDE_UNHIGHLIGHTED  0x0004

/* Function prototypes: */

extern OBJECT *new_obj(int type, int nv, int np, char *name);
extern void add_vertex(OBJECT *obj, long x, long y, long z);
extern POLY *add_poly(OBJECT *obj, unsigned color, int npoints);
extern void add_point(OBJECT *obj, POLY *p, int vertnum);
extern void delete_obj(OBJECT *obj);
extern long object_bounds(OBJECT *obj, long *x, long *y, long *z);
extern void compute_obj(OBJECT *obj);
extern void set_obj_flags(OBJECT *obj, unsigned char value);
extern unsigned char get_obj_flags(OBJECT *obj);
extern void highlight_obj(OBJECT *obj);
extern void unhighlight_obj(OBJECT *obj);

/* Bits in whichones, and returned by get_seg_limits */

#define LIM_MINTX 0x0001  /* if set, we have a minimum X limit */
#define LIM_MAXTX 0x0002  /* ... */
#define LIM_MINTY 0x0004
#define LIM_MAXTY 0x0008
#define LIM_MINTZ 0x0010
#define LIM_MAXTZ 0x0020
#define LIM_MINRX 0x0040
#define LIM_MAXRX 0x0080
#define LIM_MINRY 0x0100
#define LIM_MAXRY 0x0200
#define LIM_MINRZ 0x0400
#define LIM_MAXRZ 0x0800

/* Depth-sorting control: */

#define DEEPEST 0x0000		/* sort polys by deepest point */
#define ATBACK  0x0001		/* push this object's poly's waaaay back */
#define AVERAGE 0x0002		/* sort polys by average depth */

#define BYOBJECT   0x0100	/* sort by object */
#define BYPOLY     0x0000	/* put polys in world before sort */
#define USE_CENTER 0x0200	/* use clip sphere center as sort center */

extern int get_object_sorting(OBJECT *obj);
extern void set_object_sorting(OBJECT *obj, int depth_type);

/* Values for object flags */
#define OBJ_INVIS        0x01
#define OBJ_HIGHLIGHTED  0x02

extern OBJLIST *new_objlist();
extern void add_to_objlist(OBJLIST *list, OBJECT *obj);
extern void remove_from_objlist(OBJLIST *list, OBJECT *obj);
extern void del_objlist(OBJLIST *list);
extern OBJECT *first_in_objlist(OBJECT **objlist);
extern OBJECT *next_in_objlist(OBJECT **objlist, OBJECT *obj);
extern OBJECT *prev_in_objlist(OBJECT **objlist, OBJECT *obj);
extern int is_first_in_objlist(OBJECT **objlist, OBJECT *obj);
extern int is_last_in_objlist(OBJECT **objlist, OBJECT *obj);
extern void get_obj_info(OBJECT *obj, int *nv, int *np, char *buffer, int maxn);
extern void get_vertex_info(OBJECT *obj, int vertnum, long *x, long *y, long *z);
extern void get_vertex_world_info(OBJECT *obj, int vertnum, long *x, long *y, long *z);
extern void get_poly_info(OBJECT *obj, int polynum, unsigned *color, int *nverts, int *verts, int maxverts);
extern void set_poly_color(OBJECT *obj, int polynum, unsigned color);
extern void *get_object_owner(OBJECT *obj);
extern void set_object_owner(OBJECT *obj, void *owner);
extern OBJECT *copy_obj(OBJECT *obj, int nverts, int npolys, char *name);
extern void copy_world_to_object(OBJECT *obj);

extern void compute_view_factors(VIEW *v);

extern void fast_view_factors(VIEW *v);
		/* computes eye point/ angle movement factors only */

extern void initialize_screen_factors(VIEW *v);
		/* computes screen and viewport    */
		/* factors.  These stay constant   */
		/* over eye point changes          */
		/* setup viewwport data, do once   */
		/* unless offset, zoom etc changed */

extern void setup_render();
extern void reset_render();
extern void render(OBJLIST *objlist, VIEW *view);
extern int poly_cosine(POLY *poly);

extern long where_pt(OBJECT **objlist, long x, long y, long z, OBJECT **wobj, int *wvert);
extern OBJECT *where_screen_pt(int *pol, int *vert, int x, int y);

extern SEGMENT *new_seg(SEGMENT *parent);
extern void seg_setrep(SEGMENT *s, void *rep);
extern void *seg_getrep(SEGMENT *s);
extern void seg_getposition(SEGMENT *s, long *x, long *y, long *z, long *rx, long *ry, long *rz);
extern char *seg_getname(SEGMENT *s);
extern void seg_setname(SEGMENT *s, char *name);
extern void abs_move_segment(SEGMENT *s, long tx, long ty, long tz);
extern void rel_move_segment(SEGMENT *s, long tx, long ty, long tz);
extern void abs_rot_segment(SEGMENT *s, long rx, long ry, long rz);
extern void rel_rot_segment(SEGMENT *s, long rx, long ry, long rz);
extern void update_segment(SEGMENT *s);
extern void full_update_segment(SEGMENT *s);
extern SEGMENT *find_root_segment(SEGMENT *s);
extern SEGMENT *parent_segment(SEGMENT *s);
extern SEGMENT *child_segment(SEGMENT *s);
extern SEGMENT *sibling_segment(SEGMENT *s);
extern SEGMENT *copy_segment(SEGMENT *s, long dx, long dy, long dz, void *(*copyrep_fn)());
extern void delete_segment(SEGMENT *s, void (*delrep_fn)());
extern void attach_segment(SEGMENT *s, SEGMENT *to);
extern void detach_segment(SEGMENT *s);
extern void seg_set_load_info(SEGMENT *s, char *filename, float sx, float sy, float sz, long tx, long ty, long tz);
extern char *seg_get_load_info(SEGMENT *s, float *sx, float *sy, float *sz, long *tx, long *ty, long *tz);
extern MATRIX *get_seg_matrix(SEGMENT *s);
extern SEGMENT *find_segment_by_name(SEGMENT *s, char *name);
extern unsigned get_seg_limits(SEGMENT *seg, long lims[]);
extern unsigned set_seg_limits(SEGMENT *seg, long lims[], unsigned whichones);

extern enter_graphics();
extern exit_graphics();
extern void clear_display(int page);
extern int set_drawpage(int page);
extern int set_vidpage(int page, int wait);
extern void vgabox(int left, int top, int right, int bottom, int color);
extern void vgaline(int left, int top, int right, int bottom, int color);

extern load_pcx(FILE *in, int page);
extern int save_pcx(FILE *out, int page);

#define MONOSCOPIC 0
#define LEFT_EYE   1
#define RIGHT_EYE  2


typedef struct {   /* stereoscopic view factors */
	long phys_screen_dist;  /* eye-to-screen distance (mm;) keep < 32000  */
	long phys_screen_width; /* viewport width on screen (mm)              */
	long pixel_width;       /* width of viewport in pixels                */
	long phys_eye_spacing;  /* spacing of eyes (mm) [usually 63]          */
	long phys_convergence;  /* convergence dist. (usually screen) (mm)    */
	long world_scaling;     /* world units per physical mm                */
	} STEREO ;


/* call fast_view_factors or view_from_matrix BEFORE these! */

	/* eye view from cyclopean view */
void make_stereo_view(VIEW *root, VIEW *nview, STEREO *stereo, int eye);

	/* sufficient if only view point has changed */
void update_stereo_view(VIEW *v, STEREO *s, int eye);

	/* split screen with mirror only */
void mirr_stereo_view(VIEW *root, VIEW *nview, STEREO *stereo, int eye);

/* Matrix routines */

extern long isine(long angle);
extern long icosine(long angle);
					/* create rotation/translation */
					/* "matrix" from angle data    */
					/* CALL setup_render FIRST!    */
void make_matrix(MATRIX m,
		long rx, long ry, long rz,
		long tx, long ty, long tz);

void matrix_mult(MATRIX a, MATRIX b, MATRIX c);
					/* 3x3 section of matrixs: A*B->C */

					/* rotate/translate XYZ by matrix */
void matrix_point(MATRIX m, long *xp, long *yp, long *zp);

			/* generate inverse of rotate matrix (transpose) */
			/* ONLY WORKS FOR ORTHOGONAL MATRICES  */
void matrix_transpose(MATRIX a, MATRIX b);

				/* generate inverse rot/trans. matrix */
				/* includes translation */
void inverse_matrix(MATRIX a, MATRIX b);

				/* use "matrix" to rotate then translate */
				/* an object from object space to world  */
void apply_matrix(OBJECT *obj, MATRIX m);

void identity(MATRIX m);	/* create identity matrix */

void matrix_copy(MATRIX s, MATRIX d);    /* copy matrix */

void matrix_rot_copy(MATRIX s, MATRIX d); /* copy, zero translation */

long m_mult(long a, long b); /* perform mult. by <3.29> matrix element */

/* End of rend386.h */
