/*===========================================================================*/
/*                                                                           */
/* File    : DECTRL.C                                                        */
/*                                                                           */
/* Purpose : Simple dialog editor for the Magma Window System                */
/*                                                                           */
/* History :                                                                 */
/*                                                                           */
/* (C) Copyright 1989 Marc Adler/Magma Systems     All Rights Reserved       */
/*===========================================================================*/

#include <stdio.h>
#include <ctype.h>
#include "window.h"
#include "de.h"
#include "symbol.h"


typedef struct idtoclassname
{
  int  idCheckbox;
  char *szClassName;
} IDTOCLASSNAME;
IDTOCLASSNAME IDtoClassname[] =
{
  ID_CTRL_TEXT,       "Text",
  ID_CTRL_EDIT,       "Edit",
  ID_CTRL_LISTBOX,    "Listbox",
  ID_CTRL_CHECKBOX,   "Checkbox",
  ID_CTRL_RADIOBUTTON,"RadioButton",
  ID_CTRL_PUSHBUTTON, "PushButton",
  ID_CTRL_FRAME,      "Frame",
  ID_CTRL_BOX,        "Box",
  ID_CTRL_COMBO,      "ComboBox",
  0,                  NULL
};


/*
  AddCtrl - adds the specified control to the dialog that we're building
*/
AddCtrl(hDlg)
  HWND hDlg;
{
  char text[133];
  WORD id;
  HWND hCtrl = NULLHWND;
  CONTROL *ci;
  DWORD iStyle = 0x00L;
  IDTOCLASSNAME *pCls;
  SYMBOL *sym;
  
  /*
    Get the ID and the control's text which the user specified
  */
  GetDialogText(hDlg, ID_CTRL_ID, (char far *) text, sizeof(text));
  if (isdigit(text[0]))
    id = atoi(text);
  else if ((sym = SymbolFind(text, 0)) != NULL)
    id = sym->idSymbol;
  else
    id = 0;
  GetDialogText(hDlg, ID_CTRL_TITLE, (char far *) text, sizeof(text));
  
  /*
    Find the correct class and create the control window
  */
  for (pCls = IDtoClassname;  pCls->szClassName;  pCls++)
  {
    if (IsDialogButtonChecked(hDlg, pCls->idCheckbox))
    {
      if (pCls->idCheckbox == ID_CTRL_EDIT)
        iStyle = WIN_HAS_BORDER;
      else if (pCls->idCheckbox == ID_CTRL_LISTBOX)
        iStyle = LBS_STANDARD;
      iStyle |= WS_CLIP | WS_MOVEBOX | WS_SIZEBOX;

      hCtrl = CreateWindow(pCls->szClassName, text, iStyle,
                           CurrRect.left, CurrRect.top,
                           RECT_WIDTH(CurrRect), RECT_HEIGHT(CurrRect),
                           0x70, id, 
                           hDEDlg,
                           NULLHWND, 0, (DWORD) 0);
      break;
    }
  } 

  /*
    If we added a control, then we subclass that control by setting its
    window proc to CtrlWinProc. We also add the control's parameters
    to the list of currently defined controls.
  */
  if (hCtrl)
    CtrlAddToList(hCtrl, 0L);
}


/*****************************************************************************/
/*                                                                           */
/* CtrlWinProc - this is the "front-end" window proc for all of our          */
/* defined controls. This is an example of subclassing a control.            */
/*                                                                           */
/*****************************************************************************/
long pascal CtrlWinProc(hCtrl, message, wParam, lParam)
  HWND hCtrl;
  WORD message;
  WORD wParam;
  DWORD lParam;
{
  static bSelecting = FALSE;

  int mouserow = HIWORD(lParam);
  int mousecol = LOWORD(lParam);
  RECT ctrlRect;


  CtrlDisplayStatus();

  switch (message)
  {
    case WM_NCHITTEST   :  
      /*
        Let all controls, including statics, get the mouse for dragging.
      */
      return HTCLIENT;

    case WM_LBUTTONDOWN :
      /* 
        If we hold the mouse down on the top-left corner, then we want
        to move the control 'hCtrl'. If we hold the mouse down on the
        bottom-right corner, then we want to resize the control.
      */
      GetWindowRect(hCtrl, &ctrlRect);
      if (mouserow == ctrlRect.bottom && mousecol == ctrlRect.right)
      {
        hCtrlSize = hCtrl;
        WinRubberband(hCtrl, WM_SIZE, mouserow, mousecol);
        bDlgIsDirty++;
      }
      else if (mouserow == ctrlRect.top && mousecol == ctrlRect.left)
      {
        hCtrlMove = hCtrl;
        WinRubberband(hCtrl, WM_MOVE, mouserow, mousecol);
        bDlgIsDirty++;
      }
      else
      {
        /* 
          We clicked in the middle of the control
        */
      }
      /*
        Make all keyboard and mouse messages go to that control
      */
#if 0
      SetCapture(hCtrl);
      SetFocus(hCtrl);
#endif
      hCtrlMove = NULLHWND;
      hCtrlSize = NULLHWND;
      return TRUE;

#if 0
    case WM_LBUTTONUP :
      hCtrlMove = NULLHWND;
      hCtrlSize = NULLHWND;
      /*
        Allow other windows to get mouse messages
      */
      ReleaseCapture();
      return TRUE;
#endif

    case WM_RBUTTONDOWN :
      CtrlEditInfo(hCtrl);
      bDlgIsDirty++;
      return TRUE;

    case WM_MOUSEMOVE :
      if (hCtrlMove)
      {
        WinMove(hCtrlMove, mouserow, mousecol);
        ShowWindow(hDEDlg, TRUE);
        bDlgIsDirty++;
      }
      else if (hCtrlSize)
      {
        GetWindowRect(hCtrlSize, &ctrlRect);
        WinSetSize(hCtrlSize,mouserow-ctrlRect.top+1,mousecol-ctrlRect.left+1);
        ShowWindow(hDEDlg, TRUE);
        bDlgIsDirty++;
      }
      return TRUE;


    case WM_SETFOCUS :
      hCurrCtrl = hCtrl;
      break;

    case WM_KILLFOCUS :
      hCurrCtrl = NULLHWND;
      break;

    case WM_CHAR :
#ifdef OLDBUTNEEDED
      if ((hCtrlSize || hCtrlMove) && IsArrowKey(wParam))
      {
        (hCtrlSize) ? DE_DialogSize(hCtrlSize, wParam, 0, 0) 
                    : DE_DialogMove(hCtrlMove, wParam, 0, 0);
        return TRUE;
      }
#endif
      if (wParam == VK_CTRL_D)
      {
        CtrlDelete(hCtrl);
        return TRUE;
      }
      break;
  }

  /* Call the control's default window proc */
  return DefWinProc(hCtrl, message, wParam, lParam);
}


/*****************************************************************************/
/*                                                                           */
/*  CtrlEditInfo()                                                           */
/*    Change the information associated with a control.                      */
/*                                                                           */
/*****************************************************************************/
CtrlEditInfo(hCtrl)
  HWND hCtrl;
{
  char buf[133];
  int  id;
  SYMBOL *sym;

  /*
    Create the ctrl-info dialog box
  */
  BuildCtrlDialog();

  /*
    Fill in the three fields - the title, id, and control window class
  */
  SendMessage(hCtrl, WM_GETTEXT, sizeof(buf), (DWORD) (char far *) buf);
  SetDialogText(hCtrlDlg, ID_CTRL_TITLE, buf);

  id = SendMessage(hCtrl, WM_GETID, 0, 0L);
  if ((sym = SymbolFind(NULL, id)) != NULL)
    SetDialogText(hCtrlDlg, ID_CTRL_ID, sym->szSymbol);
  else
    SetDialogInt(hCtrlDlg, ID_CTRL_ID, id);

  CheckRadioButton(hCtrlDlg, ID_CTRL_BASE,ID_CTRL_END, ClassToCtrlID(hCtrl));

  /*
    Set CurrRect cause AddCtrl() needs it...
  */
  CurrRect = WinGetRect(hCtrl);

  if (DialogBox(hCtrlDlg) == IDOK)
    CtrlDelete(hCtrl);
}


/***************************************************************************/
/*                                                                         */
/*  CtrlDelete() - delete a control from the dialog box                    */
/*                                                                         */
/***************************************************************************/
void CtrlDelete(hCtrl)
  HWND hCtrl;
{
  LIST *p;

  if (!hCtrl)
    return;

  /*
    Remove the control from CtrlList
  */
  for (p = CtrlList;  p;  p = p->next)
    if (((CONTROL *) p->data)->hCtrl == hCtrl)
      ListDelete(&CtrlList, p);

  /*
    Destroy the control window
  */
  WinDestroy(hCtrl);

  /*
    Reset the focus to a valid place
  */
#ifdef OLD
  if (CtrlList && ((CONTROL*)CtrlList->data)->hCtrl)
    SetFocus(((CONTROL*)CtrlList->data)->hCtrl);
#else
  if ((hCtrl = GetTopWindow(hDEDlg)) != NULLHWND)
    SetFocus(hCtrl);
#endif
  else
    SetFocus(hDEDlg);

  bDlgIsDirty++;
}


/***************************************************************************/
/*                                                                         */
/***************************************************************************/
ClassToCtrlID(hCtrl)
  HWND hCtrl;
{
  switch (WinGetClass(hCtrl))
  {
    case NORMAL_CLASS :
      return ID_CTRL_BASE;
    case EDIT_CLASS   :
      return ID_CTRL_EDIT;
    case LISTBOX_CLASS:
      return ID_CTRL_LISTBOX;
    case SCROLLBAR_CLASS:
      return ID_CTRL_BASE;
    case PUSHBUTTON_CLASS :
      return ID_CTRL_PUSHBUTTON;
    case RADIOBUTTON_CLASS :
      return ID_CTRL_RADIOBUTTON;
    case CHECKBOX_CLASS :
      return ID_CTRL_CHECKBOX;
    case TEXT_CLASS :
      return ID_CTRL_TEXT;
    case FRAME_CLASS :
      return ID_CTRL_FRAME;
    case BOX_CLASS :
      return ID_CTRL_BOX;
    case COMBO_CLASS :
      return ID_CTRL_COMBO;
  }
}


BOOL pascal CtrlAddToList(hCtrl, lParam)
  HWND hCtrl;
  DWORD lParam;  /* we need this cause it's called by EnumWindows() */
{
  CONTROL *ci;

  WinSetWinProc(hCtrl, CtrlWinProc);
  ci = (CONTROL *) emalloc(sizeof(CONTROL));
  ci->hCtrl = hCtrl;
  ci->id = SendMessage(hCtrl, WM_GETID, 0, 0L);
  ListAdd(&CtrlList, ListCreate((char *) ci));
  return TRUE;
}

CtrlListToListbox(hLB)
  HWND hLB;
{
  LIST *p;
  char buf[133];
  HWND hCtrl;

  if (!hLB)
    return FALSE;

#ifdef OLD
  for (p = CtrlList;  p;  p = p->next)
  {
    CONTROL *c = (CONTROL *) p->data;
  	sprintf(buf, "%-10s  ", WinGetClassName(c->hCtrl));
    SendMessage(c->hCtrl, WM_GETTEXT, sizeof(buf) - 14, 
                             (DWORD) (char far *) (buf + 11));
    SendMessage(hLB, LB_ADDSTRING, 0, (DWORD) (char far *) buf);
  } 
#else
  for (hCtrl=GetTopWindow(hDEDlg); hCtrl; hCtrl=GetNextWindow(hCtrl,GW_HWNDNEXT))
  {
    sprintf(buf, "%-10s  ", WinGetClassName(hCtrl));
    SendMessage(hCtrl, WM_GETTEXT, sizeof(buf) - 14, (DWORD)(LPSTR)(buf + 11));
    SendMessage(hLB, LB_ADDSTRING, 0, (DWORD) (LPSTR) buf);
  } 
#endif
  
  return TRUE;
}

