/*===========================================================================*/
/*                                                                           */
/* File    : DE.C                                                            */
/*                                                                           */
/* Purpose : Simple dialog editor for the Magma Window System                */
/*                                                                           */
/* History :                                                                 */
/*                                                                           */
/* (C) Copyright 1989 Marc Adler/Magma Systems     All Rights Reserved       */
/*===========================================================================*/

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#ifdef TC
#include <alloc.h>
#include <mem.h>
#else
#include <malloc.h>
#include <memory.h>
#endif
#include "window.h"
#include "de.h"
#include "symbol.h"
#include "winrc.h"


LIST *CtrlList = NULL;
RECT CurrRect  = {1, 1, 0, 0};
RECT EmptyRect = {1, 1, 0, 0};

BOOL bExiting = FALSE;
BOOL bDlgMoveMode = FALSE;
BOOL bDlgSizeMode = FALSE;
BOOL bDlgIsDirty  = FALSE;
BOOL bGenerating  = FALSE;

HWND hMain;
HWND hMainMenu;
HWND hStatus;
HWND hCtrlDlg = NULLHWND;
HWND hDEDlg   = NULLHWND;
HWND hCurrCtrl = NULLHWND;
HWND hCtrlMove = NULLHWND;
HWND hCtrlSize = NULLHWND;

char CurrFileName[65] = { '\0' };
WORD hModule;

long (pascal *OldMainWinProc)();


/*===========================================================================*/
/*  main() - entry point of the dialog editor                                */
/*===========================================================================*/
main(argc, argv)
  int  argc;
  char **argv;
{
  MSG  msg;
  int    i = 1;

  if (argc >= 2)
  {
    for (i = 1;  i < argc;  i++)
    {
      char *arg = argv[i];
      if (*arg == '-' || *arg == '/')  /* parse the arguments   */
        switch (*++arg)
        {
          case 'm' : 
            WinUseMonoMap(TRUE);
            break;
        }
      else
        break;
    }
  }

  /*
    Initialize the dialog editor.
  */
  DEinit();

  /*
    Make sure that the user has a mouse...
  */
  if (!GetSystemMetrics(SM_MOUSEPRESENT))
  {
    if (MessageBox(NULLHWND,
                   "The dialog editor requires a mouse!\nDo you want to continue?",
                   "Warning", MB_YESNO) == IDNO)
      exit(0);
  }

  /*
    Read in the initial dialog box
  */
  if (argc > 1)
  {
    char fname[80];
    strcpy(fname, argv[1]);
    ChangeExtension(fname, ".res");
    if ((hDEDlg = DlgList(fname)))
    {
      ChangeExtension(fname, ".rc");
      strcpy(CurrFileName, fname);
      EnumChildWindows(hDEDlg, CtrlAddToList, 0L);
      WinAddChild(hMain, hDEDlg);
      ShowWindow(hDEDlg, TRUE);
      SetFocus(GetTopWindow(hDEDlg));
    }
  }

  /*
    Main message loop
  */
  while (GetMessage(&msg, NULLHWND, 0, 0))
  {
    if (hDEDlg && IsDialogMessage(hDEDlg, &msg))
      continue;
    TranslateAccelerator(&msg);
    DispatchMessage(&msg);
  }
}


/*===========================================================================*/
/*  MainWndProc - window procedure for the main window                       */
/*===========================================================================*/
long pascal MainWndProc(hWnd, message, wParam, lParam)
  HWND hWnd;
  WORD message;
  WORD wParam;
  DWORD lParam;
{
  switch (message)
  {
    case WM_QUIT       :
      /* End the dialog editor */
      exit(0);

    case WM_COMMAND    :
      /* We selected something from a menu, or pressed an accelerator key */
      DEdispatch(hWnd, message, wParam, lParam);
      break;

    case WM_HELP :
      HelpDialog();
      break;

    case WM_CHAR       :
      if (wParam == VK_ALT_X)
        PostQuitMessage(0);
      else if (wParam == VK_CTRL_D)
        CtrlDelete(hCurrCtrl);
      else
        CallWindowProc(OldMainWinProc, hWnd, message, wParam, lParam);
      break;

    case WM_SYSCOMMAND :
      if (wParam == SC_CLOSE)
        PostQuitMessage(0);
      else
        return DefWinProc(hWnd, message, wParam, lParam);

    default :
      /* Call the default window procedure for the main window */
      return DefWinProc(hWnd, message, wParam, lParam);
  }
}


/*****************************************************************************/
/*                                                                           */
/*                   INITIALIZATION ROUTINES                                 */
/*                                                                           */
/*****************************************************************************/

DEinit()
{
  HACCEL hAccel;
  HANDLE hMod;

  /*
    Initialize the window system - this *must* be done once in every appl
  */
  WinInit();
  WinUseSysColors(NULLHWND, TRUE);

  /*
    Set up the accelerator table and keys
  */
#if 0
  hAccel = AcceleratorCreate();
  AcceleratorSetFocus(hAccel);
  AcceleratorSetKey(ID_OPEN, VK_ALT_O);
  AcceleratorSetKey(ID_EXIT, VK_ALT_X);
  AcceleratorSetKey(ID_HELP, VK_F1);
#else
  hMod = OpenResourceFile("DE");
  if (hMod)
  {
    if ((hAccel = LoadAccelerators(hMod, MAKEINTRESOURCE(ID_DE))))
       AcceleratorSetFocus(hAccel);
    CloseResourceFile(hMod);
  }
#endif


  /*
    Create the main window and set the main window procedure.
    (The vertical scrollbar does nothing ... it's just for show!)
  */
  hMain = CreateWindow("Normal", "Dialog Editor",
                         (WS_BORDER | WS_CAPTION | WS_SYSMENU),
                         0, 0, 80, 24,
                         SYSTEM_COLOR,
                         0,                 /* id      */
                         NULLHWND,          /* hParent */
                         (HMENU) NULLHWND,  /* hMenu   */
                         0,                 /* hInst   */
                         (DWORD) NULL);     /* lpCreateParams */
  OldMainWinProc = WinSetWinProc(hMain, MainWndProc);
  /*
    Associate the accelerator table with the main window.
  */
  AcceleratorSetWnd(hMain);

  /*
    Create the status window...
  */
  hStatus = CreateWindow("Normal", NULL,
                         0x0000L,
                         0,24, 80, 1,
                         SYSTEM_COLOR, 0,
                         hMain,             /* hParent */
                         (HMENU) NULLHWND,  /* hMenu   */
                         0,                 /* hInst   */
                         (DWORD) NULL);     /* lpCreateParams */

  BuildMenu(hMain);
  ShowWindow(hMain, TRUE);
  SetFocus(hMain);
}


/*===========================================================================*/
/*  BuildMenu - create the main menu bar and all of the pulldowns            */
/*===========================================================================*/
HWND BuildMenu(hParent)
  HWND hParent;
{
  HWND hMenu;
  HWND hPop1, hPop2, hPop3;

  /*
    Create the menu bar
  */
  NULLCHECK(hMenu = CreateMenu(hParent));
  
  /*
    Create a pulldown
  */
  NULLCHECK(hPop1 = CreateMenu(hMenu));
  ChangeMenu(hPop1, 0, "~New",        ID_NEW,      MF_APPEND | MF_SHADOW);
  ChangeMenu(hPop1, 0, "~Open...ALT+O",ID_OPEN,    MF_APPEND);
  ChangeMenu(hPop1, 0, "~Save...",    ID_SAVE,     MF_APPEND);
  ChangeMenu(hPop1, 0, "Save ~as...", ID_SAVEAS,   MF_APPEND);
  ChangeMenu(hPop1, 0, NULL,          0,           MF_APPEND | MF_SEPARATOR);
  ChangeMenu(hPop1, 0, "~Includes...",ID_INCLUDES, MF_APPEND);
  ChangeMenu(hPop1, 0, "~Generate",   ID_GENERATE, MF_APPEND);
  ChangeMenu(hPop1, 0, "~Exit   ALT+X",ID_EXIT,    MF_APPEND);

  NULLCHECK(hPop2 = CreateMenu(hMenu));
  ChangeMenu(hPop2, 0, "~New",        ID_DLG_NEW,  MF_APPEND | MF_SHADOW);
  ChangeMenu(hPop2, 0, "~Move",       ID_DLG_MOVE, MF_APPEND);
  ChangeMenu(hPop2, 0, "~Size",       ID_DLG_SIZE, MF_APPEND);
  ChangeMenu(hPop2, 0, "~Rename...",  ID_DLG_NAME, MF_APPEND);
  
  NULLCHECK(hPop3 = CreateMenu(hMenu));
  ChangeMenu(hPop3, 0, "~New...",     ID_CTRL_NEW,    MF_APPEND | MF_SHADOW);
  ChangeMenu(hPop3, 0, "~Info...",    ID_CTRL_INFO,   MF_APPEND);
  ChangeMenu(hPop3, 0, "~Delete  CTRL+D",ID_CTRL_DELETE, MF_APPEND);
  ChangeMenu(hPop3, 0, "~Reorder...", ID_CTRL_ORDER,  MF_APPEND);
  ChangeMenu(hPop3, 0, "~Move",       ID_CTRL_MOVE,   MF_APPEND);
  ChangeMenu(hPop3, 0, "~Size",       ID_CTRL_SIZE,   MF_APPEND);
  
  /*
    Attach each of the pulldowns to their corresponding menubar entries
  */
  ChangeMenu(hMenu, 0, "~File",     hPop1, MF_POPUP | MF_APPEND);
  ChangeMenu(hMenu, 0, "~Dialog",   hPop2, MF_POPUP | MF_APPEND);
  ChangeMenu(hMenu, 0, "~Control",  hPop3, MF_POPUP | MF_APPEND);
  ChangeMenu(hMenu, 0, "~Help",     ID_HELP,MF_HELP | MF_APPEND);
  
  /*
    Associate the menubar with the main window
  */
  SetMenu(hParent, hMenu);
  return hMainMenu = hMenu;
}



/*****************************************************************************/
/*                                                                           */
/*               MAIN ROUTINES WHICH INTERACT WITH THE USER                  */
/*                                                                           */
/*****************************************************************************/

/*===========================================================================*/
/*  DEdispatch - this is called when we select a menu item                   */
/*===========================================================================*/
DEdispatch(hWnd, message, wParam, lParam)
  HWND hWnd;
  WORD message;
  WORD wParam;
  DWORD lParam;
{
  char szFileName[80];
  HWND hCtrl;

  /*
    'message' is WM_COMMAND.  'wParam' is the identifier of the menu entry.
  */
  switch (wParam)
  {
    case ID_OPEN :
      if (bDlgIsDirty)
      {
        if (DlgPromptToSave() == IDCANCEL)
          break;
      }
      if (hDEDlg)
      {
        DestroyWindow(hDEDlg);
        ListFree(&CtrlList, TRUE);
      }
      bDlgIsDirty = FALSE;

      if (DlgOpenFile(NULLHWND, "*.res", szFileName))
      {
        /*
          List all of the dialog boxes in the resource file and
          let the user choose one.
        */
        if ((hDEDlg = DlgList(szFileName)))
        {
          ChangeExtension(szFileName, ".rc");
          strcpy(CurrFileName, szFileName);
          EnumChildWindows(hDEDlg, CtrlAddToList, 0L);
          WinAddChild(hMain, hDEDlg);
          DialogBoxShow(hDEDlg);
          SetFocus(GetTopWindow(hDEDlg));
        }
      }
      break;

    case ID_SAVE   :
      if (CurrFileName[0])
      {
        DialogSave(hDEDlg, CurrFileName);
        bDlgIsDirty = FALSE;
        break;
      }	
      /* fall into ... */

    case ID_SAVEAS :
      SaveFileDlg();
      bDlgIsDirty = FALSE;
      break;

    case ID_INCLUDES :
      EditSymbols();
      break;

    case ID_GENERATE :
      bGenerating = TRUE;
      SaveFileDlg();
      bGenerating = FALSE;
      break;

    case ID_EXIT :
      if (bDlgIsDirty)
        if (DlgPromptToSave() == IDCANCEL)
          break;
      PostQuitMessage(0);
      break;
      
    case ID_HELP :
      HelpDialog();
      break;


    case ID_DLG_NEW :
      if (bDlgIsDirty)
        if (DlgPromptToSave() == IDCANCEL)
          break;
      if (hDEDlg)
      {
        DestroyWindow(hDEDlg);
        ListFree(&CtrlList, TRUE);
      }
      bDlgIsDirty = FALSE;
      DE_DialogNew();
      break;

    case ID_DLG_SIZE :
    case ID_DLG_MOVE :
      SendMessage(hWnd, WM_SYSCOMMAND,
                  (wParam == ID_DLG_MOVE) ? SC_MOVE : SC_SIZE, 0L);
      CtrlDisplayStatus();
      bDlgIsDirty++;
      break;

    case ID_DLG_NAME :
      if (DE_DialogRename())
      {
        bDlgIsDirty++;
        SendMessage(hDEDlg, WM_NCPAINT, 1, 0L);
      }
      break;

    case ID_CTRL_NEW :
      if (!RectIsEmpty(CurrRect))
      {
        BuildCtrlDialog();
        DialogBox(hCtrlDlg);
        CurrRect = EmptyRect;
        ShowWindow(hDEDlg, TRUE);      /* Get rid of the red rectangle */
        bDlgIsDirty++;
        EnableMenuItem(GetMenu(hMain), ID_CTRL_NEW, MF_BYCOMMAND|MF_DISABLED);
      }
      break;

    case ID_CTRL_INFO :
      /*
        See if the window which has the current input focus is actually
        a control of the dialog box. If so, modify its parameters.
      */
      if ((hCtrl = hCurrCtrl))
      {
#if 0
        CtrlEditInfo(hCtrl);
#else
        CtrlEditStyle();
#endif
        bDlgIsDirty++;
      }
      break;

    case ID_CTRL_DELETE :
      /*
        See if the window which has the current input focus is actually
        a control of the dialog box. If so, modify its parameters.
      */
      if ((hCtrl = hCurrCtrl))
      {
        CtrlDelete(hCtrl);
        bDlgIsDirty++;
      }
      break;

    case ID_CTRL_MOVE  :
      if (hCtrlMove)
        hCtrlMove = NULLHWND;
      else if ((hCtrl = hCurrCtrl))
        hCtrlMove = hCtrl;
      CheckMenuItem(GetMenu(hMain), ID_CTRL_MOVE,
                    MF_BYCOMMAND | (hCtrlMove ? MF_CHECKED : MF_UNCHECKED));
      break;

    case ID_CTRL_SIZE  :
      if (hCtrlSize)
        hCtrlSize = NULLHWND;
      else if ((hCtrl = hCurrCtrl))
        hCtrlSize = hCtrl;
      CheckMenuItem(GetMenu(hMain), ID_CTRL_SIZE,
                    MF_BYCOMMAND | (hCtrlSize ? MF_CHECKED : MF_UNCHECKED));
      break;


    case ID_CTRL_ORDER :
      CtrlReorder();
      bDlgIsDirty++;
      break;
  }

  return TRUE;
}


/*===========================================================================*/
/*
  DE_DialogProc()
  This is the window proc for the dialog box we are building. We are
  mainly interested in mouse messages.
*/
/*===========================================================================*/
long pascal DE_DialogProc(hDlg, msg, wParam, lParam)
  HWND hDlg;
  WORD msg;
  WORD wParam;
  DWORD lParam;
{
  static bSelecting = FALSE;
  static char *szSavedRectDEDlg = NULL;

  int mouserow = HIWORD(lParam);
  int mousecol = LOWORD(lParam);

  RECT dlgRect, ctrlRect;
  HWND hCtrl;

  GetWindowRect(hDEDlg, &dlgRect);

  switch (msg)
  {
    case WM_NCHITTEST :   /* allow clicks on the dialog box */
      return HTCLIENT;

    case WM_LBUTTONDOWN :
      if (RectContainsPoint(WinGetClient(hDEDlg), mouserow, mousecol))
      {
        SetFocus(hDEDlg);
        SetCapture(hDEDlg);

        /* 
           Pressing the left button on an unoccupied area will draw
           a rectangle which will define the control's area.
        */
        if (!RectIsEmpty(CurrRect))  /* if an old rect defined, refresh */
          ShowWindow(hDEDlg, TRUE);
        if (szSavedRectDEDlg)
          MyFree(szSavedRectDEDlg);

        szSavedRectDEDlg = emalloc(RectSize(&dlgRect) * 2);
        WinSaveRect(NULLHWND, &dlgRect, szSavedRectDEDlg);

        CurrRect.top  = CurrRect.bottom = mouserow;
        CurrRect.left = CurrRect.right  = mousecol;
        WinFillRect(NULLHWND, &CurrRect, ' ', 0x41);
        bSelecting = TRUE;
        EnableMenuItem(GetMenu(hMain), ID_CTRL_NEW, MF_BYCOMMAND|MF_ENABLED);
      }
      else
        return DefWinProc(hDlg, msg, wParam, lParam);
      break;

    case WM_LBUTTONUP :
      ReleaseCapture();
      SetFocus(hDEDlg);
      if (bDlgSizeMode)
        bDlgSizeMode = FALSE;
      else if (bDlgMoveMode)
        bDlgMoveMode = FALSE;
      else
      {
        WinFillRect(NULLHWND, &CurrRect, ' ', 0x41);
        bSelecting = FALSE;
      }
      bDlgIsDirty++;
      break;

    case WM_MOUSEMOVE :
      if (bSelecting)
      {
        WinRestoreRect(NULLHWND, &dlgRect, szSavedRectDEDlg);
/*      ShowWindow(hDEDlg, TRUE); */
        CurrRect.bottom = mouserow;
        CurrRect.right  = mousecol;
        RectSort(&CurrRect);
        WinFillRect(NULLHWND, &CurrRect, ' ', 0x41);
      }
      break;

    case WM_SYSCOMMAND :
      if (wParam == SC_CLOSE)
      {
        if (bDlgIsDirty)
        {
          if (DlgPromptToSave() == IDCANCEL)
            break;
        }
        DestroyWindow(hDEDlg);
        ListFree(&CtrlList, TRUE);
        bDlgIsDirty = FALSE;
        break;
      }
      else
        return DefWinProc(hDlg, msg, wParam, lParam);


    case WM_HELP    :
      HelpDialog();
      break;

    case WM_CHAR :
      if (wParam == VK_ESC && bSelecting)
      {
        ShowWindow(hDEDlg, TRUE);
        bSelecting = FALSE;
        return TRUE;
      }
      return FALSE;

    default :
      return DefWinProc(hDlg, msg, wParam, lParam);
  }
}


/*===========================================================================*/
/*   DE_DialogNew() - creates a new dialog box and draws it on the screen    */
/*===========================================================================*/
DE_DialogNew()
{
  hDEDlg = CreateWindow("Normal", "Dialog",
                         WS_BORDER | WS_MOVEBOX | WS_SIZEBOX | 
                                     WS_CAPTION | WS_SYSMENU,
                         10, 5, 60, 18,
                         0x70, 0,
                         hMain,             /* hParent */
                         (HMENU) NULLHWND,  /* hMenu   */
                         0,                 /* hInst   */
                         (DWORD) NULL);     /* lpCreateParams */
  NULLCHECK(hDEDlg);

  WinSetWinProc(hDEDlg, (long (pascal *)())DE_DialogProc); /* Sets the dlg box proc */
  ShowWindow(hDEDlg, TRUE);
}


DlgPromptToSave()
{
  int rc = MessageBox(NULLHWND,
                      "The dialog has been changed.\nDo you want to save it?",
                      "Warning", MB_YESNO);
  if (rc == IDYES)
    SaveFileDlg();
  return rc;
}


/*
  Rubberband()
    Lets the user drag around (or stretch) an outline of a window as the
    user resizes or moves the window.
*/
Rubberband(hWnd, bIsMoving)
  HWND hWnd;
  int  bIsMoving;
{
  RECT  r, newr, rMain;
  char  *szSavedRect;
  MSG msg;
  HWND  hOldFocus = GetFocus();

  if (!hDEDlg)
    return FALSE;

  SetFocus(hDEDlg);
  rMain = WinGetClient(hMain);
  r = WinGetRect(hWnd);
  szSavedRect = emalloc(RectSize(&rMain) * 2);
  WinSaveRect(hMain, &rMain, szSavedRect);

  VidFrame(r.top, r.left, r.bottom, r.right, WinGetAttr(hDEDlg), BORDER_DASHED);

  for (;;)
  {
    if (GetMessage(&msg, NULLHWND, 0, 0) == WM_LBUTTONUP)
      break;

    if (msg.message == WM_MOUSEMOVE)
    {
      /* Get the 0-based screen coords of the mouse */
      int mouserow = HIWORD(msg.lParam);
      int mousecol = LOWORD(msg.lParam);

      if (RectContainsPoint(rMain, mouserow, mousecol))
      {
        if (bIsMoving)
        {
          newr.top    = mouserow;
          newr.left   = mousecol;
          newr.bottom = mouserow + (r.bottom - r.top);
          newr.right  = mousecol + (r.right - r.left);
        }
        else /* bIsResizing */
        {
          newr.top    = r.top;
          newr.left   = r.left;
          newr.bottom = mouserow;
          newr.right  = mousecol;
        }
        	

        /*
          Sigh! This would be a whole lot easier if WinMove() restricted
          the area of the child window to the client rectangle of the parent...
        */
        if (newr.top  <= rMain.top  || newr.bottom >= rMain.bottom ||
            newr.left <= rMain.left || newr.right  >= rMain.right)
          continue;
      }
      else
        continue;
    }

    else if (msg.message == WM_CHAR)
    {
      newr = WinGetRect(hWnd);
      switch (msg.wParam)
      {
        case VK_LEFT :
          if (newr.left > 1 && newr.right > newr.left+1)
          {
            newr.right--;
            if (bIsMoving)
              newr.left--;
          }
          break;
        case VK_RIGHT:
          if (newr.right < VideoInfo.width - 2)
          {
            newr.right++;
            if (bIsMoving)
              newr.left++;
          }
          break;
        case VK_UP :
          if (newr.top > 2 && newr.bottom > newr.top+1)
          {
            newr.bottom--;
            if (bIsMoving)
              newr.top--;
          }
          break;
        case VK_DOWN :
          if (newr.bottom < VideoInfo.length - 2) 
          {
            newr.bottom++;
            if (bIsMoving)
              newr.top++;
          }
          break;
        case VK_ESC  :
        case '\r' :
          goto bye;
        default :
          break;
      }
    }

    WinRestoreRect(hMain, &rMain, szSavedRect);
    VidFrame(newr.top, newr.left, newr.bottom, newr.right, 0x07, 2);
    if (bIsMoving)
      WinMove(hWnd, newr.top, newr.left);
    else
      WinSetSize(hWnd, newr.bottom-newr.top+1, newr.right-newr.left+1);
    CtrlDisplayStatus();
  }

bye:
  free(szSavedRect);
  ShowWindow(hMain, TRUE);
  ReleaseCapture();
  SetFocus(hOldFocus);
  bDlgIsDirty++;
}


/***************************************************************************/
/*                                                                         */
/* CtrlDisplayStatus() - displays the status message                       */
/*                                                                         */
/***************************************************************************/
CtrlDisplayStatus()
{
  char buf[80];
  RECT r;
  HWND h;

  if (!(h = hCurrCtrl))
    h = hDEDlg;

  if (h)
  {
    r = WinGetRect(h);
    sprintf(buf, "[%-10s <%02d,%02d, %02d,%02d> %c%c]", 
                 WinGetClassName(h), r.top, r.left, r.bottom, r.right,
                 bDlgMoveMode ? 'M' : ' ', bDlgSizeMode ? 'S' : ' ');
  }
  else
    strcpy(buf, "[No current control]");

  WinPuts(hStatus, 0, 0, buf, WinGetAttr(hStatus));
}


NULLCHECK(HWND hWnd)
{
  if (hWnd == NULLHWND)
  {
    VidClearScreen(0x07);
    printf("Out of memory!!!\n");
    exit(1);
  }
}


ChangeExtension(fname, szExt)
  char *fname, *szExt;
{
  char *pExt = strchr(fname, '.');
  if (pExt)
    strcpy(pExt, szExt);
  else
    strcat(fname, szExt);
}


IsArrowKey(key)
{
  return (key == VK_LEFT || key == VK_RIGHT || key == VK_UP || key == VK_DOWN);
}


/*===========================================================================*/
/*                                                                           */
/* File    : DLGSAVE.C                                                       */
/*                                                                           */
/* Purpose :                                                                 */
/*                                                                           */
/*===========================================================================*/

extern BOOL pascal _DialogSaveChild(HWND hCtrl, DWORD lParam);
extern BOOL pascal _DlgEnumWindows(HWND hWnd,BOOL (pascal *func)(),DWORD lParam);

static FILE *DlgFp;


int  pascal DialogSave(hDlg, fname)
  HWND hDlg;
  char *fname;
{
  RECT   r;
  WINDOW *w;

  if ((DlgFp = fopen(fname, "w")) == NULL)
  {
/*
    MessageBox("Cannot open the file", fname, "for writing.", "Error!", MB_OK);
*/
    return FALSE;
  }

  if ((w = WID_TO_WIN(hDlg)) == NULL)
    return FALSE;
  r = WinGetRect(hDlg);

  SymbolWrite(DlgFp);

  fprintf(DlgFp, "%d DIALOG %d,%d,%d,%d\nCAPTION \"%s\"\nBEGIN\n",
                 w->idCtrl, r.left, r.top, RECT_WIDTH(r), RECT_HEIGHT(r),
                 (w->title) ? w->title : "");


  if (w->children)
    _DlgEnumWindows(w->children->win_id, _DialogSaveChild, 0L);
  
  fprintf(DlgFp, "END\n");
  fclose(DlgFp);
  return TRUE;
}


BOOL pascal _DialogSaveChild(hCtrl, lParam)
  HWND  hCtrl;
  DWORD lParam;
{
  RECT   r;
  int    iWidth;
  WINDOW *w;

  if ((w = WID_TO_WIN(hCtrl)) == NULL)
    return FALSE;
  r = WinGetRect(hCtrl);

  /*
    Don't output the scrollbar which belongs to a listbox.
  */
  if (w->class == SCROLLBAR_CLASS && w->parent->class == LISTBOX_CLASS)
    return TRUE;

  /*
    Don't output the child controls of a combo box
  */
  if (w->parent && w->parent->class == COMBO_CLASS)
    return TRUE;

  /*
    Adjust the width to the length of the text of a text class control,
    or in the case of a checkbox or radiobutton, the length of the
    text plus 4 for the left char, space, right char, and space.
  */
  if (w->class == TEXT_CLASS)
    iWidth = strlen(w->title);
  else if (w->class == CHECKBOX_CLASS || w->class == RADIOBUTTON_CLASS)
    iWidth = strlen(w->title) + 4;
  else
    iWidth = RECT_WIDTH(r);

  fprintf(DlgFp, "  %s \"%s\", %d, %d,%d,%d,%d 0x%lxL\n",
                  WinGetClassName(hCtrl),
                  (w->title) ? w->title : "", 
                  w->idCtrl,
                  r.left, r.top, iWidth, RECT_HEIGHT(r), w->flags);

  return TRUE;
}


BOOL pascal _DlgEnumWindows(hWnd, func, lParam)
  HWND  hWnd;
  BOOL (pascal *func)();
  DWORD lParam;
{
  WINDOW *w = WID_TO_WIN(hWnd);
  
  if (!w)
    return TRUE;

  if (w->sibling)
    _DlgEnumWindows(w->sibling->win_id, func, lParam);

  if (w->children)
    _DlgEnumWindows(w->children->win_id, func, lParam);

  (*func)(w->win_id, lParam);

  return TRUE;
}

