/*===========================================================================*/
/*                                                                           */
/* File    : DESYMBOL.C                                                      */
/*                                                                           */
/* Purpose : Simple dialog editor for the Magma Window System                */
/*                                                                           */
/* History : Symbol table routines for the dialog editor                     */
/*                                                                           */
/* (C) Copyright 1989 Marc Adler/Magma Systems     All Rights Reserved       */
/*===========================================================================*/

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#ifdef TC
#include <alloc.h>
#include <mem.h>
#else
#include <malloc.h>
#include <memory.h>
#endif

#include "window.h"
#include "symbol.h"

#include "de.h"

/****************************************************************************/
/*                                                                          */
/* SYMBOL.C                                                                 */
/*   A simple menu system editor and code generator for Presentation        */
/* Manager.                                                                 */
/*   These routines contain a simple symbol table mgmt facility for         */
/* associating textual definitions for the numeric ids.                     */
/*                                                                          */
/* (C) Copyright 1989  Marc Adler and Magma Systems  All Rights Reserved    */
/*                                                                          */
/* You are allowed to use this code for non-commercial use only. Commercial */
/* users should contact Magma Systems for licensing information.            */
/*                                                                          */
/****************************************************************************/
#define DLG_SYMBOL      260
#define ID_SYM_LISTBOX  261
#define ID_SYMNAME      262
#define ID_SYMID        263
#define ID_SYM_ADD      264
#define ID_SYM_CHANGE   265
#define ID_SYM_DELETE   266

SYMBOL *SymList = NULL;

char szIncludeFileName[65] = { '\0' };

int pascal EditSymDlgProc(HWND, WORD, WORD, DWORD);
WORD SymbolExtractID(char *szText);

HWND hSymListBox;

/*
DLG_SYMBOL DIALOG 10,5,61,17
  CAPTION "Symbols"
BEGIN
  TEXT "Symbol name:", 0,                    12,7,12,1
  EDIT "",             ID_SYMNAME,           26,7,32,1
  TEXT "Symbol ID:",   0,                    12,9,12,1
  EDIT "",             ID_SYMID,             26,9,10,1
  LISTBOX "",          ID_SYM_LISTBOX,       12,11,55,8   LBS_STANDARD
  PUSHBUTTON "~CHANGE", ID_SYM_CHANGE,        11,20,8,1
  PUSHBUTTON "~ADD",    ID_SYM_ADD,           24,20,5,1    BUTTON_DEFAULT
  PUSHBUTTON "~DELETE", ID_SYM_DELETE,        34,20,8,1
  PUSHBUTTON "~OK",     IDOK,                 47,20,4,1
END
*/

DTEMPLATE DlgSymbolTemplate[] =
{
  { "Text",        "Symbol:",0x0000,      12,7,12,1,     0             },
  { "Edit",        "",       WS_BORDER | WS_GROUP, 26,7,32,1,   ID_SYMNAME    },
  { "Text",        "ID:",    0x0000,      12,9,12,1,     0             },
  { "Edit",        "",       WS_BORDER | WS_GROUP, 26,9,10,1,   ID_SYMID      },
  { "Listbox",     "",       LBS_STANDARD,12,11,55,8,    ID_SYM_LISTBOX},

  { "PushButton",  "~CHANGE",0x0000,      11,20, 8,  1,  ID_SYM_CHANGE },
  { "PushButton",  "~ADD",   BS_DEFAULT,  29,20, 5,  1,  ID_SYM_ADD    },
  { "PushButton",  "~DELETE",0x0000,      44,20, 8,  1,  ID_SYM_DELETE },
  { "PushButton",  "~OK",    0x0000,      62,20, 4,  1,  IDOK          },
};

/****************************************************************************/
/* EditSymbols()                                                            */
/*                                                                          */
/****************************************************************************/
EditSymbols()
{
  HDLG hDlg;
  extern int pascal EditSymDlgProc();
  extern HWND hMain;

  hDlg = DialogCreateIndirect(hMain,
                            5,10,21,70,
                            "Symbols",
                            0x31, 0x0000L,
                            EditSymDlgProc,
                            DlgSymbolTemplate,
                            sizeof(DlgSymbolTemplate) / sizeof(DTEMPLATE));

  hSymListBox = GetDlgItem(hDlg, ID_SYM_LISTBOX);
  return (hDlg) ? DialogBox(hDlg) : FALSE;
}

int pascal EditSymDlgProc(HWND hDlg, WORD msg, WORD wParam, DWORD lParam)
{
  SYMBOL sym, *ps;
  int    iSel;
  char   buf[128];

  switch (msg)
  {
    case WM_INITDIALOG :
      SetDialogText(hDlg, ID_SYMNAME, "");
      SetDialogText(hDlg, ID_SYMID,   "");
      for (ps = SymList;  ps;  ps = ps->next)
      {
        sprintf(buf, "%-32s    %d", ps->szSymbol, ps->idSymbol);
        SendDlgItemMessage(hDlg, ID_SYM_LISTBOX, LB_ADDSTRING, 0,
                                                (DWORD) (LPSTR) buf);
      }
      return TRUE;

    case WM_COMMAND :
      if (wParam == ID_SYM_ADD || wParam == ID_SYM_CHANGE || wParam == ID_SYM_DELETE)
        if (GetFocus() != hSymListBox ||
            !SymbolQueryListBoxSelection(hDlg,sym.szSymbol,sizeof(sym.szSymbol),
                                          &sym.idSymbol, &iSel))
        {
          GetDialogText(hDlg, ID_SYMNAME, sym.szSymbol, sizeof(sym.szSymbol));
          GetDialogInt(hDlg, ID_SYMID, &sym.idSymbol);
        }

      switch (wParam)
      {
        case ID_SYM_ADD :
addit:
          SymbolAdd(sym.szSymbol, sym.idSymbol);
          sprintf(buf, "%-32s    %d", sym.szSymbol, sym.idSymbol);
          SendDlgItemMessage(hDlg, ID_SYM_LISTBOX, LB_ADDSTRING, 0,
                             (DWORD) (LPSTR) buf);
          return TRUE;

        case ID_SYM_CHANGE :
        case ID_SYM_DELETE :
          if (iSel >= 0)
          {
            SymbolDelete(sym.szSymbol, sym.idSymbol);
            SendDlgItemMessage(hDlg, ID_SYM_LISTBOX, LB_DELETESTRING,iSel,0L);
          }
          if (wParam == ID_SYM_CHANGE)
            goto addit;
          return TRUE;

        case IDOK     :
        case IDCANCEL :
          EndDialog(hDlg, IDCANCEL);
          return TRUE;

        case ID_SYM_LISTBOX:
          if (HIWORD(lParam) == LBN_SELCHANGE)
          {
            SymbolQueryListBoxSelection(hDlg, sym.szSymbol, sizeof(sym.szSymbol),
                                        &sym.idSymbol, &iSel);
            SetDialogText(hDlg, ID_SYMNAME, sym.szSymbol);
            SetDialogInt(hDlg, ID_SYMID, sym.idSymbol);
          }
          return TRUE;
      }
      break;  /* from WM_COMMAND */

    default:
      return FALSE;
  }
}



SymbolQueryListBoxSelection(HWND hDlg, char *szBuf, WORD iBufLen,
                                                    WORD *pID, int *pSel)
{
  *pSel = SendDlgItemMessage(hDlg, ID_SYM_LISTBOX, LB_GETCURSEL, 0, 0L);
  if (*pSel < 0)
  {
    *szBuf = '\0';
    return FALSE;
  }

  SendDlgItemMessage(hDlg, ID_SYM_LISTBOX, LB_GETTEXT, *pSel,
                                             (DWORD) (LPSTR) szBuf);
  *pID = SymbolExtractID(szBuf);

  return TRUE;
}


WORD SymbolExtractID(szText)
  char *szText;
{
  char *s;

  for (s = szText;  *s && !isspace(*s);  s++)
    ;
  for (*s++ = '\0';  *s && isspace(*s);  s++)
    ;
  return atoi(s);
}


SymbolAdd(szSym, idSym)
  char *szSym;
  WORD idSym;
{
  SYMBOL *sym = (SYMBOL *) calloc(sizeof(SYMBOL), 1);
  SYMBOL *prevsym;

  strcpy(sym->szSymbol, szSym);
  sym->idSymbol = idSym;

  if (SymList == NULL)
    SymList = sym;
  else
  {
    for (prevsym = SymList;  prevsym->next;  prevsym = prevsym->next)
      ;
    prevsym->next = sym;
  }
}


SymbolDelete(szSym, idSym)
  char *szSym;
  WORD idSym;
{
  SYMBOL *sym, *prevsym;

  for (prevsym = sym = SymList;  sym;  sym = sym->next, prevsym = sym)
    if (!strcmp(sym->szSymbol, szSym) && sym->idSymbol == idSym)
      break;

  if (!sym)
    return FALSE;

  if (sym == SymList)
    SymList = SymList->next;
  else
    prevsym->next = sym->next;

  free((char *) sym);
}


SYMBOL *SymbolFind(name, ID)
  char  *name;
  WORD  ID;
{
  SYMBOL *sym;

  if (name)
  {
    for (sym = SymList;  sym;  sym = sym->next)
      if (!strcmp(sym->szSymbol, name))
        return sym;
  }
  else
  {
    for (sym = SymList;  sym;  sym = sym->next)
      if (sym->idSymbol == ID)
        return sym;
  }
  return NULL;
}

SymbolWrite(fp)
  FILE *fp;
{
  SYMBOL *sym;

  for (sym = SymList;  sym;  sym = sym->next)
    fprintf(fp, "#define %-32s  %d\n", sym->szSymbol, sym->idSymbol);
}


/*===========================================================================*/
/*                                                                           */
/*  Routines to try to read symbols from an include file.                    */
/*                                                                           */
/*===========================================================================*/

SymbolRead(szFileName)
  char *szFileName;
{
  char szBuf[256], *s;
  char *pDefine, *pExpansion;
  int  c;
  FILE *fp;

  if (!szFileName[0])
    return TRUE;

  strcpy(szBuf, szFileName);
  if ((s = strchr(szBuf, '.')) != NULL)
    strcpy(s+1, "H");
  else
    strcat(szBuf, ".H");

  /*
    See if the include file exists, and if so, open it
  */
  if (access(szBuf, 0) != 0)
    return FALSE;
  if ((fp = fopen(szBuf, "r")) == NULL)
    return FALSE;

  /*
    Read all of the #defines from the include file
  */
  while (fgets(szBuf, sizeof(szBuf), fp))
  {
    if (szBuf[0] == '#')
    {
      szBuf[7] = '\0';
      if (!strcmp(szBuf, "#define"))
      {
        /*
          Point to the defined symbol
        */
        pDefine = szBuf + 8;
        while (isspace(*pDefine) && *pDefine != '\n')
          pDefine++;

        /*
          Search for the expansion
        */
        for (s = pDefine;  *s && !isspace(*s) && *s != '\n';  s++)
          ;
        if (*s == '\n')  /* see if we have "#define X" */
          continue;
        *s++ = '\0';

        /*
          Find the start of the macro expansion
        */
        for (  ;  *s && isspace(*s) && *s != '\n';  s++)
          ;
        if (*s == '\n')  /* see if we have "#define X" */
          continue;
        pExpansion = s;

        /*
          Get rid of trailing whitespace
        */
        if ((s = strchr(pExpansion, '\n')) != NULL)
          *s = '\0';
        rtrim(pExpansion);

        /*
          Add the symbol if it has not been defined already
        */
        if (!SymbolFind(pDefine, 0))
          SymbolAdd(pDefine, atoi(pExpansion));
      }
    }
  }

  /*
    Close the include file and return...
  */
  fclose(fp);
  return TRUE;
}

