/*
   Library: DataBase++ Ver. 2.00.
   File:    DBPPDATA.HPP
   Purpose: Declarations for class DBDatabase. Code in DBPPDATA.cpp
   Notice:  Copyright (C), 1992 - 1994, Jeff Stapleton. All rights reserved.
*/

/* Declaratons for the DBDatabase class.
*/

#ifndef __DBPPDATA_HPP
#define __DBPPDATA_HPP

#include <dbppobjt.hpp>    // DBObject class.
#include <dbppstru.hpp>    // DBStructure class.
#include <dbppindx.hpp>    // DBIndex class.
#include <dbpparay.hpp>    // DBArray class.
#include <stdio.h>

extern "C" {
#include <d4all.h>         // CodeBase
}

class DBDatabase;

typedef BOOL ( * FilterFunc ) ( DBDatabase &dbf );      // Filter function pointer.

/* DBDatabase class using the CodeBase 5.0 library.
*/
class DBDatabase : public DBObject
{
   public:

      /* Constructors and Destructor.
      */
      DBDatabase();                                        // Make a new DBDatabase.
      DBDatabase( char * fName );                          // Make a new DBDatabase, with it's file name.
      virtual ~DBDatabase();                               // Destroy it.
      
      /* Class methods.
      */
      static CODE4 *     CodeBase() { return &codeBase; }   // Return a pointer to the CODE4 struct.
      static char *      GetCentury() { return century; }   // Return the century.
      static int         Instances() { return instances; }  // Number of instances of self.
      static void        Initialize();
      static void        Shutdown();
      static int         OpenFiles() { return openFiles; }  // Number of open files.
      static void        SetCentury( char *aCentury ) { strcpy( century, aCentury ); }      // Set century.
      static void        SetMultiUser( BOOL state );        // set multi user.
      static BOOL        SetSafety( BOOL flag );            // Set safety.
      static void        CloseAll();                        // Close all open files.

      /* Instance methods.
      */
      BOOL        append();                                 // Append a blank record.
      BOOL        appendBlank() { return append(); }        // Append a blank record.
      BOOL        bof() { return d4bof( Data4 ); }         // TRUE if bof.
      BOOL        close();                                  // Close the opened file.
      BOOL        create( DBStructure& dbs );                 // Create a database.
      BOOL        create( DBStructure& dbs, DBIndexTag& idx );  // Create a database and index.
      BOOL        createIndex( DBIndexTag& idx );             // Create an index file.
      void        deleteRec() { d4delete( Data4 ); }       // Delete a record.
      BOOL        deleted() { return d4deleted( Data4 ); } // Return's TRUE if deleted.
      BOOL        eof() { return d4eof( Data4 ); }         // TRUE if eof.
      int         fieldNum( char *fldName ) { return( d4field_number( Data4, fldName ) ); }                // Return position of the field fldName.
      int         fieldType( char *fldName );               // Return the field type.
      BOOL        fileLock() { return( ( d4lock_file( Data4 ) ) == 0 ); }                              // Lock file. Return's 1 if successful.
      char *      fileName() { return FileName; }           // Return the file name.
      DATA4 *     getDATA4() const { return Data4; }      // Return pointer to Data4.
      char *      getDate( char *fldName );                 // Return a date field.
      double      getDouble( char *fldName );               // Return a double field.
      int         getInt( char *fldName );                  // Return an int.
      int         getLogical( char *fldName );              // Return TRUE if true. 
      long        getLong( char *fldName );                 // Return a long field.
      char *      getMemo( char *fldName );                 // Return a memo.
      char *      getString( char *fldName );               // Return a string field.
      void        go( long nRecno );                        // Go to a record number.
      void        goBottom();                               // Go to eof.
      void        goTop();                                  // Go to bof.
      char *      indexTag();                               // Return's name of current index tag.
      BOOL        isOpen() { return openState; }            // TRUE if open.
      void        memoPack();                               // Pack the memo file.
      int         numFields() { return( d4num_fields( Data4 ) ); }                              // Return's number of fields in DBDatabase.
      BOOL        open();                                   // Open the file.
      #if !defined( S4FOX ) && !defined( INDEX_GROUPS )
      BOOL        openIndex( char * indexFile );            // Open an index file.
      #endif
      void        pack( int mPack = 0 );                    // Pack the file. If mPack is TRUE, the associated memo file is also packed.
      long        reccount() { return d4reccount( Data4 ); }                               // Return number of records.
      void        recall() { d4recall( Data4 ); }          // Recall a deleted record.
      void        reindex();                                // Reindex the file.
      BOOL        recLock() { return( ( d4lock( Data4, recno() ) ) == 0 ); }                                // Lock current record. Return's 1 if successful.
      long        recno() { return d4recno( Data4 ); }                                  // Return record number.
      void        replace( char *fldName, char *str );      // Character, memo and date.
      void        replace( char *fldName, double d );       // double
      void        replace( char *fldName, int i );          // int's and logical
      void        replace( char *fldName, long l );         // long.
      int         seek( char *seekExpr );                   // Seek an index for seekExpr.
      void        setFileName( char * fName );              // Set the FileName.
      void        setFilter( FilterFunc ff ) { pFilterFunc = ff; }   // Set a filter function.
      BOOL        setIndexTag( char *aTagName = NULL );     // Select an index tag.
      void        skip( int nRecs = 1 );                    // Skip record(s) nRecs.
      void        unlock() { d4unlock( Data4 ); }             // Release all locks.
      int         unlockIndex();                            // Unlock current index if any.
      void        zap( int mPack = 0 );                     // Zap the file. If mPack is TRUE, the associated memo file is also packed.
      
   private:

      /* Class variables.
      */
      static CODE4   codeBase;                              // Code Base struct.
      static int     openFiles;                             // Number of open files.
      static int     instances;                             // Number of DBDatabase objects.
      static char    century[ 3 ];                          // Hold's century chars.
      static BOOL    safety;                                // FALSE if create overwrites, TRUE if warning. Set to FALSE.
      static BOOL    multiUser;                             // TRUE if multi user, otherwise FALSE;
         
      /* Instance vars.                                     
      */                                                    
      char        FileName [ 80 ];                          // Name of the file.
      DATA4 *     Data4;                                    // Pointer to data structure.
      BOOL        openState;                                // TRUE if open; FALSE if closed.
      FilterFunc  pFilterFunc;                              // Pointer to a user supplied filter function.
         
      /* Private methods                                    
      */                                                    
      void        init();                                   // Initialize this.
      void        removeFromArray();                        // Remove this from array.
};

#endif // _DBPPDATA_HPP
