// DBOBJ.H
//
// Definitions for use by DBObject and client programs
//
// (c) 1992 ShaunB - CompuServe: 70043,2641
//
// Placed in the public domain by the author
//
// Permission is hereby given to make modifications to any code provided.
//
// Author accepts no responsibility for breaking your files !!


#ifndef _DBOBJ_H
#define _DBOBJ_H

// Some errors we might encounter
#define ER_UNDEFRECORD			1
#define ER_BADRECREAD			2
#define ER_BADRECWRITE			3
#define ER_BADHDRREAD			4
#define ER_BADHDRWRITE			5
#define ER_BADOPEN				6
#define ER_BADCLOSE				7
#define ER_BADRECNUM				8
#define ER_NOEXIST				9

// Define a few datatypes
typedef unsigned char	byte;
typedef unsigned int  	word;
typedef unsigned long 	dword;
typedef unsigned int		bool;

const int true  = 1;
const int false = 0;
const int NoError = 0;

// Structure of a dBase III+ field descriptor
struct DBField {
	char	fldName[11];
	char	fldType;
	dword	dataAddress;
	byte	fldLen;
	byte	fldDec;
	byte	reserved[14];
};

// Structure of a dBase III+ header
struct DBHeader {
	byte 	version,
			lastYY,
			lastMM,
			lastDD;
	dword	lastRec;
	word	headerLen,
			recLen;
	byte	reserved[20];
};


// Declare the DBOject class
class DBObject {
	private:
		DBHeader		Header;				// Store header of file here
		DBField		*pFields;         // Array of field descriptors
		int         *pFieldOfs;       // Array of field value offsets in buffer
		char 			*cFileName;       // Name of the file
		long 			nCurPos;          // Current record position starting at 1
		void 			*pRecBuff;        // Points to record buffer
		int			nNumFields;       // Number of fields on file
		FILE 			*pFilePtr;        // 'C' file stream pointer
		int  			nErrorCode;       // Last error code encountered
		bool			bUpdated;         // Set if the file was updated
		bool			bEof;             // Set if at eof
		bool			bBof;             // Set if bof
		bool			bDirty;           // Set if buffer has changed
		bool			bAutoAccess;      // Should record be read/written automatically

		// Some private methods - not for client use
		void InitVars(void);
		const long GetFileBytePos(void)		{ return(ftell(pFilePtr)); };
		void GoFileBytePos(const long nPos)
			{ fseek(pFilePtr, nPos, SEEK_SET); };
		void LoadHeader(void);
		void WriteHeader(void);
		void ClearEof(void)						{ bEof = false; };
		void SetEof(void)							{ bEof = true; };
		void ClearBof(void)						{ bBof = false; };
		void SetBof(void)							{ bBof = true; };
		const bool ValidField(const int n);

		// Access members
		void ReadRecord(void);
		void WriteRecord(void);
		void WriteEof(void);
		void ClearBuffer(void);

	public:
		// Constructor/destructor members
		DBObject(const char *cFName);
		~DBObject(void);

		// Instantiation members
		void Open(void);
		void Close(void);

		// Data access members
		void SetError(int nCode)  		{ nErrorCode = nCode; };
		void ClearError(void) 		   { nErrorCode = NoError; };
		void SetDirty(void)				{ bDirty = true; };
		void AppendRecord(void);

		// Information members
		const char *FileName(void) 	{ return(cFileName); };
		const int RecLen(void) 			{ return(Header.recLen); };
		const long LastRec(void) 		{ return(Header.lastRec); };
		const long RecNo(void) 			{ return(nCurPos); };
		const int IsError(void) 		{ return(nErrorCode != NoError); };
		const int ErrorCode(void) 		{ return(nErrorCode); };
		const bool Exists(void);
		const int IsOpen(void)			{ return(pFilePtr != NULL); };
		const int IsClosed(void)		{ return(!IsOpen()); };
		const int FieldCount(void)		{ return(nNumFields); };
		const bool Eof(void)				{ return(bEof); };
		const bool Bof(void)				{ return(bBof); };
		const bool IsDeleted(void)		{ return( ((char *)pRecBuff)[0] == '*'); };
		void Delete(void)					{ ((char *) pRecBuff)[0] = '*'; SetDirty(); };
		void Recall(void)             { ((char *) pRecBuff)[0] = '*'; SetDirty(); };

		const char *FieldName(const int n);
		const char FieldType(const int n);
		const int FieldLen(const int n);
		const int FieldDec(const int n);
		const int FieldOfs(const int n);
		const int FieldPos(const char *name);

		// Field access methods
		const char *GetFieldValue(const int n, char *f = NULL);
		void SetFieldValue(const int n, const char *s);
		const char *GetFieldValue(const char *name, char *f = NULL);
		void SetFieldValue(const char *name, const char *s);
		const void *RecBuff(void)			{ return(pRecBuff); };

		// Positioning members
		void GoRecord(const long nRecNo);
		void GoTop(void);
		void GoBottom(void);
		void GoNextRecord(long n = 1L);
		void GoPrevRecord(long n = 1L);

		// Automatic access facilities
		void SetAutoAccessOn(void)		{ bAutoAccess = true; };
		void SetAutoAccessOff(void)	{ bAutoAccess = false; };
		const bool AutoAccess(void)	{ return(bAutoAccess); };
};

#endif