;---------------------------------------------------------------------------
;
;  Module:   newstack.asm
;
;  Purpose:
;      Provides functions to switch stacks under Windows 3.0.  These
;      functions are _PROTECTED MODE_ ONLY!  Do NOT attempt to use
;      these functions with REAL MODE - functionality is not guaranteed.
;
;  Development Team:
;     Bryan A. Woodruff
;     Tammy Steele
;
;  History:   Date       Author              Comment
;              4/17/92   BryanW, TammySt     Initial version.
;	       5/05/92   TammySt             revised to meet code standards
;---------------------------------------------------------------------------
;
;     Written by Microsoft Product Support Services, Windows Developer Support.
;     Copyright (c) 1992 Microsoft Corporation. All rights reserved.
;
;---------------------------------------------------------------------------

        .286

memM    equ     1               ; medium model

        .xlist
        include Windows.INC
        include CMacros.INC
        .list

        ?PLM = 1

pStackTop       equ     0ah
pStackMin       equ     0ch
pStackBot       equ     0eh

wOldSP          equ     02h
wOldSS          equ     04h

;------------------------------------------------------------------------
; Data
;------------------------------------------------------------------------

ExternA <__WinFlags>

sBegin DATA
   staticW  wSavedSI, 0
   staticW  wSavedDS, 0
   staticD  dwRetAddr, 0
   staticD  dwVersionInfo, 0
sEnd

;------------------------------------------------------------------------
; External procedures 
;------------------------------------------------------------------------

externFP    <SWITCHSTACKTO>
externFP    <SWITCHSTACKBACK>
externFP    <GETVERSION>
externFP    <FATALAPPEXIT>


sBegin CODE

;------------------------------------------------------------------------
; Fatal error string
;------------------------------------------------------------------------

ErrStr  db      'Fatal error! Protected mode support only!'
        db      0
       
;------------------------------------------------------------------------
; Code
;------------------------------------------------------------------------

        assumes cs, CODE
        assumes ds, nothing


;------------------------------------------------------------------------
;  VOID FAR PASCAL NewStack( UINT uStackSeg, UINT uStackPtr,
;                            UINT uStackTop )
;
;  Description:
;     Switches to a new stack in the uStackSeg segment.  uStackPtr
;     is an offset in the segment to the beginning (SP) of the stack.
;      uStackTop is in the segment to the top of the stack.
;
;                           +------------------+  x + nSize <==  uStackPtr
;                           |                  |
;                           |                  |
;                           |                  |  New Stack Seg   
;                           |                  |
;                           |                  |
;                           |                  |
;     uStackSeg => Seg XXXX +------------------+  x + 0 <==  uStackTop
;
;     where x = offset of stack into segment and nSize is the stack size,
;     for GlobalAlloc()'s, LocalInit() is required and x will be 16.
;
;  Parameters:
;     UINT uStackSeg
;        stack segment (selector)
;
;     UINT uStackPtr
;        stack pointer (SP)
;
;     UINT uStackTop
;        stack top (see above for further descriptions)
;
;  History:   Date       Author      Comment
;              4/17/92   BryanW      Initial version.
;
;------------------------------------------------------------------------

cProc NewStack, <PUBLIC, FAR>
;parmW uStackSeg
;parmW uStackPtr
;parmW uStackTop
cBegin nogen
        cCall   CheckVersion                            ; only for version 3.0
        cmp     ax, 30ah                                ; if 3.1 or greater
        jb      @F                                      ; use the Windows API
        jmp     SwitchStackTo                           ; "SwitchStackTo"

@@:
        mov     ax, __WinFlags                          ; get WinFlags
        test    al, WF_PMODE                            ; test for PMODE
        jnz     @F                                      ; OK, if set.
        push    0                                       ; must be 0
        push    cs                                      ; push segment
        push    offset ErrStr                           ; push offset
        cCall   FatalAppExit                            ; fatal error
        jmp     short NS_Exit

@@:
        mov     ax, DGROUP                              ; setup ES to our DATA
        mov     es, ax                                  ; ES <= our DATA seg
        mov     es:[wSavedSI], si
        mov     es:[wSavedDS], ds
        pop     word ptr es:[dwRetAddr]                 ; save return addr
        pop     word ptr es:[dwRetAddr + 2]
        pop     ax                                      ; uStackTop
        pop     bx                                      ; uStackPtr
        pop     dx                                      ; uStackSeg
        mov     si, bp                                  ; calculate local
        dec     si                                      ; variable area
        dec     si                                      ; dec for saved DS
        mov     cx, si                                  
        sub     cx, sp                                  ; calc size in bytes
        shr     cx, 1                                   ; convert to words
        push    bp                                      ; save bp
        mov     ax, ss                                  ; ax <= ss

        mov     es, ax                                  ; es <= ax
        mov     ds, dx                                  ; ds <= uStackSeg
        mov     ds:[2], sp                              ; save sp in segment
        mov     ds:[4], ss                              ; save ss in segment
        mov     ds:[pStackTop], ax                      ; save uStackTop
        mov     ds:[pStackMin], bx                      ; save uStackPtr
        mov     ds:[pStackBot], bx                      ; save uStackPtr
        mov     ss, dx                                  ; switch to new stack
        mov     sp, bx                                  ; sp <= uStackPtr
        mov     bp, bx                                  ; bp <= uStackPtr
        xor     ax, ax                                  ; null terminate
        push    ax                                      ; the bp chain
        jcxz    short NS_NoVars                         ; we have local vars?

NS_CopyVars:
        dec     si                                      ; copy local vars
        dec     si                                      ; to new stack
        push    es:[si]
        loop    NS_CopyVars

NS_NoVars:
        mov     ax, DGROUP
        mov     es, ax

        push    word ptr es:[dwRetAddr + 2]             ; push CS of return
        push    word ptr es:[dwRetAddr]                 ; push IP of return
        mov     si, es:[wSavedSI]                       ; restore SI
        mov     ds, es:[wSavedDS]                       ; restore DS

NS_Exit:
        ret                                             ; go back!

cEnd nogen

;------------------------------------------------------------------------
;  VOID FAR PASCAL OldStack( VOID )
;
;  Description:
;
;        Switches back to the old stack.
;
;  Parameters:
;     NONE
;
;  History:   Date       Author      Comment
;              4/17/92   BryanW      Initial version.
;
;------------------------------------------------------------------------

cProc OldStack, <PUBLIC, FAR>
cBegin nogen
        cCall   CheckVersion                            ; only for version 3.0
        cmp     ax, 30ah                                ; if 3.1 or greater
        jb      @F                                      ; use the Windows API
        jmp     SwitchStackBack                         ; "SwitchStackBack"

@@:
        mov     ax, __WinFlags                          ; get WinFlags
        test    al, WF_PMODE                            ; test for PMODE
        jnz     @F                                      ; OK, if set.
        push    0                                       ; must be 0
        push    cs                                      ; push segment
        push    offset ErrStr                           ; push offset
        cCall   FatalAppExit                            ; fatal error
        jmp     short OS_Exit

@@:
        mov     ax, DGROUP                              ; setup our data area
        mov     es, ax                                  
        pop     word ptr es:[dwRetAddr]                 ; save return IP
        pop     word ptr es:[dwRetAddr + 2]             ; save return CS
        xor     bx, bx
        xor     cx, cx
        xchg    cx, ss:[wOldSP]                         ; get old SP
        xchg    bx, ss:[wOldSS]                         ; get old SS
        mov     ss, bx                                  ; restore ss
        mov     sp, cx                                  ; restore sp
        pop     bp                                      ; restore bp
        push    word ptr es:[dwRetAddr + 2]             ; push return CS
        push    word ptr es:[dwRetAddr]                 ; push return IP

OS_Exit:
        ret                                             ; go back!

cEnd nogen

;------------------------------------------------------------------------
;  WORD NEAR PASCAL CheckVersion( VOID )
;
;  Description:
;     returns version of Windows in AX. NOTE: We will only call the
;     Windows API once.
;
;  Parameters:
;     NONE
;
;  History:   Date       Author      Comment
;              4/17/92   BryanW      Initial version.
;
;------------------------------------------------------------------------

cProc CheckVersion, <PRIVATE, NEAR>
cBegin  nogen
        push    ds
        mov     ax, DGROUP
        mov     ds, ax

        assumes ds, DATA
        mov     ax, word ptr [dwVersionInfo]            ; already got the
        cmp     ax, 0                                   ; version info?
        jne     @F

        cCall   GetVersion
        mov     word ptr [dwVersionInfo], ax
        mov     word ptr [dwVersionInfo + 2], dx

@@:
        pop     ds
        ret

cEnd    nogen

sEnd

end

;---------------------------------------------------------------------------
;  End of File: newstack.asm
;---------------------------------------------------------------------------

