/*---------------------------------------------------------------------------
	WINLIB.C  -- Window Management

	Version 1.00 (C)Copyright Andrew Trzeciak. 1993
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdarg.h>

#include "winlib.h"
#include "scrlib.h"
#include "utlib.h"

typedef unsigned int UINT;

#define WINDOW_GUARD	0x4A4E
#define WN_CHATTR(p)	((UINT)w->wdef.Wback*0x1000 + (UINT)w->wdef.Wfore*0x100 + (UINT)w->wdef.fillch)
#define WN_ATTR(p)	((UINT)w->wdef.Wback*0x10 + (UINT)w->wdef.Wfore)

WINTYPE FullScreen;

#define PRINTBUF_SIZE		2001

static UINT PrintfBufferSize = PRINTBUF_SIZE;
static UINT *PrintfBuffer = NULL;

static UINT *ScrBuffer = NULL;
static UINT ScrSize = 0;
static UINT ScrWidth = 0;
static UINT ScrDepth = 0;

static UseListPtr	UseList;

static WINTYPE windowstack;
static WINTYPE cursorstack;
static UINT CursorLines;
static int wn_initflag = CLR;

/*---------------------------------------------------------------------------

*/
static WINTYPE wn_valid (WINTYPE w)
{
extern int cwin_error;

	if (w == NULL || w->guard != WINDOW_GUARD)
		{
		cwin_error = CW_WINHANDLE_ERR;
		ut_fatalerror ();
		}
return w;
}

/*---------------------------------------------------------------------------

*/
static void wn_clipframe (register WINTYPE w)
{
int i = w->wdef.isframe ? 1 : 0;

	if (w->left < w->wdef.xleft+i)
		w->left = w->wdef.xleft+i;
	else if (w->left > w->wdef.xright-i)
      w->left = w->wdef.xright-i;

	if (w->right > w->wdef.xright-i)
		w->right = w->wdef.xright-i;
	else if (w->right < w->wdef.xleft+i)
      w->right = w->wdef.xleft+i;

	if (w->top < w->wdef.xtop+i)
		w->top = w->wdef.xtop+i;
	else if (w->top > w->wdef.xbottom-i)
		w->top = w->wdef.xbottom-i;

	if (w->bottom > w->wdef.xbottom-i)
		w->bottom = w->wdef.xbottom-i;
	else if (w->bottom < w->wdef.xtop+i)
		w->bottom = w->wdef.xtop+i;

	w->width = w->right-w->left+1;
	w->depth = w->bottom-w->top+1;
}

/*---------------------------------------------------------------------------

*/
static void wn_clipxy (register WINTYPE w, UINT *x, UINT *y)
{
	if (w->wdef.isframe && !w->wdef.iswrap)
		{
		if (*x > (UINT)w->width+1)
			*x = w->width+1;
		if (*y > (UINT)w->depth+1)
			*y = w->depth+1;
		}
	else
		{
		if (*x == 0)
			*x = 1;
		else if (*x > (UINT)w->width)
			*x = w->width;

		if (*y == 0)
			*y = 1;
		else if (*y > (UINT)w->depth)
			*y = w->depth;
		}
}

/*---------------------------------------------------------------------------

*/
static void wn_resetcursor (void)
{

	if (cursorstack == NULL ||
			wn_obscuredat(cursorstack, cursorstack->curx, cursorstack->cury))
		{
		sc_setcursize(0x2000);	/* cursor disabled */
		}
	else
		{
		sc_setcurpos(cursorstack->left+cursorstack->curx-1,
						cursorstack->top+cursorstack->cury-1);
		sc_setcursize(CursorLines);	/* cursor enabled */
		}
}

/*---------------------------------------------------------------------------

*/
static void wn_unlinkcursor (WINTYPE w)
{
register WINTYPE p;

	if (cursorstack == w)
		cursorstack = w->cursor_chain;
	else
		{
		p = cursorstack;
		while (p != NULL)
			{
			if (p->cursor_chain == w)
				{
				p->cursor_chain = w->cursor_chain;
				break;
				}
			p = p->cursor_chain;
			}
		}
	w->cursor_chain = NULL;
}

/*---------------------------------------------------------------------------

*/
void wn_cursoron (void)
{
WINTYPE cw;

	wn_unlinkcursor(cw = wn_curwin());

	cw->wdef.iscursor = SET;
	if (!cw->wdef.hidden)
		{
		cw->cursor_chain = cursorstack;
		cursorstack = cw;
		}

	wn_resetcursor();
}

/*---------------------------------------------------------------------------

*/
void wn_cursoroff (void)
{
WINTYPE cw;

	wn_unlinkcursor(cw = wn_curwin());

	cw->wdef.iscursor = CLR;

	wn_resetcursor();
}

/*---------------------------------------------------------------------------

*/
static void wn_takeoffstack (WINTYPE w)
{
register WINTYPE pw;

	if (w == windowstack)
		windowstack = w->next;
	else
		{
		pw = windowstack;
		while (pw != NULL)
			{
			if (pw->next == w)
				{
				pw->next = w->next;
				break;
				}
			pw = pw->next;
			}
		}
	 w->next = NULL;
}

/*---------------------------------------------------------------------------

	Updates the screen from the window buffer
*/
static void wn_updatescreen (WINTYPE w, UINT x, UINT y, UINT len)
{
register WINTYPE  p;
int nextLen, nextX;
int oxa,oxb,ax,bx;

	if (w->wdef.hidden)
		return;

	while (len != 0)
		{
		/* adjust co ordinates for crossing windows */
		p = windowstack;
		nextLen = 0;
		ax = x;
		bx = x+len-1;
		while (p != w && p != NULL)
			{
			if (y >= p->wdef.xtop && y <= p->wdef.xbottom)
				{
				oxa = p->wdef.xleft;
				oxb = p->wdef.xright;
				if (ax >= oxa && bx <= oxb)	/* wiped out */
					{
					ax = bx+1;
					break;
					}
				else if (ax <= oxb && bx >= oxa)	/* some interaction */
					{ 
					if (ax < oxa && bx > oxb)	 /* cut into two */
						{ 
						bx = oxa-1;
						nextX = oxb+1;
						nextLen = x+len-nextX;
						}
					else if (bx > oxb)	/* left edge cut off */
						ax = oxb+1;
					else if (ax < oxa)	/* right edge cut off */
						bx = oxa-1;
					}
				}
			p = p->next;
			}

		if ((len = bx-ax+1) != 0)
				sc_buf2scr(ax,y,&w->wbuf[(ax-w->wdef.xleft)+(y-w->wdef.xtop)*w->xwidth],len);
		x   = nextX;
		len = nextLen;
		}
}

/*---------------------------------------------------------------------------

	redraws rectangular portion of the window from the buffer
*/
static void wn_redrawsection (WINTYPE w, int xleft, int xtop, int xright, int xbottom)
{
int y;
	for (y=xtop; y<=xbottom; y++)
    wn_updatescreen(w,xleft,y,xright-xleft+1);
}

/*---------------------------------------------------------------------------

*/
static void wn_redrawwindow (WINTYPE w)
{
	wn_redrawsection (w,w->wdef.xleft,w->wdef.xtop,w->wdef.xright,w->wdef.xbottom);
}

/*---------------------------------------------------------------------------

	Re-displays windows Obscured by W from NW
*/
static void wn_displaybeneath (WINTYPE w, register WINTYPE nw)
{
	while (nw != NULL)
		{
		if (	w->wdef.xright   >= nw->wdef.xleft &&
				w->wdef.xleft    <= nw->wdef.xright &&
				w->wdef.xbottom  >= nw->wdef.xtop &&
				w->wdef.xtop     <= nw->wdef.xbottom )	/* windows cross */
			{
         /* calculate Intersection */
			wn_redrawsection (nw, max(w->wdef.xleft, nw->wdef.xleft),
										 max(w->wdef.xtop, nw->wdef.xtop),
										 min(w->wdef.xright, nw->wdef.xright),
										 min(w->wdef.xbottom, nw->wdef.xbottom) );
			}
      nw = nw->next;
		}
}

/*---------------------------------------------------------------------------

*/
static void wn_drawframe (register WINTYPE w)
{
UINT pos,attr,tl,tm;
int i;

	if (! w->wdef.isframe)
		{
		if (w->xwidth < 3 || w->xdepth < 3)
			return;
		w->wdef.isframe = SET;
		wn_clipframe(w);
		wn_resetcursor();
		}

	attr = w->wdef.Ffore + w->wdef.Fback * 0x10;

	ut_bufferwrite(&w->wbuf[0], &w->wdef.framedef[0], 1, attr);
	ut_wordfill (&w->wbuf[1], w->xwidth-2, (attr<<8)+w->wdef.framedef[1]);
	ut_bufferwrite(&w->wbuf[w->xwidth-1], &w->wdef.framedef[2], 1, attr);

	if ((tm = w->tmode-TOPLEFTTITLE) <= 2)
		{
		if ((tl = strlen(w->title)) > w->xwidth-2)
			tl = w->xwidth-2;
		switch (tm)
			{
			case 1:
				pos = (w->xwidth-tl)/2;
				break;
			case 2:
				pos = w->xwidth-tl-1;
				break;
			default:
				pos = 1;
			}
		ut_bufferwrite(&w->wbuf[pos], w->title, tl, attr);
		}
	wn_updatescreen(w,w->wdef.xleft,w->wdef.xtop,w->xwidth);

	/* Sides */
	pos = w->xwidth;
	for (i=1; i<=w->depth; i++)
		{
		ut_bufferwrite(&w->wbuf[pos],&w->wdef.framedef[3],1,attr);
      wn_updatescreen(w,w->wdef.xleft,w->wdef.xtop+i,1);
		pos += w->xwidth;
		ut_bufferwrite(&w->wbuf[pos-1],&w->wdef.framedef[4],1,attr);
      wn_updatescreen(w,w->wdef.xright,w->wdef.xtop+i,1);
		}

	pos = (w->xdepth-1)*w->xwidth;
	ut_bufferwrite(&w->wbuf[pos], &w->wdef.framedef[5], 1, attr);
	ut_wordfill (&w->wbuf[pos+1], w->xwidth-2, (attr<<8)+w->wdef.framedef[6]);
	ut_bufferwrite(&w->wbuf[pos+w->xwidth-1], &w->wdef.framedef[7], 1, attr);

	if ((tm = w->tmode-BOTTOMLEFTTITLE) <= 2)
		{
		if ((tl = strlen(w->title)) > w->xwidth-2)
			tl = w->xwidth-2;
		switch (tm)
			{
			case 1:
				pos += (w->xwidth-tl)/2;
				break;
			case 2:
				pos += w->xwidth-tl-1;
				break;
			default:
				pos += 1;
			}
		ut_bufferwrite(&w->wbuf[pos], w->title, tl, attr);
		}
	wn_updatescreen(w, w->wdef.xleft, w->wdef.xbottom, w->xwidth);
}

/*---------------------------------------------------------------------------

*/
static void wn_disposetitle (WINTYPE w)
{
	if (w->tmode != NOTITLE)
		{
		ut_memfree(w->title);
		w->tmode = NOTITLE;
		}
}

/*---------------------------------------------------------------------------

	Sets the current X Y position of the pane currently being used
*/
static void wn_igotoxy (WINTYPE w, int x, int y)
{
	w->curx = x;
	w->cury = y;
	if (w == cursorstack)
		wn_resetcursor();
}


/*---------------------------------------------------------------------------

*/
static void wn_deleteline (register WINTYPE w, UINT y)
{
UINT r,p;

	if (wn_valid(w))
		{
		p = w->left - w->wdef.xleft +
			 (w->top - w->wdef.xtop + y-1) * w->xwidth;
		for (r=y; r<w->depth; r++)
			{
			ut_wordmove(&w->wbuf[p], &w->wbuf[p+w->xwidth], w->width);
			p += w->xwidth;
			}
		ut_wordfill (&w->wbuf[p],w->width,WN_CHATTR(p));
		wn_redrawsection (w,w->left,w->top + y-1,w->right,w->bottom );
		}
}

/*---------------------------------------------------------------------------

	Opens a window on the screen ready for use
*/
WINTYPE wn_open (WINDEF *wd)
{
register WINTYPE w;
UINT size, min;

	if ((w = ut_memalloc(sizeof(WINDESC))) == NULL)
		return NULL;

	memcpy(&w->wdef,wd,sizeof(WINDEF));

	if (w->wdef.xright > ScrWidth)
		w->wdef.xright = ScrWidth;
	if (w->wdef.xbottom > ScrDepth)
		w->wdef.xbottom = ScrDepth;

	min = w->wdef.isframe ? 2 : 0;

	if (w->wdef.xleft+min > w->wdef.xright)
		w->wdef.xright = w->wdef.xleft+min;
	if (w->wdef.xtop+min > w->wdef.xbottom)
		w->wdef.xbottom = w->wdef.xtop+min;

	w->left   = w->wdef.xleft;
	w->top    = w->wdef.xtop;
	w->right  = w->wdef.xright;
	w->bottom = w->wdef.xbottom;
	w->xwidth = w->width  = w->wdef.xright-w->wdef.xleft+1;
	w->xdepth = w->depth  = w->wdef.xbottom-w->wdef.xtop+1;

	w->curx   = 1;
	w->cury   = 1;

	w->next         = NULL;
	w->cursor_chain = NULL;
	w->wbuf         = NULL;
	w->user_record  = NULL;

	w->tmode = NOTITLE;
	w->guard = WINDOW_GUARD;
	w->title = NULL;

	wn_clipframe(w);

	size = w->xwidth*w->xdepth;
	if ((w->wbuf = ut_memalloc(size*2)) == NULL)
		{
		ut_memfree(w);
		return NULL;
		}

	ut_wordfill(&w->wbuf[0],size,WN_CHATTR(w));

	if (w->wdef.isframe)
		wn_drawframe(w);

	if (w->wdef.hidden)
		wn_use(w);
	else
		wn_putontop(w);
return w;
}

/*---------------------------------------------------------------------------

	Causes all subsequent output to appear in the specified Window
	NB does not have to be wn_top Window (or in fact on the screen at all)
	UseList is the MRU window
*/
WINTYPE wn_use (WINTYPE w)
{
UseListPtr up, u;

	if (wn_valid(w))
		{
		if (UseList == NULL)
			{
			/* create the list */
			if ((UseList = ut_memalloc(sizeof(USELIST))) != NULL)
				{
				UseList->next = NULL;
				UseList->wind = w;
				return w;
				}
			}
		else
			{
			if (UseList->wind == w)
				return w;	/* already on alist */

			up = u = UseList;
			u = u->next;
			while (u != NULL)
				{
				if (u->wind == w)
					{
					up->next = u->next;		/* move to begin of list */
					u->next = UseList;
					UseList = u;
					return w;
					}
				up = u;
				u = u->next;
				}

			/* add to list at the begin */
			if ((u = ut_memalloc(sizeof(USELIST))) != NULL)
				{
				u->next = UseList;
				UseList = u;
				UseList->wind = w;
				return w;
				}
			}
		}
return NULL;
}


/*---------------------------------------------------------------------------

	Puts the specified window on the top of the window stack
	Ensuring that it is fully visible.
	If this results in other windows becoming obscured then a buffer
	is allocated for each of these windows.
	All otherwise undirected output (ie with no Use) will appear
	within this window.
*/
void wn_putontop (WINTYPE w)
{
	if (wn_valid(w))
		{
		if (w != windowstack)
			{
			wn_takeoffstack(w);
			w->next = windowstack;
			windowstack = w;
			w->wdef.hidden = CLR;
			wn_redrawwindow(w);
			}
		wn_use(w);
		if (w->wdef.iscursor)
			wn_cursoron();
		else
			wn_resetcursor();
		}
}

/*---------------------------------------------------------------------------

	Puts window W beneath window WA
*/
void wn_putbeneath (WINTYPE w, WINTYPE wa)
{
WINTYPE p;

	if (wn_valid(wa))
		{
		if (!wa->wdef.hidden)
			{
			wn_hide(w);
			p = wa->next;
			wa->next = w;
			w->next = p;
			w->wdef.hidden = CLR;
			wn_redrawwindow(w);
			wn_resetcursor();
			}
		}
}

/*---------------------------------------------------------------------------

	Updates the Window buffer from the screen
*/
void wn_snapshot (void)
{
register WINTYPE w;
UINT y, p;

	w = wn_curwin();
	p = 0;
	for (y=w->wdef.xtop; y<=w->wdef.xbottom; y++)
		{
		sc_scr2buf(w->wdef.xleft,y,&w->wbuf[p],w->xwidth);
		p+=w->xwidth;
		}
}

/*---------------------------------------------------------------------------

	Removes window from the Window stack and also the screen
	Placing the windows contents in a buffer for possible re-display later
	Uncovers obscured windows
*/
void wn_hide (WINTYPE w)
{
WINTYPE p;

	if (wn_valid(w))
		{
		if (!w->wdef.hidden)
			{
			p = w->next;
			wn_takeoffstack(w);
			wn_displaybeneath(w, p);
			if (w->wdef.iscursor)
				wn_unlinkcursor(w);
			w->wdef.hidden = SET;
			}
		wn_resetcursor();
		}
}

/*---------------------------------------------------------------------------

	removes the specified window from the screen
	deletes window descriptor and
	de-allocates any buffers previously allocated.
*/
WINTYPE wn_close (WINTYPE w)
{
UseListPtr u,up;

	if (wn_valid(w))
		{
		wn_hide(w);
		wn_disposetitle(w);			/* dispose window title */
		if (w->wbuf != NULL)
			ut_memfree(w->wbuf);		/* dispose window buffer */

		if (UseList != NULL)
			{
			u = UseList;
			if (u->wind == w)
				{
				UseList = u->next;
				u->wind = NULL;
				ut_memfree(u);			/* dispose window use list */
				}
			else
				{
				up = u;
				u = u->next;
				while (u != NULL)
					{
					if (u->wind == w)
						{
						up->next = u->next;
						u->wind = NULL;
						ut_memfree(u);			/* dispose window use list */
						}
					up = u;
					u = u->next;
					}
				}
			}

		w->guard = 0;
		ut_memfree(w);					/* dispose window desc */
		w = NULL;
		}
return NULL;
}

/*---------------------------------------------------------------------------

*/
WINTYPE wn_curwin (void)
{
	if (UseList != NULL)
		return UseList->wind;

return windowstack;
}


/*---------------------------------------------------------------------------

*/
WINTYPE wn_screen (void)
{

return FullScreen;
}


/*---------------------------------------------------------------------------

	Returns The current window being used for output for this process
	If no window assigned by Use then returns wn_top
*/
WINTYPE wn_used (void)
{
return wn_curwin();
}

/*---------------------------------------------------------------------------

	Returns The current top Window
*/
WINTYPE wn_top (void)
{
return windowstack;
}

/*---------------------------------------------------------------------------

Returns the window displayed at the absolute position X,Y
*/
WINTYPE wn_winatpos (UINT x, UINT y)
{
register WINTYPE w;

	w = windowstack;
	while (w != NULL)
		{
		if (y >= w->wdef.xtop && y <= w->wdef.xbottom &&
			x >= w->wdef.xleft && x <= w->wdef.xright)
      w = w->next;
		}
return w;
}

/*---------------------------------------------------------------------------

	Returns if the specified window is obscured at the specified position
*/
int wn_obscuredat (WINTYPE w, UINT x, UINT y)
{
register WINTYPE p;
int b;

	if (wn_valid(w))
		{
		if (w->wdef.hidden)
			b = SET;
		else
			{
			x += w->left-1;
			y += w->top-1;
			p = windowstack;
			b = CLR;
			while (p != w)
				{
				if (y >= p->wdef.xtop && y <= p->wdef.xbottom &&
					x >= p->wdef.xleft && x <= p->wdef.xright)
					{
					b = SET;
					break;
					}
				p = p->next;
				}
			}
		}
return b;
}



/*---------------------------------------------------------------------------

   Put a frame around the specified window
   With title String and border definition (see above)
   having specified fore/background colours for the border
*/
void wn_setframe (WINTYPE w, char *frame, int fore, int back)
{
	if (wn_valid(w))
		{
		memcpy(w->wdef.framedef,frame,sizeof(framestr));
		w->wdef.Ffore = fore;
		w->wdef.Fback = back;
		wn_drawframe(w);
		}
}

/*---------------------------------------------------------------------------

	Changes the position of the top window
	The contents of the window will be moved with it
*/
void wn_move (int dx, int dy)
{
register WINTYPE w;
WINTYPE wsave;

	if (wn_valid(w = wn_top()))
		{
		if (dx > 0)
			{
			if (w->wdef.xright + dx > ScrWidth)
				dx = 0;
			}
		else if (dx < 0)
			{
			if (w->wdef.xleft + dx > ScrWidth)
				dx = 0;
			}

		if (dy > 0)
			{
			if (w->wdef.xbottom + dy > ScrDepth)
				dy = 0;
			}
		else if (dy < 0)
			{
			if (w->wdef.xtop + dy > ScrDepth)
				dy = 0;
			}

		if (dx != 0 || dy != 0)		/* any move */
			{
			wsave = wn_curwin();
			wn_hide(w);

			w->wdef.xleft   += dx;
			w->wdef.xtop    += dy;
			w->wdef.xright	 += dx;
			w->wdef.xbottom += dy;
			w->left    += dx;
			w->top     += dy;
			w->right	  += dx;
			w->bottom  += dy;

			wn_putontop(w);
			wn_use(wsave);		/* restore used window */
			}
		}
}

/*---------------------------------------------------------------------------

	clears the current window
*/
void wn_clear (void)
{
UINT r,p;
WINTYPE w;

	w = wn_curwin();
	p = w->left - w->wdef.xleft +
		(w->top - w->wdef.xtop) * w->xwidth;
	for (r=0; r<w->depth; r++)
		{
		ut_wordfill (&w->wbuf[p], w->width, WN_CHATTR(p));
		p += w->xwidth;
		}
	wn_redrawsection(w, w->left, w->top, w->right, w->bottom);
	wn_igotoxy(w, 1, 1);
}

/*---------------------------------------------------------------------------

	clears from the cursor to the end of line
*/
void wn_clreol (void)
{
WINTYPE w;

	w = wn_curwin();
	ut_wordfill (&w->wbuf[w->left-w->wdef.xleft + w->curx-1 +
					(w->top-w->wdef.xtop + w->cury-1)*w->xwidth],
					w->width-w->curx+1, WN_CHATTR(p));
	wn_redrawsection(w, w->left - 1 + w->curx, w->top - 1 + w->cury,
								w->right, w->top-1+w->cury);
}

/*---------------------------------------------------------------------------

*/
void wn_wputch (register WINTYPE w, char c)
{
	switch (c)
		{
		case '\f':
			wn_clear();
			wn_igotoxy(w,1,1);
			break;

		case '\r':
			wn_igotoxy(w,1,w->cury);
			break;

		case '\n':
			if (w->cury == w->depth)
				wn_deleteline(w,1);
			else
				wn_igotoxy(w,w->curx,w->cury+1);
			break;

		case '\b':
			if (w->curx > 1)
				{
				wn_igotoxy(w, w->curx-1, w->cury);
				wn_wputch(w,' ');
				wn_igotoxy(w, w->curx-1, w->cury);
				}
			break;

		case '\a':
			sc_bell();
			break;

		default:
			if (w->curx <= w->width)
				{
				ut_bufferwrite(&w->wbuf[w->left-w->wdef.xleft+w->curx-1 +
							(w->top-w->wdef.xtop+w->cury-1)*w->xwidth],&c,1,
							w->wdef.Wfore + w->wdef.Wback*0x10);
				wn_updatescreen(w,w->left+w->curx-1,w->top+w->cury-1,1);
				if (w->curx != w->width || !w->wdef.iswrap)
					wn_igotoxy(w, w->curx+1, w->cury);
				else
					{
					if (w->cury == w->depth)
						{
						wn_deleteline(w, 1);
						wn_igotoxy(w, 1, w->cury);
						}
					else
						wn_igotoxy(w, 1, w->cury+1);
					}
				}
		}
}

/*---------------------------------------------------------------------------

*/
void wn_putch (char c)
{
	wn_wputch(wn_curwin(),c);
}

/*---------------------------------------------------------------------------

*/
int wn_wputs (WINTYPE w, char *msg)
{
int i = 0;
	while(*msg)
		{
		wn_wputch(w,*msg++);
		i++;
		}
return i;
}

/*---------------------------------------------------------------------------

*/
int wn_puts (char *msg)
{

return wn_wputs(wn_curwin(),msg);
}

/*---------------------------------------------------------------------------

*/
int wn_wprintf (WINTYPE w, char *fmt, ...)
{
va_list  argptr;
int cnt;

	va_start(argptr,fmt);

	cnt = vsprintf((char*)PrintfBuffer, fmt, argptr);

	wn_wputs(w,(char*)PrintfBuffer);

	va_end(argptr);

return cnt;
}


/*---------------------------------------------------------------------------

*/
int wn_printf (char *fmt, ...)
{
va_list  argptr;
int cnt;

	va_start(argptr,fmt);

	cnt = vsprintf((char*)PrintfBuffer, fmt, argptr);

	wn_puts((char*)PrintfBuffer);

	va_end(argptr);

return cnt;
}

/*---------------------------------------------------------------------------

	writes directly to current window at the specified X,Y coordinates
	with no check for special (ie control) chars or eol wrap
*/
void wn_directwrite ( UINT x, UINT y, char *str, UINT len, UINT attr)
{
WINTYPE w;
UINT px,py;

	if (len > 0)
		{
		w = wn_curwin();
		px = w->wdef.xleft + x;
		py = w->wdef.xtop + y;
		if (len+px > w->wdef.xright+1)
			len = w->wdef.xright-px+1;
		if (len > 0 && py <= w->wdef.xbottom)
			{
			ut_bufferwrite(&w->wbuf[x+y*w->xwidth],str,len,attr);
   	   wn_updatescreen(w,px,py,len);
			}
		}
}

/*---------------------------------------------------------------------------

	writes directly to current window at the specified X,Y coordinates
	with no check for special (ie control) chars or eol wrap
*/
void wn_directwritestr ( UINT x, UINT y, char *str, UINT len)
{
WINTYPE w;
UINT px,py;

	if (len > 0)
		{
		w = wn_curwin();
		px = w->wdef.xleft + x;
		py = w->wdef.xtop + y;
		if (len+px > w->wdef.xright+1)
			len = w->wdef.xright-px+1;
		if (len > 0 && py <= w->wdef.xbottom)
			{
			ut_bufferwritestr(&w->wbuf[x+y*w->xwidth],str,len);
			wn_updatescreen(w,px,py,len);
			}
		}
}

/*---------------------------------------------------------------------------

	writes directly to current window at the specified X,Y coordinates
	with no check for special (ie control) chars or eol wrap
*/
void wn_directwriteattr (UINT x, UINT y, UINT len, UINT attr)
{
WINTYPE w;
UINT px,py;

	if (len > 0)
		{
		w = wn_curwin();
		px = w->wdef.xleft + x;
		py = w->wdef.xtop + y;
		if (len+px > w->wdef.xright+1)
			len = w->wdef.xright-px+1;
		if (len > 0 && py <= w->wdef.xbottom)
			{
			ut_bufferwriteattr(&w->wbuf[x+y*w->xwidth],len,attr);
   	   wn_updatescreen(w,px,py,len);
			}
		}
}


/*---------------------------------------------------------------------------

*/
void wn_gotoxy (UINT x, UINT y)
{
WINTYPE w;

  w = wn_curwin();
  wn_clipxy(w,&x,&y);
  wn_igotoxy(w,x,y);
}

/*---------------------------------------------------------------------------

*/
int wn_wherex (void)
{
WINTYPE w;

  w = wn_curwin();
return w->curx;
}

/*---------------------------------------------------------------------------

*/
int wn_wherey (void)
{
WINTYPE w;

  w = wn_curwin();
return w->cury;
}

/*---------------------------------------------------------------------------

*/
void wn_convertcoords (WINTYPE w, UINT x, UINT y, UINT *xo, UINT *yo)
{
	wn_valid(w);
	*xo = x + w->left - 1;
	*yo = y + w->top - 1;
}


/*---------------------------------------------------------------------------

*/
void wn_insline (void)
{
register WINTYPE w;
UINT r,p;

	w = wn_curwin();
	p = w->left - w->wdef.xleft +
		(w->top - w->wdef.xtop + w->depth - 1) * w->xwidth;
	for (r=w->cury; r<w->depth; r++)
		{
      p -= w->xwidth;
		ut_wordmove(&w->wbuf[p+w->xwidth], &w->wbuf[p], w->width);
		}
	ut_wordfill (&w->wbuf[p],w->width, WN_CHATTR(p));
	wn_redrawsection(w, w->left, w->cury + w->top - 1, w->right, w->bottom);
}

/*---------------------------------------------------------------------------

*/
void wn_delline (void)
{
WINTYPE w;

	w = wn_curwin();
	wn_deleteline(w, w->cury);
}

/*---------------------------------------------------------------------------

*/
void wn_setcolor (int c)
{
WINTYPE w;

	w = wn_curwin();
	w->wdef.Wfore = c;
}

/*---------------------------------------------------------------------------

*/
void wn_setbkcolor (int  c)
{
WINTYPE w;

	w = wn_curwin();
	w->wdef.Wback = c;
}


/*---------------------------------------------------------------------------

*/
void wn_setwrap (int  on)
{
WINTYPE w;

	w = wn_curwin();
	w->wdef.iswrap = on;
}

/*---------------------------------------------------------------------------

	gets information for specified window
*/
WINDEF *wn_getdef (WINTYPE w)
{
return &w->wdef;
}


/*---------------------------------------------------------------------------

	updates the window title within the window frame,
	positioning it in the position defined by the title mode
*/
void wn_settitle (WINTYPE w, char *Newtitle, TITLEMODE Mode)
{
	wn_valid(w);
	wn_disposetitle(w);
	if (Mode != NOTITLE)
		{
		if ((w->title = ut_memalloc(strlen(Newtitle)+1)) != NULL)
			strcpy(w->title,Newtitle);
		}
	w->tmode = Mode;
	wn_drawframe(w);
}

/*---------------------------------------------------------------------------

*/
int wn_video (void)
{

return sc_video();
}

/*---------------------------------------------------------------------------

*/
int wn_init (WINDEF *InitDef, int ClearOnEntry)
{
UINT crow,ccol;
UINT p,y;

	if (!wn_initflag)
		{
		windowstack = NULL;
		cursorstack = NULL;
		UseList     = NULL;

		switch(sc_video())
			{
			case 3:		/* mono */
				sc_init(7,CLR);
				break;

			case 2:		/* ega & vga */
				sc_init(3,CLR);
				sc_egablibkbit(CLR);
				break;

			case 1:		/* cga & other */
			default:
				sc_init(3,SET);
				break;
			}

		CursorLines = sc_getcursor(&crow,&ccol);

		ScrSize = sc_getscrsize(&ScrWidth, &ScrDepth);
		if ((ScrBuffer = ut_memalloc(ScrSize)) == NULL)
			{
			sc_exit();
			return CW_FAIL;
			}

		/* save entry screen */
		p = 0;
		for (y=0; y<=ScrDepth; y++)
			{
			sc_scr2buf(0, y, &ScrBuffer[p], ScrWidth+1);
			p += ScrWidth+1;
			}

		if ((PrintfBuffer = ut_memalloc(PrintfBufferSize)) == NULL)
			{
			sc_exit();
			return CW_FAIL;
			}

		InitDef->xleft   = 0;
		InitDef->xtop    = 0;
		InitDef->xright  = ScrWidth;
		InitDef->xbottom = ScrDepth;

		if (InitDef->Wfore < 0)
			InitDef->Wfore = BLUE;
		if (InitDef->Wback < 0)
			InitDef->Wback = LIGHTGRAY;

		if (InitDef->fillch == 0)
			InitDef->fillch  = ClearOnEntry ? '' : ' ';

		InitDef->hidden = ClearOnEntry ? CLR : SET;

		if (InitDef->Wfore < 0)
			InitDef->Ffore = RED;
		if (InitDef->Wback < 0)
			InitDef->Fback = LIGHTGRAY;

		if (InitDef->isframe != 0 && InitDef->framedef[0] == 0)
			memcpy(InitDef->framedef,SSSSFrame,sizeof(framestr));

		if ((wn_valid(FullScreen = wn_open(InitDef))) == NULL)
			{
			sc_exit();
			return CW_FAIL;
			}

		if (! ClearOnEntry)
			{
			wn_snapshot();
			wn_putontop(FullScreen);
			wn_gotoxy(crow+1,ccol+1);
			}
		}
return CW_OKAY;
}

/*---------------------------------------------------------------------------

*/
void wn_exit (void)
{
UINT p, y;

	sc_exit();

	while (UseList != NULL)
		wn_close(UseList->wind);

	if (PrintfBuffer)
		ut_memfree(PrintfBuffer);

	if (ScrBuffer)
		{
		p = 0;
		for (y=0; y<=ScrDepth; y++)
			{
			sc_buf2scr(0, y, &ScrBuffer[p], ScrWidth+1);
			p += ScrWidth+1;
			}
		ut_memfree(ScrBuffer);
		}

}

