/*---------------------------------------------------------------------------
	MENULIB.C

	Version 1.00 (C)Copyright Andrew Trzeciak. 1993
*/
#include <string.h>
#include <ctype.h>

#include "winlib.h"
#include "menulib.h"
#include "strlib.h"
#include "scrlib.h"
#include "bioskeys.h"
#include "keylib.h"
#include "utlib.h"

#define MENUGUARD 0xABCD			/* Signature word for menu descriptor */
#define RANGE(a,b,c)		(a < b && a > c)
#define MaxDftKey		SIZETAB(DefaultKeyMap)

typedef unsigned int UINT;
typedef unsigned char UCHAR;

static MenuListPtr MenuList = NULL;

static KEYDEF DefaultKeyMap[] =
	{
	{KB_CH_N_UP,         KB_SC_N_UP,             0, MN_UP,       0},
	{KB_CH_N_DOWN,       KB_SC_N_DOWN,           0, MN_DOWN,     0},
	{KB_CH_N_LEFT,       KB_SC_N_LEFT,           0, MN_LEFT,     0},
	{KB_CH_N_RIGHT,      KB_SC_N_RIGHT,          0, MN_RIGHT,    0},
	{KB_CH_N_HOME,       KB_SC_N_HOME,           0, MN_FIRST,    0},
	{KB_CH_N_END,        KB_SC_N_END,            0, MN_LAST,     0},
	{KB_CH_N_PGUP,       KB_SC_N_PGUP,           0, MN_FIRST,    0},
	{KB_CH_N_PGDN,       KB_SC_N_PGDN,           0, MN_LAST,     0},
	{KB_CH_N_GRAY_UP,    KB_SC_N_GRAY_UP,        0, MN_UP,       0},
	{KB_CH_N_GRAY_DOWN,  KB_SC_N_GRAY_DOWN,      0, MN_DOWN,     0},
	{KB_CH_N_GRAY_LEFT,  KB_SC_N_GRAY_LEFT,      0, MN_LEFT,     0},
	{KB_CH_N_GRAY_RIGHT, KB_SC_N_GRAY_RIGHT,     0, MN_RIGHT,    0},
	{KB_CH_N_GRAY_HOME,  KB_SC_N_GRAY_HOME,      0, MN_FIRST,    0},
	{KB_CH_N_GRAY_END,   KB_SC_N_GRAY_END,       0, MN_LAST,     0},
	{KB_CH_N_GRAY_PGUP,  KB_SC_N_GRAY_PGUP,      0, MN_FIRST,    0},
	{KB_CH_N_GRAY_PGDN,  KB_SC_N_GRAY_PGDN,      0, MN_LAST,     0},
	{KB_CH_N_TAB,        KB_SC_N_TAB,            0, MN_NEXT,     0},
	{KB_CH_S_TAB,        KB_SC_S_TAB,            0, MN_PREVIOUS, 0},
	{KB_CH_N_ENTER,      KB_SC_N_ENTER,          0, MN_NOMOVE,   MN_TRANSMIT},
	{KB_CH_N_PAD_ENTER,  KB_SC_N_PAD_ENTER,      0, MN_NOMOVE,   MN_TRANSMIT},
	{KB_CH_N_ESC,        KB_SC_N_ESC,            0, MN_NOMOVE,   MN_ABORT | MN_REMOVE}
	};

/*---------------------------------------------------------------------------

*/
static MENUTYPE mn_valid (MENUTYPE m)
{
extern int cwin_error;

	if (m == NULL || m->guard != MENUGUARD)
		{
		cwin_error = CW_MENUHANDLE_ERR;
		ut_fatalerror();
		}
return m;
}

/*---------------------------------------------------------------------------

*/
static KEYDEFLIST *mn_addkey (KEYDEFLIST * head, int  ch, int sc, int pos)
{
KEYDEFLIST *p;

	if ((p = ut_memalloc(sizeof(KEYDEFLIST))) != NULL)
		{
		p->defkey.ch = ch;
		p->defkey.sc = sc;
		p->defkey.itempos = pos;
		p->defkey.barmove = MN_GOTO;
		p->defkey.action  = MN_TRANSMIT;

		p->nextkey = head;
		head = p;
		}
return head;
}

/*---------------------------------------------------------------------------

*/
static KEYDEFLIST *mn_keylist (ITEM *itemtab, int itemtabsize)
{
KEYDEFLIST *PKey;
ITEM *ip;
int ch,j;

	PKey = NULL;
	for (j=itemtabsize-1; j>=0; j--)
		{
		ip = &itemtab[j];
		switch (ip->itype)
			{
			case MN_FIRSTCHAR:
				ip->ipos = strnotblank(ip->itemstr); /* find first char */
			case MN_USERDEF:
				ch = ip->itemstr[ip->ipos];			/* get highlighted char */
				if (isalnum(ch))
					{
					if (islower(ch))
						{
						PKey = mn_addkey(PKey,ch,kb_ch2sc(ch),j);
						ch = _toupper(ch);
						}
					else if (isupper(ch))
						{
						PKey = mn_addkey(PKey,ch,kb_ch2sc(ch),j);
						ch = _tolower(ch);
						}
					PKey = mn_addkey(PKey,ch,kb_ch2sc(ch),j);
					}
				break;
			}
		}
return PKey;
}

/*---------------------------------------------------------------------------

	Causes all subsequent output to appear in the specified Window
	NB does not have to be wn_top Window (or in fact on the screen at all)
	UseList is the MRU window
*/
static MENUTYPE mn_putonlist (MENUTYPE mt)
{
MenuListPtr m;

	if (mn_valid(mt))
		{
		if ((m = ut_memalloc(sizeof(MENULIST))) != NULL)
			{
			m->next = MenuList != NULL ? MenuList : NULL;
			MenuList = m;
			MenuList->menu = mt;
			return mt;
			}
		}
return NULL;
}

/*---------------------------------------------------------------------------

*/
MENUTYPE mn_create (MENUDEF *md, ITEM itemtab[], int sizeitemtab)
{
MENUTYPE mt;
WINDEF wd;

	if ((mt = ut_memalloc(sizeof(MENUDESC))) == NULL)
		return NULL;

	wd.xleft     = md->xleft;
	wd.xtop      = md->xtop;
	wd.xright    = md->xright;
	wd.xbottom   = md->xbottom;
	wd.Wfore     = md->Mfore;
	wd.Wback     = md->Mback;
	wd.fillch    = md->fillch;
	wd.iscursor  = CLR;
	wd.iswrap    = CLR;
	wd.hidden    = SET;
	wd.isframe   = md->isframe;
	memmove(wd.framedef,md->framedef,sizeof(framestr));
	wd.Ffore     = md->Ffore;
	wd.Fback     = md->Fback;

	mt->guard     = MENUGUARD;
	if ((mt->winptr = wn_open(&wd)) == NULL)
		{
		ut_memfree(mt);
		return NULL;
		}

	mt->curpos     = 0;
	mt->lastitemno = sizeitemtab;
	mt->itemptr    = itemtab;
	mt->Nfore      = md->Nfore;
	mt->Nback      = md->Nback;
	mt->Bfore      = md->Bfore;
	mt->Bback      = md->Bback;
	mt->Hfore      = md->Hfore;
	mt->Hback      = md->Hback;

	mt->islotus   = CLR;
	mt->lotptr    = NULL;
	mt->lotwinptr = NULL;

	mt->KeyRef = NULL;
	mt->lastkeyno = CLR;

	mt->Keys = mn_keylist(itemtab,sizeitemtab);

	if (mn_putonlist(mt) == NULL)
		{
		wn_close(mt->winptr);
		ut_memfree(mt);
		return NULL;
		}

return mt;
}

/*---------------------------------------------------------------------------

*/
WINTYPE mn_setlotus (MENUTYPE m, LOTUSDEF *ld, LOTITEM lottab[])
{
WINDEF wd;

	wd.xleft     = ld->xleft;
	wd.xtop      = ld->xtop;
	wd.xright    = ld->xleft+ld->width-1;
	wd.xbottom   = ld->xtop + (ld->isframe ? 2 : 0);
	wd.Wfore     = ld->Lfore;
	wd.Wback     = ld->Lback;
	wd.fillch    = ld->fillch;
	wd.iscursor  = CLR;
	wd.iswrap    = CLR;
	wd.hidden    = SET;
	wd.isframe   = ld->isframe;
	memmove(wd.framedef,ld->framedef,sizeof(framestr));
	wd.Ffore     = ld->Ffore;
	wd.Fback     = ld->Fback;
	if ((m->lotwinptr = wn_open(&wd)) != NULL)
		{
		m->islotus = SET;
		m->lotptr  = lottab;
		m->lleft   = ld->left;
		m->ltop    = ld->top;

		if (!ld->isframe)
			{
			if (m->lleft > 0) m->lleft--;
			if (m->ltop > 0)  m->ltop--;
			}
		}
return m->lotwinptr;
}

/*---------------------------------------------------------------------------

*/
void mn_keys (MENUTYPE m, KEYDEF kd[], int size)
{
	if (mn_valid(m))
		{
		m->KeyRef     = kd;
		m->lastkeyno  = size;
		}
}

/*---------------------------------------------------------------------------

*/
void mn_settitle (MENUTYPE m, char *Title, TITLEMODE mode)
{
	if (mn_valid(m))
		wn_settitle(m->winptr,Title,mode);
}

/*---------------------------------------------------------------------------

*/
void mn_setframe (MENUTYPE m, char *frame, UINT fore, UINT back)
{
	if (mn_valid(m))
		wn_setframe(m->winptr,frame,fore,back);
}


/*---------------------------------------------------------------------------

*/
MENUTYPE mn_destroy(MENUTYPE m)
{
KEYDEFLIST *PKey;
KEYDEFLIST *QKey;
MenuListPtr mc,mp;

	if (mn_valid(m))
		{
		if (m->islotus)
			wn_close(m->lotwinptr);
		wn_close(m->winptr);

		PKey = m->Keys;
		while (PKey != NULL)
			{
			QKey = PKey;
			PKey = PKey->nextkey;
			ut_memfree(QKey);
			}

		if (MenuList != NULL)
			{
			mc = MenuList;
			if (mc->menu == m)
				{
				MenuList = mc->next;
				mc->menu = NULL;
				ut_memfree(mc);			/* dispose menu use list */
				}
			else
				{
				mp = mc;
				mc = mc->next;
				while (mc != NULL)
					{
					if (mc->menu == m)
						{
						mp->next = mc->next;
						mc->menu = NULL;
						ut_memfree(mc);			/* dispose menu use list */
						}
					mp = mc;
					mc = mc->next;
					}
				}
			}
		m->guard = 0;
		ut_memfree(m);						/* dispose menu desc */
		m = NULL;
		}	

return m;
}

/*---------------------------------------------------------------------------

*/
void mn_remove (MENUTYPE m)
{
	if (mn_valid(m))
		{
		wn_hide(m->winptr);
		if (m->islotus)
			wn_hide(m->lotwinptr);
		}
}


/*---------------------------------------------------------------------------

*/
static void mn_display_item (MENUTYPE m, UINT No, int ishibar)
{
ITEM *ip = &m->itemptr[No];
UINT x,y,attr;
			
	wn_gotoxy(x = ip->x, y = ip->y);
	attr = ishibar ? (UINT)m->Bback*0x10 + (UINT)m->Bfore		/* Bar */
						: (UINT)m->Nback*0x10 + (UINT)m->Nfore;	/* Text */
	if (! m->winptr->wdef.isframe)
		{
		if (x > 0)
			x--;
		if (y > 0)
			y--;
		}
	if (ip->itype == MN_EDITITEM)
		{
		if (ip->ipos > 0)
			wn_directwrite(x,y,ip->itemstr,ip->ipos,attr);
		}
	else
		{
		wn_directwrite(x,y,ip->itemstr,strlen(ip->itemstr),attr);
		if (ip->ipos >= 0)
			wn_directwriteattr(x+ip->ipos,y,1,(UINT)m->Hback*0x10+(UINT)m->Hfore);
		}
	if (m->islotus)
		{
		wn_use(m->lotwinptr);
		wn_clear();
		wn_directwritestr(m->lleft,m->ltop,m->lotptr[m->curpos].str,strlen(m->lotptr[m->curpos].str));
      wn_use(m->winptr);
		}
}

/*---------------------------------------------------------------------------

*/
static KEYDEF *mn_searchkey (MENUTYPE m, int ch, int sc)
{
KEYDEF *kk;
KEYDEFLIST *kp;
int i;

	/* User defined key */
	for(i=0; i<m->lastkeyno; i++)
		{
		kk = &m->KeyRef[i];
		if (kk->ch == ch && kk->sc == sc)
			return kk;
		}

	/* Highlited characters */
	kp = m->Keys;
	while (kp != NULL)
		{
		if (kp->defkey.ch == ch && kp->defkey.sc == sc)
			return &kp->defkey;
		kp = kp->nextkey;
		}

	/* Default keys */
	for(i=0; i<MaxDftKey; i++)
		{
		if (DefaultKeyMap[i].ch == ch && DefaultKeyMap[i].sc == sc)
			return &DefaultKeyMap[i];
		}
return NULL;
}

/*---------------------------------------------------------------------------

*/
static KEYDEF *mn_searchuserkey (MENUTYPE m, int ch, int sc)
{
KEYDEF *kk;
int i;

	/* User defined key */
	for(i=0; i<m->lastkeyno; i++)
		{
		kk = &m->KeyRef[i];
		if (kk->ch == ch && kk->sc == sc)
			return kk;
		}
return NULL;
}

/*---------------------------------------------------------------------------

*/
static int mn_nextpos (MENUTYPE m, MOTION Move)
{
int i,pos;
UINT x,y;
UINT curx,cury;

	curx = m->itemptr[m->curpos].x;
	cury = m->itemptr[m->curpos].y;
	pos  = -1;
	for (i=0; i<m->lastitemno; i++)
		{
		x = m->itemptr[i].x;
		y = m->itemptr[i].y;
		switch (Move)
			{
			case MN_UP:
				if (x == curx &&
					(((pos < 0 && y < cury) || RANGE(y,cury,m->itemptr[pos].y))))
					pos = i;
				break;

			case MN_DOWN:
				if (x == curx &&
					(((pos < 0 && y > cury) || RANGE(y,m->itemptr[pos].y,cury))))
					pos = i;
				break;

			case MN_LEFT:
				if (y == cury &&
					(((pos < 0 && x < curx) || RANGE(x,curx,m->itemptr[pos].x))))
					pos = i;
				break;

			case MN_RIGHT:
				if (y == cury &&
					(((pos < 0 && x > curx) || RANGE(x,m->itemptr[pos].x,curx))))
					pos = i;
				break;
			}
		}
return pos < 0 ? m->curpos : pos;
}

/*---------------------------------------------------------------------------

	Inputs a string, using Window module, allowing normal line editing
*/
static void mn_edit_item (MENUTYPE m, UINT *startpos, int *ch, int *sc)
{
static int Ins = CLR;
ITEM *ip=&m->itemptr[m->curpos];
int pos, done, ret;
char *editstr;

	if (ip->ipos > 0 && (editstr = ut_memalloc(ip->ipos+1)) != NULL)
		{
		strpad(editstr,ip->itemstr,ip->ipos);
		pos = *startpos;
		wn_cursoron();
		do
			{
			done = 0;

			if (pos > ip->ipos-1)
				pos = ip->ipos-1;
			wn_gotoxy(ip->x+pos, ip->y);
			wn_directwritestr(ip->x,ip->y,editstr,ip->ipos);
			kb_read(ch,sc);

			if (*ip->userfunc)
				{
				if ((ret = (ip->userfunc)(ch,sc)) != MN_USER_NOTHING)
					{
					if (ret == MN_USER_BREAK)
						break;
					else if (ret == MN_USER_CONTINUE)
						continue;
					}
				}

			if (mn_searchuserkey(m,*ch,*sc))
				break;
			else
				{
				if (*ch=='\x0' || *ch=='\xe0')
					{
					done = 0;
					switch (*sc)
						{
						case KB_SC_N_HOME:
							pos = 0;
							break;
						case KB_SC_N_END:
							pos = strlast(editstr)+1;
							break;
						case KB_SC_N_LEFT:
							if (pos > 0) pos--;
							break;
						case KB_SC_N_RIGHT:
							if (pos < ip->ipos-1) pos++;
							break;
						case KB_SC_N_DEL:
							strdelch(editstr,1,pos,ip->ipos);
							break;
						case KB_SC_N_INS:
							Ins = !Ins;
							break;
						default:
							done = 1;
						}
					}
				else if (*ch == '\b')
					{
					if (pos > 0)
						strdelch(editstr,1,--pos,ip->ipos);
					}
				else if (*ch >= 0x20 && *ch <= 0xff)
					{
					if (Ins)
						strinsch(editstr,*ch,pos,ip->ipos);
					else
						editstr[pos] = *ch;
					pos++;
					}
				else
					done = 1;
				}
			}
		while (! done);
		wn_cursoroff();
		*startpos = pos;
		strncpy(ip->itemstr,editstr,ip->ipos);
		ut_memfree(editstr);
		}
}

/*---------------------------------------------------------------------------

*/
int mn_rdmenu (MENUTYPE m, int *ch, int *sc)
{
WINTYPE wsave;
KEYDEF *kd;
int i, it = MN_NOSELECT, lastpos, ret;
UINT startpos = 0;

	if (mn_valid(m))
		{
		wsave = wn_used();

		wn_use(m->winptr);
		for (i=0; i<m->lastitemno; i++)
			mn_display_item(m, i, m->curpos == i ? SET : CLR);
		if (m->islotus)
			wn_putontop(m->lotwinptr);
		wn_putontop(m->winptr);
		do
			{
			lastpos = m->curpos;

			if (m->itemptr[m->curpos].itype == MN_EDITITEM)
				mn_edit_item(m,&startpos,ch,sc);
			else
				{
				kb_read(ch,sc);

				if (*m->itemptr[m->curpos].userfunc)
					{
					if ((ret = (m->itemptr[m->curpos].userfunc)(ch,sc)) != MN_USER_NOTHING)
						{
						if (ret == MN_USER_BREAK)
							break;
						else if (ret == MN_USER_CONTINUE)
							continue;
						}
					}
				}

			if ((kd = mn_searchkey(m,*ch,*sc)) != NULL)
				{
				switch (kd->barmove)
					{
					case MN_NOMOVE   : break;
					case MN_UP       : m->curpos = mn_nextpos(m,MN_UP); break;
					case MN_DOWN     : m->curpos = mn_nextpos(m,MN_DOWN); break;
					case MN_RIGHT    : m->curpos = mn_nextpos(m,MN_RIGHT); break;
					case MN_LEFT     : m->curpos = mn_nextpos(m,MN_LEFT); break;
					case MN_FIRST    : m->curpos = 0; break;
					case MN_LAST     : m->curpos = m->lastitemno-1; break;
					case MN_PREVIOUS : if (m->curpos > 0) m->curpos--; break;
					case MN_NEXT     : if (m->curpos < m->lastitemno-1) m->curpos++; break;
					case MN_GOTO     : m->curpos = kd->itempos; break;
					}
				if (lastpos != m->curpos)
					{
					startpos = 0;
					mn_display_item(m,lastpos,CLR);		/* clear old bar */
					mn_display_item(m,m->curpos,SET);		/* new bar */
					}
				if (kd->action & MN_BEEP) sc_bell();
				if (kd->action & MN_REMOVE) mn_remove(m);
				if (kd->action & MN_TRANSMIT) {it = m->curpos+1; break; }
				if (kd->action & MN_CONDTRANS && lastpos == kd->itempos) { it = m->curpos+1; break; }
				if (kd->action & MN_ABORT) {it = MN_NOSELECT; break;}
				}
			}
		while (1);

   	wn_use(wsave);
		}
return it;
}

/*---------------------------------------------------------------------------

*/
WINTYPE mn_win (MENUTYPE m)
{

return m->winptr;
}

/*---------------------------------------------------------------------------

*/
void mn_exit (void)
{
	while (MenuList)
		mn_destroy(MenuList->menu);
}

