/*

	binder.hpp
	8-27-91
	Loose Data Binder

	Copyright 1991
	John W. Small
	All rights reserved

	PSW / Power SoftWare
	P.O. Box 10072
	McLean, Virginia 22102 8072 USA

	John Small
	Voice: (703) 759-3838
	CIS: 73757,2233

*/


#ifndef BINDER_HPP
#define BINDER_HPP

#include <limits.h>	// UINT_MAX
#include <iostream.h>
#include <iomanip.h>


#define ID_Streamable	0
#define ID_UnknownStreamable	ID_Streamable
#define ID_Binder	1


class Streamable;
typedef Streamable * StreamablE;
#define StreamablE0 ((StreamablE)0)

class StreamableClassRegistry;
#define SCRegistry  SCReg
extern StreamableClassRegistry SCRegistry;

/*

	A class must include the STREAMABLE macro in its
	public section and be derived either publicly or
	privately from Streamable but never virtually or
	otherwise multiply inherited in order for it to
	be "streamable" (see Binder below).

*/

class Streamable  {
	friend StreamableClassRegistry;
	unsigned id;
	unsigned streamed;
/*
	Stream contents:

			1st & Only	1st of Many	MultiRef
	id		x               x		x
	streamed        0               1		2
	spos		x		x		x
	data            x               x
*/
	long spos;
	unsigned refCount;
public:
	static int debug;
	static int refDebug;
protected:
	// Static so that the static load() can call!
	static void lserror(const char *msg, unsigned id);
	virtual void serror(const char *msg);
	virtual void swarn(const char *msg);
	virtual ostream& store(ostream& os)
		{ return os; }
	static StreamablE load(istream& is,
		StreamablE InstancE);
	void setID(unsigned id)
		{ this->id = id; }
public:
	static char memberTermChar;
	enum { CLASS_ID = 0 };
	// The first parameter below, "dummy", insures that
	// the STREAMABLE macro defines a unique constructor
	// for derived classes!
	#pragma argsused
	Streamable(StreamableClassRegistry& dummy,
		unsigned id = Streamable::CLASS_ID);
	static void registerClass(unsigned id = CLASS_ID,
		StreamablE (*loader)(istream& is,
		StreamablE InstancE) = load);
	operator StreamablE() { return this; }
	unsigned ID() { return id; }
	virtual void restream();
	unsigned unlink();
	StreamablE link();
	virtual ~Streamable() {}
};


// stream manipulator: insert end of member field char

extern ostream& endm(ostream& os);

// stream manipulator: extract end of member field char

extern istream& nextm(istream& is);


/*
	The STREAMABLE macro declares three functions:
	store(), load(), and restream() which you must
	define to store member data onto a stream, to
	load that data back from that stream, and to
	reset the stream automatical multiple linking
	mechanism respectively.  You must assign a unique
	id to each streamable class and register every class
	that will be stored on a stream!  See Binder below
	for an example of how to use STREAMABLE.
*/

#define STREAMABLE(class, id, base) \
private:  \
	friend StreamableClassRegistry;  \
protected:  \
	base :: lserror;  \
	base :: serror;  \
	base :: swarn;  \
	virtual ostream& store(ostream& os);  \
	static StreamablE load(istream& is,  \
		StreamablE InstancE);  \
	base :: setID;  \
public:  \
	enum { CLASS_ID = id };  \
	class (StreamableClassRegistry& dummy,  \
		unsigned cid = class ::CLASS_ID) \
		: base (dummy,cid) { return; }  \
	operator StreamablE()  \
		{ return (StreamablE)this; }  \
	static void registerClass(unsigned cid =   \
		class ::CLASS_ID,  \
		StreamablE (*loader)(istream& is,  \
		StreamablE InstancE) = class ::load)  \
		{ base :: registerClass(cid,loader); }  \
	base :: ID;  \
	base :: unlink;  \
	base :: link;  \
	virtual void restream()


//  value for dummy parameter of constructor above

#define UNIQUE_STREAMABLE  SCRegistry



/*

	The "Loose Data Binder", or Binder for short, binds
	any type of data together in hybrid stack-queue-
	deque-list-array structure.  Like a loose leaf
	notebook where you can insert or arrange leafs in
	any fashion, the "Loose Data Binder" allows you to
	collect and arrange data in any fashion.

*/


typedef void * voiD;
#define voiD0  ((voiD)0)
typedef voiD * voiDV;
#define voiDV0 ((voiDV)0)

#define BDR_MAXNODES	(UINT_MAX/sizeof(voiD))
#define BDR_LIMIT	20
#define BDR_DELTA	10
#define BDR_NOTFOUND	BDR_MAXNODES

/*  Search/sort compare function pointer type:  */

typedef int (*BDRcomparE)(const voiD D1, const voiD D2);
#define BDRcomparE0 ((BDRcomparE)0)


/*  Function pointer types for the Binder iterators:  */

typedef void (*BDRforEachBlocK)
		(voiD D, voiD M, voiD A);
typedef int (*BDRdetectBlocK)
		(voiD D, voiD M);
class Binder;
typedef Binder * BindeR;
typedef void (*BDRcollectBlocK)
		(voiD D, BindeR R, voiD M, voiD A);



class Binder : Streamable  {
	unsigned lowLimit, lowThreshold, first;
	voiDV linkS;
	unsigned limit, delta, nodes, maxNodes;
	unsigned curNode, flags;
	BDRcomparE comparE;
protected:
	void construct(unsigned flags, unsigned maxNodes,
		unsigned limit, unsigned delta);
	virtual int Dfree(voiD D);
	virtual void Dstore(ostream& os, const voiD D);
	virtual voiD Dload(istream& is);
	virtual voiD DinsQ(voiD D);
public:
	enum FLAG_BITS { SORTED = 1, ALL_FREE_DESTRUCT = 2,
		STREAMABLE_NODES = 4 };
	STREAMABLE(Binder,ID_Binder,Streamable);
	Binder(unsigned flags = 0, // see FLAG_BITS above
		unsigned maxNodes = BDR_MAXNODES,
		unsigned limit = BDR_LIMIT,
		unsigned delta = BDR_DELTA)
		: Streamable(UNIQUE_STREAMABLE,CLASS_ID)
		{ construct(flags,maxNodes,limit,delta); }
	Binder(voiDV argv, int argc = 0, unsigned flags = 0);
	voiDV vector();
	virtual ~Binder();
	unsigned Limit()  { return limit; }
	unsigned setLimit(unsigned newLimit);
	unsigned pack()  { return setLimit(nodes); }
	unsigned Delta()  { return delta; }
	unsigned setDelta(unsigned newDelta = BDR_DELTA);
	unsigned Nodes()  { return nodes; }
	unsigned MaxNodes()  { return maxNodes; }
	unsigned setMaxNodes(unsigned newMaxNodes
		= BDR_MAXNODES);
	unsigned vacancy()  { return maxNodes - nodes; }
	unsigned vacancyNonElastic()
			{ return limit - nodes; }
	voiD atIns(unsigned n, const voiD D);
	voiD atDel(unsigned n);
	int  allDel();
	int  atFree(unsigned n)  { return Dfree(atDel(n)); }
	int  allFree();
	voiD atPut(unsigned n, const voiD D);
	voiD atGet(unsigned n);
	voiD operator[](unsigned n)  { return atGet(n); }
	voiD at(unsigned n, const voiD D = voiD0);
	unsigned index(const voiD D);
	voiD add(const voiD D)  { return atIns(nodes,D); }
	Binder& operator+=(const voiD D)
		{ atIns(nodes,D); return *this; }
	voiD subtract(const voiD D)
		{ return atDel(index(D)); }
	Binder& operator-=(const voiD D)
		{ atDel(index(D)); return *this; }
	int  forEach(BDRforEachBlocK B, voiD M = voiD0,
		voiD A = voiD0);
	unsigned firstThat(BDRdetectBlocK B, voiD M = voiD0);
	unsigned lastThat(BDRdetectBlocK B, voiD M = voiD0);
	int  collect(BDRcollectBlocK B, BindeR R,
		voiD M = voiD0, voiD A = voiD0);

/*  FlexList like primitives:  */

	voiD top()  { return atGet(0); }
	voiD current()  { return atGet(curNode); }
	operator voiD()  { return atGet(curNode); }
	/*
		The implicit (voiD) cast is meant to be
		used in conjunction with the iterators and
		not the << and >> operators!
	*/
	voiD bottom()  { return atGet(nodes-1); }
	unsigned CurNode();
	int  setCurNode(unsigned n = BDR_MAXNODES);
	int  Sorted()  { return (flags & SORTED); }
	void unSort()  { flags &= ~SORTED; }
	BDRcomparE ComparE()  { return comparE; }
	void setComparE(BDRcomparE comparE)
		{ this->comparE = comparE; flags &= ~SORTED; }
	voiD push(const voiD D)  { return atIns(0,D); }
	voiD pop()  { return atDel(0); }
	int  popFree()  { return Dfree(atDel(0)); }
	Binder& operator>>(voiD& D)
		{ D = atDel(0); return *this; }
	voiD insQ(const voiD D)
		{ return atIns(nodes,D); }
	Binder& operator<<(const voiD D)
		{ atIns(nodes,D); return *this; }
	Binder& operator<<(Binder& (*manipulator)
		(Binder& B));
	voiD rmQ()  { return atDel(0); }
	int  rmQFree()  { return Dfree(atDel(0)); }
	voiD unQ() /* Remove from rear of Q */
		{ return atDel(nodes-1); }
	int  unQFree()  { return Dfree(atDel(nodes-1)); }
	voiD ins(const voiD D);
	voiD insSort(const voiD D);
	voiD del();
	int  delFree()  { return Dfree(del()); }
	voiD next();
	voiD operator++()  { return next(); }
	voiD prev();
	voiD operator--()  { return prev(); }
	voiD findFirst(const voiD K);
	voiD findNext(const voiD K);
	voiD findLast(const voiD K);
	voiD findPrev(const voiD K);
	int  sort(BDRcomparE comparE = BDRcomparE0);
};





/*

	The StreamableClassRegistry holds the records of
	classes and their static load functions.  Load
	functions have to be static because constructors
	don't have addresses in C++.  Store functions are
	virtual and thus can be called automatically.

*/


struct StreamableClassRecord {
	unsigned id;
	StreamablE (*load)(istream& is,
		StreamablE InstancE);
	StreamableClassRecord(unsigned id,
		StreamablE (*load)(istream& is,
			StreamablE InstancE))
		{ this->id = id; this->load = load; }
};
typedef StreamableClassRecord * SCRecorD;
#define SCRecorD0 ((SCRecorD)0)

struct StreamableClassHoldingRecord  {
	StreamablE InstancE;
	long spos;
	StreamableClassHoldingRecord(StreamablE InstancE,
		long spos) { this->InstancE = InstancE;
		this->spos = spos; }
};
typedef StreamableClassHoldingRecord * SCHRecorD;
#define SCHRecorD0 ((SCHRecorD)0)


class StreamableClassRegistry  {
	Binder ClassRecords;
	Binder InstanceLinks;
public:
	static int debug;
protected:
	virtual void error(char *msg, unsigned id);
	virtual void warn(char *msg, unsigned id);
public:
	StreamableClassRegistry()
		: ClassRecords(Binder::ALL_FREE_DESTRUCT),
		InstanceLinks() {}
	void restream()
		{ InstanceLinks.allDel(); }
	void registerClass(unsigned id, StreamablE (*loader)
		(istream& is, StreamablE InstancE));
	void forgetClasses();
	istream& get(istream& is, StreamablE& InstancE);
	ostream& put(ostream& os, StreamablE InstancE);
	~StreamableClassRegistry() { forgetClasses(); }
};

#define RestreamRegistry()  SCRegistry.restream()


inline istream& operator>>(istream& is, StreamablE& InstancE)
{
	return SCRegistry.get(is,InstancE);
}

inline ostream& operator<<(ostream& os, StreamablE InstancE)
{
	return SCRegistry.put(os, InstancE);
}



/*

	FncPtrRegistry is provided so that you can stream
	function pointers.

*/

typedef void (*GenericFnC)();
#define GenericFnC0 ((GenericFnC)0)
#define ID_UnknownGenericFnC	0

struct StreamableFncPtrRecord {
	unsigned id;
	GenericFnC fnC;
	StreamableFncPtrRecord(unsigned id,
		GenericFnC fnC)
		{ this->id = id; this->fnC = fnC; }
};
typedef StreamableFncPtrRecord * SFPRecorD;
#define SFPRecorD0 ((SFPRecorD)0)

class StreamableFncPtrRegistry  {
	Binder FncPtrRecords;
public:
	static int debug;
protected:
	virtual void error(char *msg, unsigned id);
	virtual void warn(char *msg, unsigned id);
public:
	StreamableFncPtrRegistry() : FncPtrRecords()  {}
	void registerFunction(unsigned id,
		GenericFnC fnC);
	GenericFnC FnC(unsigned id);
	unsigned ID(GenericFnC fnC);
	~StreamableFncPtrRegistry()
		{ FncPtrRecords.allFree(); }
};

#define SFPRegistry  SFPReg

extern StreamableFncPtrRegistry SFPRegistry;

#define RegisterFunction(id, fnC)  \
		SFPRegistry.registerFunction(id,fnC)
#define FncPtrToID(fnC)  \
		SFPRegistry.ID(fnC)
#define IDtoFncPtr(id)  \
		SFPRegistry.FnC(id)


#endif
