/*$mmod s */
/*$ilib struct */

#define WANTMAIN 0

/*==============================================================================
|  Modification date  3-6-88						       |
==============================================================================*/
/*==============================================================================
|      Name:   CHAINRW.c						       |
|      Desc:   two modules -- one to write out chains, another to read them.   |
|									       |
| Processes:   int							       |
|	       write_chain (anchor, outp)				       |
|	       void **anchor; FILE * outp;				       |
|									       |
|	       int							       |
|	       read_chain (anchor , inpt)				       |
|	       void **anchor ; FILE *inpt;				       |
|									       |
|     Input:   anchor is the address of your chain anchor pointer.  This may   |
|		      may be NULL (empty chain) or for an existing chain.      |
|	       outp/inpt  are FILE pointers to a file already OPENed in        |
|		      BINARY mode !					       |
|									       |
|    Output:   An in-memory chain is either written to the file or read        |
|	       from the file.						       |
|	       Both routines return the count of members processed if	       |
|	       successful. Or a negative error code.			       |
|	     "WRITE_CHAIN"                                                     |
|	     "READ_CHAIN"                                                      |
|	       -1 = invalid header ( file not positioned properly)	       |
| Side eff.:								       |
|									       |
| Narrative:  These two write and read chins.  An alignment block is written   |
|	       first.  Then the data blocks, with headers containing a	       |
|	       length and signature.  Finally a signature block to insure      |
|	       final checking of alignment.				       |
==============================================================================*/
#include <stdio.h>
#include <stdlib.h>

   extern int chain_member_size;       /* global from chain2		     */

   /*Ŀ
      Initial structure written prior to writing the chain members.	       
      For reading, this structure must be the first entity read from	       
      the current position of the file.				       
   */
   static struct  {
       long	   curr_pos;	       /* current file position 	     */
       long	   signature;	       /* chainrw ID signature		     */
       int	   member_cnt;	       /* count of members in the chain      */
       } header;

   /*Ŀ
      Structure pre-pended to each member as they are written. 	       
   */
   static struct  {
       unsigned    flag ;	       /* signature to insure alignment      */
       int	   length;	       /* length of data portion.	     */
       } rec_hdr;

   /*Ŀ
      Structure written after all members.  This is checked to insure	       
      correct alignment within the file.				       
   */
   static struct  {
       unsigned    flag ;
       int	   length ;
       long	   curr_pos;
       } footer;

   static long posit;		       /* file position storage 	     */

   extern void * chain_to (void **anchor, int size);
   extern void * get_1st_chain (void * anchor);
   extern void * get_next_chain (void * member);
   extern int	 chain_count (void * anchor);

/*==============================================================================
|  Write chain to file 'outf' -- assumed open 'wb' -- must be binary!!         |
|									       |
|									       |
==============================================================================*/
int
write_chain (anchor, outp)
void **anchor; FILE * outp;
  {
   void * rec_ptr;
   int i = 0;

   /*Ŀ
      Build the header structure for alignment 			       
   */
   header.curr_pos = ftell (outp);     /* current file position 	     */
   header.signature = 987654321L;      /* ID signature			     */
				       /* get the member count for chain     */
   header.member_cnt = chain_count (*anchor);
				       /* write the result		     */
   fwrite (&header,1, sizeof(header), outp);

   rec_hdr.flag = 0xdbdb;	       /* set the member ID flag	     */
				       /*  arbitrary value that is easy to   */
				       /*  spot in dumps, or 'list's.        */
   rec_ptr = get_1st_chain (*anchor);  /* get 1st member record 	     */

   /*Ŀ
      Loop for all remaining members					       
   */
   while (rec_ptr != NULL)  {
				       /* pick up size of member	     */
       rec_hdr.length = chain_member_size;
				       /* write header			     */
       fwrite (&rec_hdr, sizeof (rec_hdr),1, outp);
				       /* write member data		     */
       fwrite (rec_ptr, chain_member_size,1, outp);
       i++;
				       /* go on to next member		     */
       rec_ptr = get_next_chain (rec_ptr);
       }

   /*Ŀ
      Prepare and write footer alignment block 			       
   */
   footer.flag	   = 0xdbdb;
   footer.length   = 0;
   footer.curr_pos = ftell (outp);
   fwrite (&footer,1, sizeof(footer), outp);
   return (i);
   }


/*==============================================================================
|									       |
==============================================================================*/
int
read_chain (anchor , inpt)
void **anchor ; FILE *inpt;
  {
   long    curr_pos;
   void * rec_ptr;


   /*Ŀ
      Validate correct file position					       
   */
   curr_pos = ftell (inpt);	       /* get current file location	     */
				       /* get file header		     */
   fread (&header, sizeof(header),1, inpt);
				       /* check for valid header	     */
   if ((header.signature != 987654321L) ||
       (header.curr_pos  != curr_pos))	{
	   return (-1); 	       /* error in positioning		     */
	   }

   /*Ŀ
      Good position and header.					       
      Now read for specified number of members.			       
   */
   while (header.member_cnt -- > 0)  {
				       /* read member header to get data size*/
       fread (&rec_hdr, sizeof (rec_hdr),1,inpt);
				       /* have chain_to allocate data space  */
       rec_ptr = chain_to (anchor,rec_hdr.length);
				       /* read the data portion 	     */
       fread (rec_ptr, rec_hdr.length, 1, inpt);
       }

   /*Ŀ
      Check file position when specified number of members read.	       
   */
   curr_pos = ftell (inpt);	       /* get file position		     */

   /*Ŀ
      Now read the trailer and verify positioning			       
   */
   fread (&rec_hdr, sizeof (rec_hdr),1,inpt);
   if (rec_hdr.length != 0) return (-1);

   fread (&posit, sizeof (long),1,inpt);/* read the trailing position	      */
   if (curr_pos != posit)  return (-2);/* position out of sink		     */

   /*Ŀ
      Get count of members from the chain.				       
   */
   return (chain_count (*anchor));     /* return count of records in chain   */
   }

#if WANTMAIN > 0
/*==============================================================================
|  Test program 							       |
==============================================================================*/
   static void * anchor = NULL;
   FILE * test_out;
main ()
  {
   int i,j;
   void * m1;

   for (i=0; i<14; i++)  {
       m1 = chain_to(&anchor,10);      /* members 10 bytes long 	     */
       if (m1==NULL)  {
	   printf ("NULL returned from chain_to!!\n");
	   exit (-1);
	   }
       itoa (i,m1,10);		       /* ascii digit (base 10) 	     */
       printf (" %d ",chain_count(anchor));
       }

   /*Ŀ
      NOTE!!	       BINARY ----------v				       
   */
   if ((test_out = fopen ("rwxxx.tst","wb"))== NULL)  {
       perror ("Open 'rwxxx.tst'");
       }

   i = write_chain (&anchor, test_out);
   printf ("\nWRITE_CHAIN did %d records\n", i);
   fclose (test_out);		       /* close stored file		     */
   free_chain (&anchor);	       /* deallocate the chain. 	     */
				       /*  sets anchor to NULL		     */



   printf ("Testing open and read of chain\n");

   if ((test_out = fopen ("rwxxx.tst","rb"))== NULL)  {
       perror ("Open 'rwxxx.tst'");
       }

   i = read_chain (&anchor, test_out);
   printf ("READ_CHAIN did %d records\n", i);
   fclose (test_out);

   printf ("Members of chain are:\n");
   m1 = get_1st_chain (anchor);
   while (m1 != NULL)  {
       printf ("  <%s>\n",m1);
       m1 = get_next_chain (m1);
       }

   return (0);
   }
#endif
