/***********************************************************************
* Prototypes and typedefs for the CRC utility routines.
*
* Author: Gary P. Mussar
* This code is released to the public domain. There are no restrictions,
* however, acknowledging the author by keeping this comment around
* would be appreciated.
***********************************************************************/

/***********************************************************************
* If we can handle ANSI prototyping, lets do it.
***********************************************************************/

#ifdef ANSIPROTO
#define PARMS(x) x
#else
#define PARMS(x) ()
#endif

/***********************************************************************
* The following #defines are used to define the types of variables
* used to hold or manipulate CRCs. The 16 bit CRCs require a data
* type with at least 16 bits. The 32 bit CRCs require a data type
* with at least 32 bits. In addition, the data bits reserved for the
* CRC must be manipulated in an unsigned fashion. It is possible to
* define a data type which is larger than required to hold the CRC,
* however, this is an inefficient use of memory and usually results
* in less efficient code when manipulating the CRCs.
***********************************************************************/

#ifdef MSDOS
#define CRC16 unsigned short
#else
#define CRC16 unsigned
#endif

#define CRC32 unsigned long

/***********************************************************************
* Utilities for fast CRC using table lookup
*
* I_CRCxx  -  Initialize the 256 entry CRC lookup table based on the
*             specified generator polynomial.
* Input:
*    Table[256]     - Lookup table
*    *GenPolynomial - Pointer to generator polynomial
*
* F_CRCxx  -  Calculate CRC over an array of characters using fast
*             table lookup.
* Input:
*    Table[256]    - Lookup table
*    *CRC          - Pointer to the variable containing the result of
*                    CRC calculations of previous characters. The CRC
*                    variable must be initialized to a known value
*                    before the first call to this routine.
*    *dataptr      - Pointer to array of characters to be included in
*                    the CRC calculation.
*    count         - Number of characters in the array.
*
* S_CRCxx  -  Calculate CRC over an array of characters using slower but
*             smaller non-table lookup method.
* Input:
*    *GenPolynomial - Pointer to generator polynomial
*    *CRC           - Pointer to the variable containing the result of
*                     CRC calculations of previous characters. The CRC
*                     variable must be initialized to a known value
*                     before the first call to this routine.
*    *dataptr       - Pointer to array of characters to be included in
*                     the CRC calculation.
*    count          - Number of characters in the array.
***********************************************************************/
extern void I_CRC16 PARMS((CRC16 Table[256],         \
					CRC16 *GenPolynomial));

extern void F_CRC16 PARMS((CRC16 Table[256],         \
					CRC16 *CRC,                      \
					void *dataptr,                   \
					unsigned int count));

extern void S_CRC16 PARMS((CRC16 *GenPolynomial,     \
					CRC16 *CRC,                      \
					void *dataptr,                   \
					unsigned int count));

extern void I_CRC32 PARMS((CRC32 Table[256],         \
					CRC32 *GenPolynomial));

extern void F_CRC32 PARMS((CRC32 Table[256],         \
					CRC32 *CRC,                      \
					void *dataptr,                   \
					unsigned int count));

extern void S_CRC32 PARMS((CRC32 *GenPolynomial,     \
					CRC32 *CRC,                      \
					void *dataptr,                   \
					unsigned int count));
