/*
   CPPTask - A Multitasking Kernel For C++

   TASK.HPP - Type and Class definitions and global routine prototypes.

   Version 1.0 08-12-91

   Ported by Rich Smith from:

   Public Domain Software written by
      Thomas Wagner
      Patschkauer Weg 31
      D-1000 Berlin 33
      West Germany
*/


#include "tskconf.hpp"
#include <stdio.h>
#include <dos.h>

#define _Near
#define TURBO  1
#define MSC    0

#define local static      // Comment out the "static" for debugging

typedef unsigned char byte;
typedef unsigned short word;
typedef unsigned long dword;
typedef void (cdecl far *funcptr)();
typedef void far *farptr;
typedef byte far *byteptr;
typedef word far *wordptr;

#define TTIMEOUT ((farptr) -1L)
#define TWAKE    ((farptr) -2L)

// Task states

#define  ST_KILLED   0
#define  ST_STOPPED  1
#define  ST_DELAYED  2
#define  ST_WAITING  3
#define  ST_ELIGIBLE 4
#define  ST_RUNNING  5

// Task flags

#define  F_TEMP   0x80     // Task is temporary, free on kill
#define  F_STTEMP 0x40     // Task stack is temporary, free on kill
#define  F_CRIT   0x01     // Task is critical, may not be preempted

#define  FL_SYSM  0xf0     // Mask for system flags
#define  FL_USRM  0x0f     // Mask for user flags

// Timer queue element states

#define  TSTAT_REMOVE      -1    // Remove element from queue
#define  TSTAT_IDLE        0     // Not in queue
#define  TSTAT_COUNTDOWN   1     // Count down timeout, then remove
#define  TSTAT_REPEAT      2     // Count down, reload when done

// Timer queue element control structure pointer kinds

#define  TKIND_NULL        0     // NULL timer, used for deleted task
#define  TKIND_TASK        1     // tcbptr, Wakeup associated task
#define  TKIND_WAKE        2     // tcbptr, but not same task
#define  TKIND_PROC        3     // call function
#define  TKIND_FLAG        4     // flagptr, set flag
#define  TKIND_COUNTER     5     // counterptr, increment counter

#define  TKIND_TEMP        0x80  // Hi bit set means temporary element

extern "C" {
   void asm_schedule(void);
   void asm_c_schedule(void);
   int tsk_dis_int(void);
   void tsk_ena_int(int);
   void tsk_nop(void);
}

class task;    // forward reference
typedef task far *tcbptr;

typedef tcbptr far *tqueptr;

class timer; // forward reference
typedef timer far *tlinkptr;

class timer {
   tlinkptr next;
   dword    timeout;    // Timeout counter
   dword    reload;     // Timeout counter reload value
   farptr   strucp;     // Pointer to control structure
   byte     tstate;     // Element state
   byte     tkind;      // Kind of control structure
   void     tsk_timer_action (void);
   friend   local void far timer_main (void);
   friend   task;
public:
   timer(dword tout, farptr strucp, byte kind, byte rept);
   ~timer(void);
   void change_timer(dword tout, byte rept);
};

class tasker_class {
public:
   tasker_class(byte varpri, int speedup);
   ~tasker_class(void);
   void preempt_on(void);
   void preempt_off(void);
   void far schedule(void);
   void far c_schedule(void); 
   void tsk_dis_preempt(void);
   void tsk_ena_preempt(void);
};

class task {
// private data members
   tcbptr   next;       // Next in queue
   tqueptr  queue;      // Queue head pointer
   byteptr  stack;      // Task stack
   byteptr  stkbot;     // Task stack bottom
   word     prior;      // Task priority
   word     initprior;  // Initial Task priority
   byte     state;      // Task state
   byte     flags;      // Task flags
   timer   *timerq;     // Timer queue link
   farptr   retptr;     // Event return pointer
   int      retsize;    // Return buffer size for pipes
// private member functions
   friend int  far t_delay(dword ticks);
   friend void far tsk_wait (tqueptr que, dword timeout);
   friend void     killretn(void);
   friend void far tsk_kill_queue(tqueptr que);
   void            tsk_kill(void);
   void            tsk_wakeup(void);
public:
   task(funcptr func, byteptr stack, word stksz,
        word priority, farptr arg);
   ~task(void);
   void* operator new(size_t size);
   int start_task(void);
   int wake_task(void);
   void tsk_enqueue(tqueptr que);
   void tsk_unqueue(void);
   void tsk_enqtimer(dword tout);
   void tsk_unqtimer(void);
   tcbptr tsk_runable(void);

   void set_priority(word newprior);
   word get_priority(void);
   void set_state(byte newstate) { state = newstate; }
   byte get_state(void) { return state; }
   void set_user_flags(byte newflags);
   void set_system_flags(byte newflags);
   byte get_flags(void) { return flags; }
   void set_retptr(farptr newptr) { retptr = newptr; }
   farptr get_retptr(void) { return retptr; }
   void set_retsize(int newsize) { retsize = newsize; }
   int get_retsize(void) { return retsize; }
   tcbptr get_next(void) { return next; }
};

class resource {
   tcbptr  waiting;   // queue for tasks waiting for resource
   tcbptr   owner;    // current owner of the resource
   int      state;    // resource state
   byte     flags;    // resource flags
public:
   resource();
   ~resource(void);
   void release_resource(void);
   int request_resource(dword timeout);
   int c_request_resource(void);
   int check_resource(void);
   int get_state(void) { return state; }
   byte get_flags(void) { return flags; }
};

typedef resource far *resourceptr;

class flag {
   tcbptr   wait_set;     // queue for tasks waiting for flag to be set
   tcbptr   wait_clear;   // queue for tasks waiting for flag to be cleared
   int      state;        // flag state
   byte     flags;        // flag flags
public:
   flag();
   ~flag(void);
   void set_flag(void);
   void clear_flag(void);
   int wait_flag_set(dword timeout);
   int wait_flag_clear(dword timeout);
   int clear_flag_wait_set(dword timeout);
   int check_flag(void);
};

typedef flag far *flagptr;

class counter {
   tcbptr   wait_set;    // queue for tasks waiting for counter to be set
   tcbptr   wait_clear;  // queue for tasks waiting for counter to be cleared
   dword    state;       // counter state
   byte     flags;       // counter flags
public:
   counter();
   ~counter(void);
   void set_flag(void);
   void clear_counter(void);
   int wait_counter_set(dword timeout);
   int wait_counter_clear(dword timeout);
   void inc_counter(void);
   dword check_counter(void);
};

typedef counter far *counterptr;

struct msg_header {
   struct msg_header far *next;
};

typedef struct msg_header far *msgptr;

class mailbox {
   tcbptr   waiting;     // queue for tasks waiting for mail
   msgptr   mail_first;  // pointer to first message in mailbox
   msgptr   mail_last;   // pointer to last message in mailbox
   byte     flags;       // mailbox flags
public:
   mailbox();
   ~mailbox(void);
   void send_mail(farptr msg);
   farptr wait_mail(dword timeout);
   farptr c_wait_mail(void);
   int check_mailbox(void);
};

typedef mailbox far *mailboxptr;

class pipe {
   tcbptr    wait_read;    // queue for tasks blocked on read
   tcbptr    wait_write;   // queue for tasks blocked on write
   tcbptr    wait_clear;   // queue for tasks waiting for pipe to be clear
   word      bufsize;      // size of pipe buffer
   word      filled;       // number of bytes in the pipe
   word      inptr;        // head pointer for circular buffer
   word      outptr;       // tail pointer for circular buffer
   byteptr   contents;     // pointer to pipe contents
   byte      flags;        // pipe flags

// private member functions
   byte      tsk_getcpipe ();
   void      tsk_putcpipe (byte c);
public:
   pipe(farptr buf, word buffersize);
   ~pipe(void);
   int read_pipe(dword timeout);
   int c_read_pipe(void);
   int write_pipe(byte ch, dword timeout);
   int c_write_pipe(byte ch);
   int wait_pipe_empty(dword timeout);
   int check_pipe(void);
   word pipe_free(void);
   void flush_pipe(void);
   word get_bufsize(void) { return bufsize; }
   word get_filled(void) { return filled; }
   byte get_flags(void) { return flags; }
};

typedef pipe far *pipeptr;

class wpipe {
   tcbptr    wait_read;    // queue for tasks blocked on read
   tcbptr    wait_write;   // queue for tasks blocked on write
   tcbptr    wait_clear;   // queue for tasks waiting for pipe to be clear
   word      bufsize;      // size of pipe buffer
   word      filled;       // number of words in the pipe
   word      inptr;        // head pointer for circular buffer
   word      outptr;       // tail pointer for circular buffer
   wordptr   wcontents;    // pointer to pipe contents
   byte      flags;        // pipe flags

// private member functions
   word      tsk_getwpipe ();
   void      tsk_putwpipe (word c);
public:
   wpipe(farptr buf, word buffersize);
   ~wpipe(void);
   word read_wpipe(dword timeout);
   word c_read_wpipe(void);
   int write_wpipe(word ch, dword timeout);
   int c_write_wpipe(word ch);
   int wait_wpipe_empty(dword timeout);
   word check_wpipe(void);
   word wpipe_free(void);
   void flush_wpipe(void);
   word get_bufsize(void) { return bufsize; }
   word get_filled(void) { return filled; }
   byte get_flags(void) { return flags; }
};

typedef wpipe far *wpipeptr;

class buffer {
   resource    buf_write;  // write buffer resource
   resource    buf_read;   // read buffer resource
   wpipe       pip;        // pipe used to implement buffer
   word        msgcnt;     // number of messages in buffer
   byte        flags;      // buffer flags

// private member function
   int         tsk_wrbuf (word w, dword timeout);
public:
   buffer(farptr buf, word bufsize);
   ~buffer(void);
   int read_buffer(farptr msg, int size, dword timeout);
   int c_read_buffer(farptr msg, int size);
   int write_buffer(farptr msg, int size, dword timeout);
   int c_write_buffer(farptr msg, int size);
   word check_buffer(void);
};

typedef buffer far *bufferptr;

// create definitions for critical code section handling

#define CRITICAL  int crit_intsav
#define C_ENTER   crit_intsav = tsk_dis_int()
#define C_LEAVE   tsk_ena_int (crit_intsav)


extern void far tsk_kill_queue(tqueptr);

extern farptr tsk_alloc (word size);
extern void tsk_free (farptr item);
extern resource alloc_resource;

// provide "C" linkage for routines to be called from assemply language

extern "C" void far asm_remove_tasker(void);
extern "C" void far asm_inc_counter(counterptr cnt);
extern "C" void far asm_set_flag(flagptr flg);
extern "C" void far asm_clear_flag(flagptr flg);
extern "C" int far asm_wait_flag_clear(flagptr flg, dword timeout);
extern "C" int far asm_request_resource(resourceptr rsc, dword timeout);
extern "C" void far asm_release_resource(resourceptr rsc);
extern "C" word far asm_read_wpipe(wpipeptr pip, dword timeout);
extern "C" word far asm_check_wpipe(wpipeptr pip);
extern "C" int far asm_c_write_wpipe(wpipeptr pip, word ch);

extern task main_tcb;
extern tasker_class tasker;

inline void tsk_cli(void) { disable(); }
inline void tsk_sti(void) { enable(); }
inline void tsk_outp(int port, byte b) { outportb(port, b); }
inline byte tsk_inp(int port) { return ((byte) inportb(port)); }

