/*
   CPPTask - A Multitasking Kernel For C++

   Version 1.0 08-12-91

   Ported by Rich Smith from:

   Public Domain Software written by
      Thomas Wagner
      Patschkauer Weg 31
      D-1000 Berlin 33
      West Germany

   TSKWPIP.CPP - Word Pipe handling routines.

   Subroutines:
       wpipe::wpipe
       wpipe::~wpipe
       wpipe::read_wpipe
       wpipe::c_read_wpipe
       wpipe::write_wpipe
       wpipe::c_write_wpipe
       wpipe::wait_wpipe_empty
       wpipe::check_wpipe
       wpipe::wpipe_free
       wpipe::flush_wpipe
       wpipe::get_bufsize
       wpipe::get_filled
       wpipe::get_flags

*/

#include <stdio.h>

#include "task.hpp"
#include "tsklocal.hpp"

/*
   tsk_getwpipe - get a word from a pipe. For internal use only.
                  Critical section assumed entered.
*/

word wpipe::tsk_getwpipe (void)
{
   word c;

   c = wcontents [outptr++];
   if (outptr >= bufsize)
      outptr = 0;
   filled--;
   return c;
}


/*
   tsk_putwpipe - put a word to a pipe. For internal use only.
                  Critical section assumed entered.
*/

void wpipe::tsk_putwpipe (word c)
{
   wcontents [inptr++] = c;
   if (inptr >= bufsize)
      inptr = 0;
   filled++;
}


/* -------------------------------------------------------------------- */


/*
   wpipe - initialises wpipe.
*/

wpipe::wpipe (farptr buf, word buffersize)
{
   flags = 0;

   wait_read = wait_write = wait_clear = NULL;
   outptr = inptr = filled = 0;
   bufsize = buffersize >> 1;
   wcontents = (wordptr)buf;
}


/*
   ~wpipe - kills all processes waiting for reading from or writing
                  to the wpipe.
*/

wpipe::~wpipe ()
{
   CRITICAL;

   C_ENTER;
   tsk_kill_queue (&wait_read);
   tsk_kill_queue (&wait_write);
   tsk_kill_queue (&wait_clear);
   outptr = inptr = filled = 0;
   C_LEAVE;

}


/*
   read_wpipe - Wait until a word is written to the wpipe. If there 
                is a word in the wpipe on entry, it is assigned to 
                the caller, and the task continues to run. If there are
                tasks waiting to write, the first task is made eligible,
                and the word is inserted into the wpipe.
*/

word far wpipe::read_wpipe (dword timeout)
{
   tcbptr curr;
   word res;
   CRITICAL;

   C_ENTER;

   if (filled)
      {
      res = tsk_getwpipe ();

      if ((curr = wait_write) != NULL)
         {
         tsk_putwpipe (curr->get_retsize());
         wait_write = curr->tsk_runable ();
         curr->set_retptr(NULL);
         }
      else if (!filled)
         while (wait_clear != NULL)
            wait_clear = wait_clear->tsk_runable ();

      C_LEAVE;
      return res;
      }

   tsk_wait (&wait_read, timeout);
   return (word)tsk_current->get_retptr();
}


/*
   c_read_wpipe - If there is a word in the wpipe on entry,
                  read_wpipe is called, otherwise an error status is returned.
*/

word far wpipe::c_read_wpipe (void)
{
   CRITICAL, res;

   C_ENTER;
   res = (filled) ? read_wpipe (0L) : (word)-1;
   C_LEAVE;
   return res;
}



/*
   write_wpipe - Wait until space for the word to be written to the 
                 wpipe is available. If there is enough space in the wpipe 
                 on entry, the word is inserted into the wpipe, and
                 the task continues to run. If there are tasks waiting 
                 to read, the first task is made eligible, and the word
                 is passed to the waiting task.
*/

int far wpipe::write_wpipe (word ch, dword timeout)
{
   tcbptr curr;
   CRITICAL;

   C_ENTER;

   if (filled < bufsize)
      {
      if ((curr = wait_read) != NULL)
         {
         wait_read = curr->tsk_runable ();
         curr->set_retptr((farptr)ch);
         C_LEAVE;
         return 0;
         }

      tsk_putwpipe (ch);
      C_LEAVE;
      return 0;
      }

   tsk_current->set_retsize(ch);
   tsk_wait (&wait_write, timeout);
   return (int)tsk_current->get_retptr();
}


/*
   c_write_wpipe - If there is space for the word in the wpipe on entry,
                   write_wpipe is called, otherwise an error status is returned.
*/

int far wpipe::c_write_wpipe (word ch)
{
   int res;
   CRITICAL;

   C_ENTER;
   res = (filled < bufsize) ? write_wpipe (ch, 0L) : -1;
   C_LEAVE;
   return res;
}


/*
   wait_wpipe_empty - Wait until the pipe is empty. If the pipe is
                      empty on entry, the task continues to run.
*/

int far wpipe::wait_wpipe_empty (dword timeout)
{
   CRITICAL;

   C_ENTER;
   if (!filled)
      {
      C_LEAVE;
      return 0;
      }

   tsk_current->set_retptr(NULL);
   tsk_wait (&wait_clear, timeout);
   return (int)tsk_current->get_retptr();
}


/*
   check_wpipe - returns -1 if there are no words in the wpipe, else
                 the first available word.
*/

word far wpipe::check_wpipe (void)
{
   return (filled) ? wcontents [outptr] : (word)-1;
}


/*
   wpipe_free - returns the number of free words in the pipe.
*/

word far wpipe::wpipe_free (void)
{
   return bufsize - filled;
}

/*
   flush_wpipe - Empty the pipe buffer, activate tasks waiting for 
                 pipe clear.
*/

void far wpipe::flush_wpipe (void)
{
   CRITICAL;

   C_ENTER;
   inptr = outptr = filled = 0;

   while (wait_clear != NULL)
      wait_clear = wait_clear->tsk_runable ();
   C_LEAVE;
}



word far asm_read_wpipe(wpipeptr pip, dword timeout)
{
    return pip->read_wpipe(timeout);
}

word far asm_check_wpipe(wpipeptr pip)
{
    return pip->check_wpipe();
}

int far asm_c_write_wpipe(wpipeptr pip, word ch)
{
    return pip->c_write_wpipe(ch);
}

