/*==================================================================*/
/*	CONFIG Version 1.0  <August 21, 1988>			    */
/*								    */
/*	Written by Conrad Kwok, Division of Computer Science, UCD   */
/*								    */
/*	Permission is granted for freely distribution of this file  */
/*		provided that this message is included.		    */
/*==================================================================*/

#include <stdio.h>
#include <ctype.h>
#include <varargs.h>
#include "config.h"

#ifdef MSDOS
#include "config.dcl"
#endif

#ifdef NoVPRINTF
#define vfprintf(fp,fmt,arg) _doprnt(fmt,arg,fp)
#endif

#define SigKeyStrLen 20
#define NuOfUserType 2
#define NoMore EOF
#define MinFloat -1.0e+38
#define MaxFloat  1.0e+38
#define MinDouble -1.0e+38
#define MaxDouble  1.0e+38
#define cfgMaxStrLen 512

#define mywhite(ch) (isspace(ch) || ch=='{' || ch=='}')

int cfgLine=0, cfgCol=0;
char StrQuChar = '"';
int KeyCaseSensitive = FALSE;
int cfgScan = FALSE;
int WasComma;
int cfgVerbose = FALSE;
char KeyStr[SigKeyStrLen+1];
int (*UserDefHook)()=NULL;
int (*UTypeHook[NuOfUserType])() = {
    NULL, NULL
};
FILE *cfgFile;

double cfgRange[8][2] = {
    { -128  , 127   },
    { 0     , 255   },
    { -32768, 32767 },
    { 0     , 65535 },
#ifdef INT32BIT
    { -2147483648.0, 2147483647.0 },
    { 0          , 4294967295.0 },
#else
    { -32768, 32767 },
    { 0     , 65535 },
#endif
    { -2147483648.0, 2147483647.0 },
    { 0          , 4294967295.0 } };

void *malloc();
KeyRec *searchkey();

int cfggetc()
{
    int ch;

reread:
    ch = getc(cfgFile);
    if (ch == '\n') {
        cfgLine++;
        cfgCol = 0;
    } else if (ch == EOF) {
        if (cfgCol!=0) {
            cfgLine++;
            cfgCol = 0;
        }
    } else if (ch=='#' || ch==';') {
        do {
            ch = getc(cfgFile);
        } while (ch!=EOF && ch!='\n');
        cfgLine++;
        cfgCol = 0;
        goto reread;
    } else {
        cfgCol++;
    }
    return(ch);
}

cfgfldc()
{
    int ch;

    ch=cfggetc();
    if (cfgCol == 1 && !isspace(ch)) {
        cfgungetc(ch);
        return(EOF);
    }
    return(ch);
}


cfgungetc(ch)
int ch;
{
    if (ch==EOF) return;
    cfgCol--;
    if (cfgCol < 0) {
        cfgLine--;
    }
    ungetc(ch, cfgFile);
}

#ifndef MSDOS
strnicmp(s, t, len)
char *s, *t;
int len;
{
    char a, b;

    for (; len > 0 ; len--) {
        a= *s++; b= *t++;
        a = islower(a) ? toupper(a) : a;
        b = islower(b) ? toupper(b) : b;
        if (a != b) break;
        if (a == '\0') return(0);
    }
    return(a - b);
}
#endif

readkey(str, strsize)
char *str;
int strsize;
{
    int lencnt, ch;

    lencnt=strsize-1;
    do {
        ch=cfggetc();
    } while (isspace(ch));
    if (ch==EOF) return(EOF);
    if (cfgCol != 1) {
        cfgerror("Configuration Keyword must begin at first column\n");
    }
    while (ch!=EOF && !isspace(ch)) {
        if (lencnt-- > 0) {
            *str++ = ch;
        }
        ch=cfggetc();
    }
    *str = NULL;
    if (ch == EOF && lencnt+1 != strsize) {
        cfgerror("Configuration File Ended Unexpectly\n");
    }
    return(ch);
}

#ifdef MSDOS
cfgerror(va_alist, ...)
#else
cfgerror(va_alist)
#endif
va_dcl
{
    char *fmt;
    va_list arg_ptr;

    va_start(arg_ptr);
    fmt = va_arg(arg_ptr, char *);
    fprintf(stderr,"CFG Error--L%d C%d:", cfgLine+1, cfgCol);
    vfprintf(stderr, fmt, arg_ptr);
    va_end(arg_ptr);
    exit(1);
}

#ifdef MSDOS
cfgwarning(va_alist, ...)
#else
cfgwarning(va_alist)
#endif
va_dcl
{
    char *fmt;
    va_list arg_ptr;

    va_start(arg_ptr);
    fmt = va_arg(arg_ptr, char *);
    fprintf(stderr,"CFG Warning--L%d C%d:", cfgLine+1, cfgCol);
    vfprintf(stderr, fmt, arg_ptr);
    va_end(arg_ptr);
}

readconfig(filename)
char *filename;
{
    KeyRec *idx;
    int tmp;

    if ((cfgFile=fopen(filename, "r")) == NULL) {
        return FALSE;
    }
    while (readkey(KeyStr, sizeof(KeyStr)) != EOF) {
        if ((idx=searchkey(KeyTable, KeyStr)) != NULL) {
            if (idx->hook != NULL) {
		if ((*idx->hook)(0, idx) == FALSE) {
		    continue;
		}
            }
            switch (idx->vtype) {
            case V_byte:
            case V_ubyte:
            case V_short:
            case V_ushort:
            case V_int:
            case V_uint:
            case V_long:
/*            case V_ulong: */
                readint(idx);
                break;
            case V_float:
            case V_double:
            	readreal(idx);
            	break;
            case V_string:
            case V_charptr:
            	readstr(idx);
            	break;
            case V_char:
            	readchar(idx);
            	break;
            case V_intkw:
                readintkw(idx);
                break;
            case V_usertype0:
            case V_usertype1:
                tmp = idx->vtype - V_usertype0;
                if (UTypeHook[tmp] != NULL) {
                    (*UTypeHook[tmp])(idx);
                    break;
                }
            default:
                if (UserDefHook == NULL || (*UserDefHook)(idx)==FALSE) {
		    cfgwarning("Unknown vtype %d\n", idx->vtype);
                }
                break;
            }
            if (idx->hook != NULL) {
		if ((*idx->hook)(1, idx) == FALSE) {
		    continue;
		}
            }
            junkcheck(idx->val_set == cfgPreset);
        } else {
            if (!cfgScan) {
                cfgwarning("Keyword '%s' *NOT* found\n", KeyStr);
            }
            junkcheck(FALSE);
        }
    }
    return(TRUE);
}

junkcheck(skip)
int skip;
{
    int ch, echo;

    echo = FALSE;
    do {
        if (echo) {
	    putc(ch, stderr);
        }
        do {
            ch=cfggetc();
            if (ch == EOF) return;
            if (echo && cfgCol != 1) {
		putc(ch, stderr);
	    } else if (cfgCol != 1 && !mywhite(ch)) {
                if (!skip) {
                    cfgwarning("The following data are ignored:\n");
                    echo = TRUE;
		    putc(ch, stderr);
		}
            }
        } while (cfgCol != 1);
    } while (isspace(ch));
    cfgungetc(ch);
    return;
}

readint(idx)
KeyRec *idx;
{
    long val;
    int loop, arraysz, ret, nettype;
    void *num;
    double lowerlimit, upperlimit;

    if (idx->val_set == cfgPreset) return;
    if (idx->val_set == TRUE) {
        cfgwarning("%s already defined\n", KeyStr);
        return;
    }
    WasComma = FALSE;
    arraysz = idx->arraysize;
    nettype = idx->vtype - V_byte;
    if (arraysz <= 0) arraysz = 1;
    num = idx->addr;
    lowerlimit = idx->lower;
    upperlimit = idx->upper;
    if (lowerlimit >= upperlimit) {
        lowerlimit=cfgRange[nettype][0];
        upperlimit=cfgRange[nettype][1];
    } else if (lowerlimit < cfgRange[nettype][0]) {
        cfgwarning("User range of %s < lower limit. Default Used\n",
                     KeyStr);
        lowerlimit = cfgRange[nettype][0];
    } else if (upperlimit > cfgRange[nettype][1]) {
        cfgwarning("Usere range of %s > upper limit. Default Used\n",
                     KeyStr);
        upperlimit = cfgRange[nettype][1];
    }
    for (loop=0; loop < arraysz; loop++) {
        if ((ret=readlong(&val)) == NoMore) {
            cfgerror("Insufficient Field\n");
        }
        if (ret==TRUE) {
            if (val < lowerlimit || val > upperlimit) {
                cfgerror("Field %d out of range (%lg to %lg)\n", loop+1,
                            lowerlimit, upperlimit);
            }
       	    if (cfgVerbose) {
       	        printf("CFG: %s(%d)=%ld\n", KeyStr, loop, val);
       	    }
            switch (idx->vtype) {
            case V_byte:
                *((char *) num) = val;
#ifdef OldFashion
		num += (sizeof (char *));
#else
                ((char *) num)++;
#endif
                break;
            case V_ubyte:
                *((unsigned char *) num) = val;
#ifdef OldFashion
		num += (sizeof (unsigned char *));
#else
                ((unsigned char *) num)++;
#endif
                break;
            case V_short:
                *((short *) num) = val;
#ifdef OldFashion
		num += (sizeof (short *));
#else
                ((short *) num)++;
#endif
                break;
            case V_ushort:
                *((unsigned short *) num) = val;
#ifdef OldFashion
		num += (sizeof (unsigned short *));
#else
                ((unsigned short *) num)++;
#endif
                break;
            case V_int:
                *((int *) num) = val;
#ifdef OldFashion
		num += (sizeof (int *));
#else
                ((int *) num)++;
#endif
                break;
            case V_uint:
                *((unsigned int *) num) = val;
#ifdef OldFashion
		num += (sizeof (unsigned int *));
#else
                ((unsigned int *) num)++;
#endif
                break;
            case V_long:
                *((long *) num) = val;
#ifdef OldFashion
		num += (sizeof (long *));
#else
                ((long *) num)++;
#endif
                break;
            case V_ulong:
                *((unsigned long *) num) = val;
#ifdef OldFashion
		num += (sizeof (unsigned long *));
#else
                ((unsigned long *) num)++;
#endif
                break;
            default:
               cfgerror("Almost Impossible Error. Unknown vtype");
               break;
            }
	} else {
	    if (idx->val_set != cfgDefault) {
	        cfgerror("Field %d in '%s' missing\n", loop+1, KeyStr);
	    } else {
	        switch (idx->vtype) {
                case V_byte:
#ifdef OldFashion
		    num += (sizeof (char *));
#else
                    ((char *) num)++;
#endif
                    break;
                case V_ubyte:
#ifdef OldFashion
		    num += (sizeof (unsigned char *));
#else
                    ((unsigned char *) num)++;
#endif
                    break;
                case V_short:
#ifdef OldFashion
		    num += (sizeof (short *));
#else
                    ((short *) num)++;
#endif
                    break;
                case V_ushort:
#ifdef OldFashion
		    num += (sizeof (unsigned short *));
#else
                    ((unsigned short *) num)++;
#endif
                    break;
                case V_int:
#ifdef OldFashion
		    num += (sizeof (int *));
#else
                    ((int *) num)++;
#endif
                    break;
                case V_uint:
#ifdef OldFashion
		    num += (sizeof (unsigned int *));
#else
                    ((unsigned int *) num)++;
#endif
                    break;
                case V_long:
#ifdef OldFashion
		    num += (sizeof (long *));
#else
                    ((long *) num)++;
#endif
                    break;
                case V_ulong:
#ifdef OldFashion
		    num += (sizeof (unsigned long *));
#else
                    ((unsigned long *) num)++;
#endif
                    break;
                default:
                   cfgerror("Almost Impossible Error. Unknown vtype");
                   break;
                }
            }
	}
    }
    idx->val_set = TRUE;
}

readlong(l)
long *l;
{
    int ch, nlen, neg;
    long temp;

    do {
        ch = cfgfldc();
    } while (mywhite(ch));
    neg = FALSE;
    if (ch == '-') {
    	neg = TRUE;
    	ch = cfgfldc();
    } else if (ch == '+') {
    	ch = cfgfldc();
    }
    temp = 0;
    nlen = 0;
    while (isdigit(ch)) {
        nlen++;
        temp = temp*10 + ch - '0';
        ch = cfgfldc();
    }
    while (mywhite(ch)) ch=cfgfldc();
    if (nlen==0) {
        if ((ch==EOF && WasComma) || ch==',') {
            return(FALSE);
        } else {
            return(NoMore);
        }
    } else {
	*l = neg ? -temp : temp;
        WasComma = (ch==',');
        if (!WasComma) {
            cfgungetc(ch);
        }
    }
    return(TRUE);
}

readreal(idx)
KeyRec *idx;
{
    double val;
    int loop, arraysz, ret;
    void *num;
    double lowerlimit, upperlimit;

    if (idx->val_set == cfgPreset) return;
    if (idx->val_set == TRUE) {
        cfgwarning("%s already defined\n", KeyStr);
        return;
    }
    WasComma = FALSE;
    arraysz = idx->arraysize;
    if (arraysz <= 0) arraysz = 1;
    num = idx->addr;
    lowerlimit = idx->lower;
    upperlimit = idx->upper;
    if (lowerlimit >= upperlimit) {
        switch (idx->vtype) {
        case V_float:
            lowerlimit = MinFloat;
            upperlimit = MaxFloat;
            break;
	case V_double:
            lowerlimit = MinFloat;
            upperlimit = MaxFloat;
            break;
        default:
            cfgerror("Almost impossible error. Unknown Type\n");
        }
    }
    for (loop=0; loop < arraysz; loop++) {
        if ((ret=readdouble(&val)) == NoMore) {
            cfgerror("Insufficient Field\n");
        }
        if (ret==TRUE) {
            if (val < lowerlimit || val > upperlimit) {
                cfgerror("Field %d out of range (%lg to %lg)\n", loop+1,
                            lowerlimit, upperlimit);
            }
       	    if (cfgVerbose) {
       	        printf("CFG: %s(%d)=%lf\n", KeyStr, loop, val);
       	    }
            switch (idx->vtype) {
            case V_float:
                *((float *) num) = val;
#ifdef OldFashion
                num += (sizeof (float *));
#else
                ((float *) num)++;
#endif
                break;
            case V_double:
                *((double *) num) = val;
#ifdef OldFashion
                num += (sizeof (double *));
#else
                ((double *) num)++;
#endif
                break;
            default:
               cfgerror("Almost Impossible Error. Unknown vtype");
               break;
            }
	} else {
	    if (idx->val_set != cfgDefault) {
	        cfgerror("Field %d in '%s' missing\n", loop+1, KeyStr);
	    }
	    switch (idx->vtype) {
	    case V_float:
#ifdef OldFashion
                num += (sizeof (float *));
#else
                ((float *) num)++;
#endif
                break;
            case V_double:
#ifdef OldFashion
                num += (sizeof (double *));
#else
                ((double *) num)++;
#endif
            default:
               cfgerror("Almost Impossible Error. Unknown vtype");
               break;
            }
	}
    }
    idx->val_set = TRUE;
}

readdouble(dbl)
double *dbl;
{
    int ch, nlen;
    char dblstr[80];
    double atof();

    do {
        ch = cfgfldc();
    } while (mywhite(ch));
    nlen=0;
    while (isdigit(ch) || ch=='-' || ch=='+' || ch=='e' || ch=='E' ||
		ch=='.') {
        dblstr[nlen++] = ch;
        ch = cfgfldc();
    }
    dblstr[nlen]=NULL;
    while (mywhite(ch)) ch=cfgfldc();
    if (nlen==0) {
        if ((ch==EOF && WasComma) || ch==',') {
            return(FALSE);
        } else {
            return(NoMore);
        }
    } else {
	*dbl = atof(dblstr);
        WasComma = (ch==',');
        if (!WasComma) {
            cfgungetc(ch);
        }
    }
    return(TRUE);
}

readstr(idx)
KeyRec *idx;
{
    char s[cfgMaxStrLen];
    int loop, arraysz, ret, slen;
    char *sptr;
    int minlen, maxlen;

    if (idx->val_set == cfgPreset) return;
    if (idx->val_set == TRUE) {
        cfgwarning("%s already defined\n", KeyStr);
        return;
    }
    WasComma = FALSE;
    arraysz = idx->arraysize;
    if (arraysz <= 0) arraysz = 1;
    sptr = idx->addr;
    minlen = idx->lower;
    maxlen = idx->upper;
    if (minlen < 0) minlen = 0;
    if (maxlen > cfgMaxStrLen) {
        cfgwarning("Max string length is %d (config error)\n", cfgMaxStrLen);
        maxlen = cfgMaxStrLen;
    }
    if (minlen > maxlen || (maxlen == 0 && minlen == 0) ) {
        minlen = 0;
        maxlen = cfgMaxStrLen;
    }
    for (loop=0; loop < arraysz; loop++) {
        if ((ret=readstring(s, sizeof(s), StrQuChar)) == NoMore) {
            cfgerror("Insufficient Field\n");
        }
        if (ret==TRUE) {
            slen = strlen(s);
            if (slen+1 > maxlen) {
                cfgerror("Field %d too long (%d)\n", loop+1, maxlen);
            }
            if (cfgVerbose) {
                printf("CFG: %s(%d)=%s\n", KeyStr, loop, s);
            }
            switch (idx->vtype) {
            case V_string:
                strcpy(sptr, s);
                sptr += maxlen;
                break;
            case V_charptr:
                if (slen == 0 && minlen <= 0) {
                    *((char **) sptr) = NULL;
                } else {
                    if (slen < minlen) slen = minlen;
                    if ((*((char **) sptr) = malloc(slen)) == NULL) {
                        cfgerror("Error in allocating memory for string\n");
                    }
                    strcpy( *((char **) sptr), s);
#ifdef OldFashion
                    sptr += (sizeof (char **));
#else
                    ((char **) sptr)++;
#endif
                }
                break;
            }
        } else {
            if (idx->val_set != cfgDefault) {
                cfgerror("Field %d in '%s' missing\n", loop+1, KeyStr);
	    }
            switch(idx->vtype) {
            case V_string:
                sptr += maxlen;
                break;
            case V_charptr:
#ifdef OldFashion
                    sptr += (sizeof (char **));
#else
                    ((char **) sptr)++;
#endif
                break;
            }
        }
    }
    idx->val_set = TRUE;
}

readstring(str, totalsize, qchar)
char *str;
int totalsize;
char qchar;
{
    static char special_ch[26] = {
           0, '\b',    0,    0,    0, '\f',   7,   0,
           0,    0,    0, '\f',    0, '\n',   0,   0,
           0, '\r',    0, '\t',    0,    0,   0,   0,
           0,    0};
    int ch, slen, quoted, done;
    char ch1;

    do {
        ch =cfgfldc();
    } while (mywhite(ch));
    slen=0;
    done = FALSE;
    quoted= ch == qchar;
    if (quoted) ch = cfgfldc();
    while (!done) {
        if (ch=='\\') {
            ch = cfgfldc();
            if (ch==EOF) {
                if (quoted) {
                    cfgwarning("Missing closed quote!\n");
                }
                done =TRUE;
                ch = '\\';
            } else if (isalpha(ch)) {
                    ch1 = islower(ch) ? toupper(ch) : ch;
                    ch1 = special_ch[ch1-'A'];
                    if (ch1 != NULL) ch=ch1;
            }
        } else if ((quoted && ch==qchar) ||
                   (!quoted && (ch==',' || mywhite(ch)))) {
            ch=cfgfldc();
	    break;
        }
        if (ch == EOF) {
            if (quoted) {
                cfgwarning("Missing closed quote!\n");
            }
            break;
        }
        if (++slen < totalsize) {
            *str++ = ch;
        }
        ch = cfgfldc();
    }
    *str = NULL;
    while (mywhite(ch)) ch=cfgfldc();
    if (slen==0 && !quoted) {
        if ((ch==EOF && WasComma) || ch==',') {
            return(FALSE);
        } else {
            return(NoMore);
        }
    } else {
        WasComma = (ch==',');
        if (!WasComma) {
            cfgungetc(ch);
        }
    }
    return(TRUE);
}

readchar(idx)
KeyRec *idx;
{
    char s[3];
    int loop, arraysz, ret, slen;
    char *chptr;

    if (idx->val_set == cfgPreset) return;
    if (idx->val_set == TRUE) {
        cfgwarning("%s already defined\n", KeyStr);
        return;
    }
    WasComma = FALSE;
    arraysz = idx->arraysize;
    if (arraysz <= 0) arraysz = 1;
    chptr= idx->addr;
    for (loop=0; loop < arraysz; loop++) {
        if ((ret=readstring(s, sizeof(s), '\'')) == NoMore) {
            cfgerror("Insufficient Field\n");
        }
        if (ret==TRUE) {
            slen = strlen(s);
            if (slen > 1) {
                cfgerror("Only 1 character is allowed in char constant\n");
            }
            if (cfgVerbose) {
                printf("CFG: %s(%d)=%c\n", KeyStr, loop, s[0]);
            }
            *chptr = s[0];
            chptr++;
        } else {
            if (idx->val_set != cfgDefault) {
                cfgerror("Field %d in '%s' missing\n", loop+1, KeyStr);
            }
            chptr++;
        }
    }
    idx->val_set = TRUE;
}

readintkw(idx)
KeyRec *idx;
{
    char s[SigKeyStrLen+1];
    int loop, arraysz, ret, slen;
    int *val;

    if (idx->val_set == cfgPreset) return;
    if (idx->val_set == TRUE) {
        cfgwarning("%s already defined\n", KeyStr);
        return;
    }
    WasComma = FALSE;
    arraysz = idx->arraysize;
    if (arraysz <= 0) arraysz = 1;
    val = idx->addr;
    for (loop=0; loop < arraysz; loop++) {
        if ((ret = readstring(s, sizeof(s), '\0')) == NoMore) {
            cfgerror("Insufficient Field\n");
        }
        if (ret == TRUE) {
            ret=searchkw((char **) idx->userdata, s,
                            SigKeyStrLen, idx->userflag);
            if (ret < 0) {
                cfgerror("Keyword %s not found in table of %s\n", s, KeyStr);
            }
            if (cfgVerbose) {
                printf("CFG: %s(%d)=%d+%d [%s]\n", KeyStr, loop, ret,
                            (int) idx->lower, s);
            }
            *val = ret + (int) idx->lower;
            val++;
        } else {
            if (idx->val_set != cfgDefault) {
                cfgerror("Field %d in '%s' missing\n", loop+1, KeyStr);
	    }
            *val++;
        }
    }
    idx->val_set = TRUE;
}

int searchkw(table, s, sig, cs)
char *table[];
char *s;
int sig, cs;
{
    int count;

    count = 0;
    for ( ; *table != NULL; table++, count++) {
        if (cs) {
            if (strncmp(*table, s, sig) == 0) {
                return count;
            }
        } else {
            if (strnicmp(*table, s, sig) == 0) {
                return count;
            }
        }
    }
    return(-1);
}

KeyRec *searchkey(table, str)
KeyRec *table;
char *str;
{
    for ( ;table->keystr != NULL; table++) {
    	if (KeyCaseSensitive) {
	    if (strncmp(table->keystr, str, SigKeyStrLen) == 0) {
	        return(table);
	    }
	} else {
	    if (strnicmp(table->keystr, str, SigKeyStrLen) == 0) {
	        return(table);
	    }
	}
    }
    return(NULL);
}

int PresetKey(key)
char *key;
{
    KeyRec *idx;

    if ((idx=searchkey(KeyTable, key)) != NULL) {
        idx->val_set = cfgPreset;
	return TRUE;
    } else {
	return FALSE;
    }
}

CheckAllKeys()
{
    KeyRec *idx;
    int flag = TRUE;

    for(idx=KeyTable; idx->keystr != NULL; idx++) {
        if (idx->val_set == FALSE) {
            flag = FALSE;
            fprintf(stderr, "WARNING: Key %s undefined\n", idx->keystr);
        }
    }
    return flag;
}
