#if !defined( __COMSTR_H )
#define __COMSTR_H

#if !defined( __IOSTREAM_H )
#include <Iostream.h>
#endif

#if !defined( __STOPWATCH_H )
#include "Stopwatch.h"
#endif

// definitions for parity types.

#define NONE	0
#define ODD		1
#define EVEN	2

// definitions for RS232 control lines.

#define	DTR			0x01
#define RTS			0x02

// definitions for errors.

#define TIMEOUT		0x01
#define OR			0x02
#define PE			0x04
#define FE			0x08
#define BI			0x10

// Structure
//
//	CommFormat
//
// Purpose
//
//	Holds communication format information. This includes: baud rate,
//	bits per byte, parity, and the number of stop bits. When used with
//	the CommPort class, the range of these values should be:
//		baud		50L .. 115200L
//		bits		6 .. 8
//		parity  	NONE, EVEN, or ODD
//		stopbits    1 or 2

struct CommFormat
	{
	long baud;
	int bits, parity, stopbits;

	CommFormat( long, int, int, int );
	};

inline CommFormat::CommFormat( long aBaud, int aBits, int aParity, int aStopBits ) :
	baud(aBaud), bits(aBits), parity(aParity), stopbits(aStopBits)
	{}

// forward class reference for use in class CommPort

class combuf;

// Class
//
//	CommPort (Abstract)
//
// Purpose
//
//  Provides the (mostly virtual) methods required to open, transmit,
//	receive, and close a serial communications port.
//
// Methods
//
//	CommPort()		constructor
//
//	~CommPort()		virtual destructor
//
//	int open( combuf *cb, long baud, int bits, int parity, int sbits )
//
//		Calls the pure virtual methods:
//		setBaud( baud )
//		setBits( bits )
//		setParity( parity )
//		setStopBits( sbits )
//		open( cb )
//		Returns:	0 for failure, non-zero for ok.
//
//	int open( combuf *cb )			pure virtual
//
//		Opens the port for communication and initializes it as necessary.
//		Returns:	0 for failure, non-zero for ok.
//
//	int close()						pure virtual
//
//		Closes the port, cleans up as necessary.
//		Returns:	0 for failure, non-zero for ok.
//
//	int	okToSend()					pure virtual
//
//      Checks whether port is prepared to IMMEDIATELY transmit a char.
//		Should check both the port status AND handshaking status.
//		Returns:	0 for not ready, non-zero for ready.
//
//	int sendChar( int ch )			pure virtual
//
//		Transmits ch imediately. Note that this function should NOT wait
//		around for some conditions to be met (that's what okToSend() is
//		for).
//		Returns:	0 on failure, non-zero on ok.
//
//	int assertLine( int pattern )	pure virtual
//
//		Asserts the RS232 control line(s) in pattern.
//		Returns:	0 on failure, non-zero on ok.
//
//  int unassertLine( int pattern )	pure virtual
//
//		Unasserts the RS232 control line(s) in pattern.
//		Returns:	0 on failure, non-zero on ok.
//
//	int setFormat( CommFormat& cf )
//
//		Calls the pure virtual methods:
//		setBaud( cf.baud )
//		setBits( cf.bits )
//		setParity( cf.parity )
//		setStopBits( cf.stopbits )
//		Returns:	0 on failure, non-zero on ok.
//
//  int setBaud( long baud )		pure virtual
//
//		Sets the port baud rate to baud.
//		Returns:	0 on failure, non-zero on ok.
//
//	int setParity( int parity )		pure virtual
//
//		Sets the port parity to parity. Arg parity can be one of:
//		NONE (0), ODD (1), or EVEN (2).
//		Returns:	0 on failure, non-zero on ok.
//
//  int setBits( int bits )			pure virtual
//
//		Sets the bits-per-byte for the port. Arg bits can be one of:
//		6, 7, or 8.
//		Returns:	0 on failure, non-zero on ok.
//
//	int setStopBits( int sbits )	pure virtual
//
//		Sets the number of stop bits for the port. Arg sbits can be one
//		of: 1 or 2.
//		Returns:	0 on failure, non-zero on ok.
//
//  void getFormat( CommFormat& cf )
//
//		Performs the following:
//		cf.baud = getBaud()
//		cf.bits = getBits()
//		cf.parity = getParity()
//		cf.stopbits = getStopBits()
//
//  long getBaud()					pure virtual
//
//		Returns the current baud rate of the port.
//
//  int getParity()					pure virtual
//
//		Returns the current parity setting of the port. Should be one of:
//		NONE (0), ODD (1) or EVEN (2).
//
//	int getBits()					pure virtual
//
//		Returns the current number of bits-per-byte for the port. Should be
//		one of: 6, 7 or 8.
//
//	int getStopBits()				pure virtual
//
//		Returns the current number of stop bits for the port. Should be one
//		of: 1 or 2.
//
//  int errStatus()
//
//		Returns and clears the errFlags member. When an error occurs,
//		member functions (methods) should set the appropriate bits in
//		errFlags. Defined errors are:
//		TIMEOUT:	Timeout condition has occurred.
//		OR:			Overrun (bytes sent/received too fast) has occurred.
//      PE:			Parity error has occurred.
//		FE:			Framing error has occurred.
//		BI:			Break indicator (port has received a BREAK condition).

class CommPort
	{
	public:
		CommPort();
virtual	~CommPort();

		int open( combuf *, long, int, int, int );
virtual int open( combuf * ) = 0;
virtual	int close() = 0;

virtual int okToSend() = 0;
virtual	int sendChar( int ) = 0;
virtual int assertLine( int ) = 0;
virtual int unassertLine( int ) = 0;

		int setFormat( CommFormat& );
virtual	int setBaud( long ) = 0;
virtual int setParity( int ) = 0;
virtual int setBits( int ) = 0;
virtual int setStopBits( int ) = 0;

		void getFormat( CommFormat& );
virtual long getBaud() = 0;
virtual int getParity() = 0;
virtual int getBits() = 0;
virtual int getStopBits() = 0;

		int errStatus();

	protected:
		int errFlags;
		combuf *parent;
	};

// Class
//
//	StdPort		Inherits from class CommPort
//
// Purpose
//
//  Defines a usable CommPort-derived class based on the standard PC
//	serial ports (COM1..4).
//
// Methods
//
//  StdPort( int portNr )
//
//		Initializes for a standard PC port. Arg portNr is one of the
//		following: 0 = COM1, 1 = COM2, 2 = COM3, 3 = COM4. Reads the
//		port address (member port) from the BIOS memory area. Sets
//		member irq (the IRQ interrupt number) as follows: 0x0C for
//		COM1 or COM3, 0x0B for COM2 or COM4. Sets the member picMask
//		(bit mask for enable/disable of the PIC) as follows: 0x10 for
//		COM1 or COM3, 0x08 for COM2 or COM4. Finally checks the port
//		to ensure it exists.
//
//	~StdPort()
//
//		Calls close().
//
//  int open( combuf *cb )			virtual
//
//		Checks to ensure port is valid (member port != 0), that there are
//		no port conflicts (COM1 and COM3 {or COM2 and COM4} cannot be
//		open at the same time). Next installs the correct interrupt
//		service routine (handler0x0C() for COM1/3, handler0x0B for COM2/4).
//		Initializes the UART.
//
//  int close()						virtual
//
//		If open, closes the port.
//
//  int okToSend()					virtual
//
//		Returns 1 if the port is open and the UARTs transmitter buffer
//		is empty; else 0.
//
//  int sendChar( int ch )			virtual
//
//      If the port is open and the UARTs transmitter buffer is empty,
//		sends the character (writes to port+0) and returns 1; else
//		returns 0.
//
//  int assertLine( int pattern )	virtual
//
//		Asserts pattern by OR'ing it with what's currently in the Modem
//		Control Register and writing it out.
//
//	int unassertLine( int pattern )	virtual
//
//		Unasserts pattern by anding (~pattern) with what's currently in
//		the Modem Control Register and writing it out.
//
//  void setDtrOnClose( int i )
//
//		For some purposes (and mine during testing), you want to manipulate
//		the port but leave DTR asserted after closing (otherwise the modem
//		will hangup and break connection). By default, StdPort will unassert
//		DTR when it closes - this method allows you to change that so that
//		DTR will remain asserted after closing.
//
//  int setBaud( long baud )		virtual
//
//		If the port is valid (but not necessarily open), sets the baud rate
//		to baud and returns 1; else returns 0. Only works for baud rates
//		from 50 to 115200. Also fails if the alternate port (ie this is
//		COM1, the alternate is COM3) is open (would change THAT port's
//		baud.
//
//  int setParity( int parity )		virtual
//
//		If the port is valid (but not necessarily open), sets the parity
//		to parity and returns 1; else returns 0. Only works for parity
//		values of: 0 (no parity), 1 (odd parity) or 2 (even parity). Also
//		fails if the alternate port is open.
//
//	int setBits( int bits )			virtual
//
//		If the port is valid (but not necessarily open), sets the bits/byte
//		to bits and returns 1; else returns 0. Only works for bits of 6..8.
//		Also fails if the alternate port is open.
//
//  int setStopBits( int sbits )	virtual
//
//		If the port is valid (but not necessarily open), sets the stop bits
//		to sbits and returns 1; else returns 0. Only works for sbits of 1/2.
//		Also fails if the alternate port is open.
//
//  long getBaud()					virtual
//
//		If the port is valid (but not necessarily open), reads and returns
//		the UARTs set baud rate; else returns 0.
//
//  int getParity()					virtual
//
//		If the port is valid (but not necessarily open), reads the parity
//		from the UART and converts it to one of: NONE (0), ODD (1) or EVEN
//		(2); else returns -1.
//
//  int getBits()					virtual
//
//		If the port is valid (but not necessarily open), reads and returns
//		the bits/byte from the UART; else returns 0.
//
//	int getStopBits()				virtual
//
//		If the port is valid (but not necessarily open), reads and returns
//		the numer of stop bits from the UART; else returns 0.
//
//  void irqHandler()
//
//		When the UART generates an interrupt, the CPU will call either
//		static member handler0x0C() (for COM1/3) or handler0x0B (for COM2/4).
//		These Interrupt Service Routines (ISR) will then call this method
//		for the open StdPort object. This method services the interrupt
//		by performing one of the following:
//			CAUSE OF INT		RESPONSE
//			Received a byte     Read the byte, insert it into the combuf.
//			Transmitter empty	Extracts a byte from the combuf; if its NOT
//								EOF (combuf empty), transmits the byte.
//          Modem status		Reads the UART Modem Status Register.
//			Line status			Reads the UART Line Status Register, then
//								sets errFlag bits as appropriate.
//
//	void interrupt far handler0x0B	static
//	void interrupt far handler0x0C	static
//
//		When the UART for COM1/3 generates an interrupt, handler0x0C is
//		called by the CPU (handler0x0B for COM2/4). If this port is open
//		(member owner[0] != 0 for COM2/4, owner[1] != 0 for COM1/3), this
//		ISR calls owner[0/1]->irqHandler() to service the interrupt. Since
//		these ISRs have to be static members to use them as interrupt
//		handlers, they don't know which instantiation of StdPort has
//		'opened' that port - thus the use of the member owner[]. But if
//		there was only one handler and, say, you had both COM1 and COM2
//		open, it would have to go through extra checking (time) to find
//		out which port caused the interrupt. Thus I use handler0x0C to
//		handle COM1/3 interrupts (only one can be in use at a time), and
//		handler0x0B for COM2/4.

class StdPort : public CommPort
	{
	public:
		StdPort( int );
		~StdPort();

		int open( combuf * );
		int close();

		int okToSend();
		int sendChar( int );
		int assertLine( int );
		int unassertLine( int );
		void setDtrOnClose( int );

		int setBaud( long );
		int setParity( int );
		int setBits( int );
		int setStopBits( int );

		long getBaud();
		int getParity();
		int getBits();
		int getStopBits();

	protected:
		int port, irq, picMask;
		int oldIER, oldMCR, closeMCR;
		void interrupt far (*oldhandler)(...);
static	StdPort *owner[];

		void irqHandler();
static	void interrupt far handler0x0B(...);
static	void interrupt far handler0x0C(...);
	};

inline void StdPort::setDtrOnClose( int i )
	{
	closeMCR = (i != 0) ? 1 : 0;
	}

// Class
//
//	combuf		inherited from class streambuf
//
// Purpose
//
//	Defines a streambuf-derived class that works with the class CommPort
//	to allow stream operations on a serial port. Since CommPort is an
//	abstract class (contains pure virtual methods), you cannot use it
//	directly with combuf; but you CAN use either StdPort (for normal
//	COM1..4 usage) or derive a class from CommPort that uses a third-party
//	serial communications board. If this level of abstraction slows down
//	communications too badly, you will have to forgo using these classes
//	and create new ones. But this shouldn't be a problem except with
//	slow computers using fast baud rates.
//
// Methods
//
//  combuf( CommPort *cp )			constructor
//
//      Instantiates a combuf object with a buffer of 0 bytes, using a
//		CommPort *cp. Not overly usable unless the setbuf method(s) are
//		called to set a buffer.
//
//	combuf( CommPort *cp, char *buf, int iSize, int oSize )	constructor
//
//		Instantiates a combuf object with a buffer at buf, the input size
//		at iSize (byte), the output size of oSize (bytes), using a CommPort
//		*cp. For downloading or normal operations, I'd recommend a larger
//		input buffer size (iSize) than output (oSize) to protect from
//		overflow errors on input. For uploading, you might want a large
//		output buffer (oSize) - this way you might be able to put the
//		whole output string in the buffer and not worry about it.
//
//	~combuf()						destructor
//
//      Closes the CommPort (calls server->close()).
//
//  streambuf *setbuf( signed char *buf, int len )	virtual
//
//		Initializes the input/output buffer to buf. Since combuf actually
//		uses 2 seperate areas of this buffer for input and output, it sets
//		the input size to len/2 and the output size to len - len/2.
//
//  combuf *setbuf( char *buf, int iSize, int oSize )
//
//		Initializes the input/output buffer to buf. Since combuf actually
//		uses 2 seperate areas of this buffer for intput and output, the
//		input size is set to iSize and the output size is set to oSize.
//		Note that the full size of buf MUST be iSize+oSize.
//
//  int underflow()						virtual
//
//		streambuf methods call this method whenever you try to extract
//		from combuf and the input buffer is empty (you haven't received the
//		byte from the serial port yet). Waits the timeout period for a
//		byte to be received; if one is received before timeout, it is
//		returned, else errFlags is set to TIMEOUT and EOF is returned.
//
//  int do_sputn( char *buf, int len )	virtual
//
//      streambuf methods call this method whenever you try to insert
//		a string to the buffer but its full (or its empty). There is an
//		important concept going on here; for interrupt-driven output to
//		the serial port, you have to 'prime the pump' - that is, when you
//		send the last character from the buffer, the UART will generate
//		a TRANSMITTER-EMPTY interrupt and the CommPort will try to get the
//		next character to send. Since the output buffer is empty, the
//		interrupt returns and that's it. When you put another string in the
//		output buffer, it will just sit there - the UART won't generate
//		another TRANSMITTER-EMPTY interrupt since it has already done so.
//		So combuf must transmit the first character manually - the rest will
//		be extracted automatically by CommPort. So 2 operations are crucial
//		for interrupt-driven output to work: 1) combuf must 'prime the pump'
//		for the first byte sent to an empty buffer, and 2) combuf must
//		indicate an empty buffer when CommPort extracts the last character
//		in the output buffer. But it can't JUST set the buffer as empty
//		(that's part of the streambuf methods); otherwise, when you write
//		to the buffer streambuf says "there's plenty of room!" and just puts
//		the characters in the buffer - without priming the pump. So the
//		method extractc() (called by CommPort) must set the buffer to NULL
//		when the last char is extracted - this forces streambuf to call
//		this method to output the new string.
//		When this method is called and the output buffer is set to NULL,
//		buf+1 (len-1 bytes) is inserted into the output buffer and *buf is
//		sent out the port to prime the pump (server->sendChar(*buf)). If
//		this method is called and there's already data in the output buffer
//		(just not enough room to hold len bytes), streambuf::do_sputn is
//		called.
//
//  int overflow( int ch = EOF )		virtual
//
//		Calls do_sputn( (char *)&ch, 1 ).
//
//  int open()
//
//      Initializes the buffer pointers (see class streambuf) and returns
//		the value from server->open( this ).
//
//  int close()
//
//		Returns the value from server->close().
//
//  CommPort *portServer()
//
//		Returns the CommPort member's address. This allows the programmer
//		to utilize the Commport methods. For example:
//			cbuf.portServer()->setBaud( 2400 );
//
//  int status()
//
//		Returns (and clears) the combuf/CommPort errFlags.
//
//  void setTimeout( long t )
//
//      Sets the timeout period to t (in 1/100ths of a second). Note that
//		this value is only accurate to +/- 55 mSec.
//
//  int extractc()
//
//		This method is used by CommPort to get the next character to be
//		sent out the serial port when a TRANSMITTER-EMPTY interrupt has
//		been generated ( see CommPort::irqHandler ).
//
//  void insertc()
//
//		This method is used by CommPort to insert a character that has been
//		received from the serial port when a RECEIVER-FULL interrupt has
//		been generated ( see CommPort::irqHandler ).
//
//  unsigned long timeToTicks( long l )	static
//
//		Converts a time l (in 1/100ths of a second) to timeticks (1/18.2 sec).


class combuf : public streambuf
	{
	public:
		combuf( CommPort * );
		combuf( CommPort *, char *, int, int );
		~combuf();

		streambuf *setbuf( signed char *, int );
		combuf *setbuf( char *, int, int );

		int underflow();

		int do_sputn( const char *, int );
		int overflow(int = EOF);

		int open();
		int close();

		CommPort *portServer();

		int status();
		void setTimeout( long );

		int extractc();
		void insertc( int );

	protected:
		CommPort *server;
		int errFlags;
		int inSize, outSize;
		unsigned long timeout_;

static	unsigned long timeToTicks( long );
	};

inline int combuf::overflow( int c )
	{
	if( c != EOF )
		{
		char b = c;
		return do_sputn( &b, 1 );
		}
	return EOF;
	}

inline int combuf::status()
	{
	int n = errFlags | server->errStatus();
	errFlags = 0;
	return n;
	}

inline CommPort *combuf::portServer()
	{
	return server;
	}

#endif	// __COMSTR_H
