#if !defined( __COMSTR_H )
#include "Comstr.h"
#endif

#if !defined( __DOS_H )
#include <Dos.h>
#endif

// definitions for UART registers.

#define IER	1
#define IIR	2
#define LCR	3
#define MCR	4
#define LSR	5
#define MSR	6

// definitions for UART interrupt cause identification.

#define MODEMSTAT	0x00
#define XMTEMPTY	0x02
#define RCVFULL		0x04
#define LINEFAIL	0x06

// definitions for UART line control bits.

#define BREAKCTRL	0x40
#define DLAB		0x80

// definitions for UART modem control bits.

#define GP01		0x04
#define GP02		0x08
#define LOOPBACK	0x10

// definitions for UART line status bits.

#define	RxRDY		0x01
#define TBE			0x20
#define TxE			0x40

// definitions for UART modem status bits.

#define dCTS		0x01
#define dDSR		0x02
#define dRI			0x04
#define dDCD		0x08
#define CTS			0x10
#define DSR			0x20
#define RI			0x40
#define DCD			0x80

// macros for manipulating port register bits.

#define ClearPortBit(x,y) (outportb((x),inportb(x)&(0xFF^((y)&0xFF))))
#define SetPortBit(x,y)   (outportb((x),inportb(x)|(y)))
#define GetPortBit(x,y)	  (inportb(x) & (y))

//-------------------------------------------------------------
// class COMMPORT	constructor
//-------------------------------------------------------------

CommPort::CommPort()
	{
	// clear errors.
	errFlags = 0;

	// set as not 'owned' (in use).
	parent = 0;
	}

//-------------------------------------------------------------
// class COMMPORT	destructor	virtual
//-------------------------------------------------------------

CommPort::~CommPort()
	{}

//-------------------------------------------------------------
// class COMMPORT	method 	open( combuf *, long, int, int, int )
//-------------------------------------------------------------

int CommPort::open( combuf *p, long baud, int bits, int parity, int stopbits )
	{
	// initlialize flag.
	int result = 1;

	// set result to 0 if any format call fails.
	result *= setBaud( baud );
	result *= setBits( bits );
	result *= setParity( parity );
	result *= setStopBits( stopbits );

	// fail if unable to set format.
	if( result == 0 )	return 0;

	// return open( combuf *) result.
	return open( p );
	}

//-------------------------------------------------------------
// class COMMPORT	method 	setFormat( CommFormat& )
//-------------------------------------------------------------

int CommPort::setFormat( CommFormat& f )
	{
	// Save the original format.
	CommFormat oldf(getBaud(),getBits(),getParity(),getStopBits());

	// result maintains bit flags for which format call passes.
	int result = 0;
	if( setBaud( f.baud ) != 0 )			result |= 1;
	if( setBits( f.bits) != 0 )				result |= 2;
	if( setParity( f.parity ) != 0 )    	result |= 4;
	if( setStopBits( f.stopbits ) != 0 )    result |= 8;

	// if bits 0..3 are set, everyone passes.
	if( result == 15 )
		return 1;

	// Some failed - restore their original values.
	if( (result & 1) == 0 )	setBaud( oldf.baud );
	if( (result & 2) == 0 )	setBits( oldf.bits );
	if( (result & 4) == 0 )	setParity( oldf.parity );
	if( (result & 8) == 0 )	setStopBits( oldf.stopbits );
	return 0;
	}

//-------------------------------------------------------------
// Class COMMPORT	method	getFormat( CommFormat& )
//-------------------------------------------------------------

void CommPort::getFormat( CommFormat& f )
	{
	f.baud = getBaud();
	f.bits = getBits();
	f.parity = getParity();
	f.stopbits = getStopBits();
	}

//-------------------------------------------------------------
// Class COMMPORT	method	errStatus()
//-------------------------------------------------------------

int CommPort::errStatus()
	{
	// store flags temporarily.
	int n = errFlags;

	// clear flags.
	errFlags = 0;
	return n;
	}


//-------------------------------------------------------------
// Class STDPORT	member	owner[]		static
//-------------------------------------------------------------

StdPort *StdPort::owner[2] = { 0, 0 };

//-------------------------------------------------------------
// Class STDPORT	method	handler0x0B		static
//-------------------------------------------------------------

void interrupt far StdPort::handler0x0B(...)
	{
	// get pointer to StdPort that 'owns' this port.
	StdPort *p = owner[0];

	// call the owners irqHandler method.
	if( p != 0 )
		p->irqHandler();
	else

	// no owner - signal End Of Interrupt to the Programmable Int Controller.
		outportb( 0x20, 0x20 );
	}

//-------------------------------------------------------------
// Class STDPORT	method	handler0x0C		static
//-------------------------------------------------------------

void interrupt far StdPort::handler0x0C(...)
	{
	// get pointer to StdPort that 'owns' this port.
	StdPort *p = owner[1];

	// call the owners irqHandler method.
	if( p != 0 )
		p->irqHandler();
	else

	// no owner - signal EOI to the PIC.
		outportb( 0x20, 0x20 );
	}

//-------------------------------------------------------------
// Class STDPORT	method		constructor
//-------------------------------------------------------------

StdPort::StdPort( int comNr )
	{
	// by default, the MCR register is set to 0 on exit.
	closeMCR = 0;

	// comNr only valid for 0..3 (COM1..4).
	if( comNr >= 0 && comNr < 4 )
		{
		// Read port address from BIOS memory area.
		unsigned far *pPort = (unsigned far *) MK_FP( 0x40, 0 );
		port = pPort[comNr];

		// set irq/picMask to defaults for this port.
		irq = (comNr & 1) ? 0x0B : 0x0C;
		picMask = (comNr & 1) ? 0x08 : 0x10;

		// If the port's in use, it must be ok. If not,
		// test to ensure there's a UART.
		if( owner[irq-0x0B] == 0 )
			{
			int tempMCR = inportb(port+MCR);

			// Set the port to loopback mode with test pattern 5 in
			// the lower nybble.
			outportb( port+MCR, LOOPBACK | 5 );

			// Read the status register and ensure that we echo a
			// pattern of Ah in the upper nybble.
			if( (inportb(port+MSR) & 0xF0) != 0x60 )
				port = irq = picMask = 0;
			else
				{
				// Do the same test with pattern Ah.
				outportb( port+MCR, LOOPBACK | 0x0A );
				if( (inportb(port+MSR) & 0xF0) != 0x90 )
					port = irq = picMask = 0;
				else
					outportb( port+MCR, tempMCR );
				}
			}

		}
	else
		port = irq = picMask = 0;

	// initialize pointer to original ISR to 0.
	oldhandler = 0;
	}

//-------------------------------------------------------------
// Class STDPORT	method	destructor
//-------------------------------------------------------------

StdPort::~StdPort()
	{
	// close the port - this will perform appropriate cleanup.
	close();
	}

//-------------------------------------------------------------
// Class STDPORT	method		open( combuf * )
//-------------------------------------------------------------

int StdPort::open( combuf *p )
	{
	// Fail if: the pointer p is NULL, the port = 0 (invalid),
	// this serial port is in use.
	if( p == 0 || port == 0 || irq == 0 || picMask == 0 ||
			( irq == 0x0B && owner[0] != 0 ) ||
			( irq == 0x0C && owner[1] != 0 ) )
		return 0;

	// save the combuf 'parent'.
	parent = p;

	// set owner[] to this to show that we are using it.
	owner[irq-0x0B] = this;

	// save the original ISR handler for this interrupt.
	oldhandler = getvect( irq );

	// save the original Interrupt Enable and Modem Control registers.
	oldIER = inportb(port+IER);
	oldMCR = inportb(port+MCR);

	// disable interrupts from the UART.
	outportb( port+MCR, 0 );
	outportb( port+IER, 0 );

	// clear BREAK condition and DLAB bit in UART.
	outportb( port+LCR, inportb(port+LCR) & 0x3F );

	// clear pending interrupts by reading all UART registers.
	for( int i = 0; i < 7; i++ )
		inportb( port+i );

	// install the appropriate ISR.
	setvect( irq, ((irq == 0x0B) ? handler0x0B : handler0x0C) );

	// enable this interrupt through the PIC.
	ClearPortBit( 0x21, picMask );

	// enable UART to talk to the PIC.
	SetPortBit( port+MCR, GP02 );

	// enable all UART interrupts.
	outportb( port+IER, 0x0F );
	return 1;
	}

//-------------------------------------------------------------
// Class STDPORT	method		close()
//-------------------------------------------------------------

int StdPort::close()
	{
	// fail if invalid or we don't own the port.
	if( irq == 0 || owner[irq-0x0B] != this )
		return 0;

	// disable this interrupt in the PIC.
	SetPortBit( 0x21, picMask );

	// set the Modem Control register.
	if( closeMCR != 0 )	oldMCR |= DTR;
	outportb( port+MCR, oldMCR );
	if( closeMCR == 0 )	ClearPortBit( port+MCR, DTR );

	// reset the Interrupt Enable register.
	outportb( port+IER, oldIER );

	// install the original interrupt handler.
	setvect( irq, oldhandler );

	// indicate that this port is no longer 'owned'.
	owner[irq-0x0B] = 0;
	return 1;
	}

//-------------------------------------------------------------
// Class STDPORT	method		okToSend()
//-------------------------------------------------------------

int StdPort::okToSend()
	{
	// return 1 if: valid port, open, and the Transmitter Buffer Empty
	// bit in the Line Status register is set; otherwise return 0.
	return ( irq == 0 || owner[irq-0x0B] != this ||
			GetPortBit( port+LSR, TBE ) == 0 ) ? 0 : 1;
	}

//-------------------------------------------------------------
// Class STDPORT	method		sendChar( int )
//-------------------------------------------------------------

int StdPort::sendChar( int ch )
	{
	// Ensure: valid port, open, and the Transmitter Buffer Empty bit in
	// the Line Status register is set.
	if( irq == 0 || owner[irq-0x0B] != this ||
			GetPortBit( port+LSR, TBE ) == 0 )
		return 0;

	// transmit the byte ch.
	outportb( port, ch );
	return 1;
	}

//-------------------------------------------------------------
// Class STDPORT	method		assertLine( int )
//-------------------------------------------------------------

int StdPort::assertLine( int bit )
	{
	// Fail if: invalid or not open.
	if( irq == 0 || owner[irq-0x0B] != this )	return 0;

	// Or bit with the contents of the Modem Control register.
	SetPortBit( port+MCR, bit );
	return 1;
	}

//-------------------------------------------------------------
// Class STDPORT	method		unassertLine( int )
//-------------------------------------------------------------

int StdPort::unassertLine( int bit )
	{
	// Fail if: invalid or not open.
	if( irq == 0 || owner[irq-0x0B] != this )	return 0;

	// And bit with the contents of the Modem Control register.
	ClearPortBit( port+MCR, bit );
	return 1;
	}

//-------------------------------------------------------------
// Class STDPORT	method		setBaud( long )
//-------------------------------------------------------------

int StdPort::setBaud( long baud )
	{
	// Fail if invalid or bad baud rate.
	if( port == 0 || baud < 50L || baud > 115200L )	return 0;

	// Fail if someone else is using this port.
	if( owner[irq-0x0B] != 0 && owner[irq-0x0B] != this )	return 0;

	// Enable the DLAB bit in the Line Control register.
	SetPortBit( port+LCR, DLAB );

	// Write the Baud Rate Divisor (115200/baud).
	unsigned u = (unsigned)(115200L/baud);
	outportb( port, (u & 0xFF) );
	outportb( port+1, (u >> 8) );

	// Clear the DLAB bit in the Line Control register.
	ClearPortBit( port+LCR, DLAB );
	return 1;
	}

//-------------------------------------------------------------
// Class STDPORT	method		setParity( int )
//-------------------------------------------------------------

int StdPort::setParity( int parity )
	{
	// Fail if invalid or bad parity.
	if( port == 0 || parity < NONE || parity > EVEN )
		return 0;

	// Fail if someone else is using this port.
	if( owner[irq-0x0B] != 0 && owner[irq-0x0B] != this )	return 0;

	// Set/clear appropriate bits in the Line Control register.
	int n = ( parity == NONE) ? 0 : (parity == ODD) ? 0x08 : 0x18;
	outportb( port+LCR, (inportb(port+LCR) & 0x07) | n );
	return 1;
	}

//-------------------------------------------------------------
// Class STDPORT	method		setBits( int )
//-------------------------------------------------------------

int StdPort::setBits( int bits )
	{
	// Fail if invalid or bad bits.
	if( port == 0 || bits < 6 || bits > 8 )	return 0;

	// Fail if someone else is using this port.
	if( owner[irq-0x0B] != 0 && owner[irq-0x0B] != this )	return 0;

	// Set/clear appropriate bits in the Line Control register.
	outportb( port+LCR, (inportb(port+LCR) & 0x3C) | (bits-5) );
	return 1;
	}

//-------------------------------------------------------------
// Class STDPORT	method		setStopBits( int )
//-------------------------------------------------------------

int StdPort::setStopBits( int stopbits )
	{
	// Fail if invalid or bad stopbits.
	if( port == 0 || stopbits < 1 || stopbits > 2 )	return 0;

	// Fail if someone else is using this port.
	if( owner[irq-0x0B] != 0 && owner[irq-0x0B] != this )	return 0;

	// Set/clear appropriate bits in the Line Control register.
	outportb( port+LCR, (inportb(port+LCR) & 0x3B) | (stopbits-1)*4 );
	return 1;
	}

//-------------------------------------------------------------
// Class STDPORT	method		getBaud()
//-------------------------------------------------------------

long StdPort::getBaud()
	{
	// Fail if invalid.
	if( port == 0 )	return 0;

	// Read the baud rate from the UART (see method setBaud).
	SetPortBit( port+LCR, DLAB );
	long l = inportb(port)+(inportb(port) << 8);
	ClearPortBit( port+LCR, DLAB );
	return (115200L/l);
	}

//-------------------------------------------------------------
// Class STDPORT	method		getParity()
//-------------------------------------------------------------

int StdPort::getParity()
	{
	// Fail if invalid.
	if( port == 0 )	return -1;

	// Read the parity from the UART (see method setParity).
	int n = inportb( port+LCR ) & 0x38;
	return ( n == 0 ) ? NONE : (n == 0x08) ? ODD : (n == 0x18) ? EVEN : -1;
	}

//-------------------------------------------------------------
// Class STDPORT	method		getBits()
//-------------------------------------------------------------

int StdPort::getBits()
	{
	// Fail if invalid.
	if( port == 0 )	return 0;

	// Read the bits/byte from the UART (see method setBits).
	return ( (inportb(port+LCR) & 0x03) + 5 );
	}

//-------------------------------------------------------------
// Class STDPORT	method		getStopBits()
//-------------------------------------------------------------

int StdPort::getStopBits()
	{
	// Fail if invalid.
	if( port == 0 )	return 0;

	// Read the stop bits from the UART (see method setStopBits).
	return( (inportb(port+LCR) & 0x04)/4+1 );
	}

//-------------------------------------------------------------
// Class STDPORT	method		irqHandler()
//-------------------------------------------------------------

void StdPort::irqHandler()
	{
	// Get the cause of the interrupt from the Interrupt Identification reg.
	int cause = inportb( port+IIR ) & 0x0F;

	// Handle: byte received.
	if( cause == RCVFULL && parent != 0 )
		parent->insertc( inportb(port) );

	// Handle: transmitter is empty.
	else if( cause == XMTEMPTY && parent != 0 )
		{
		// get next character to transmit from the combuf.
		int ch = parent->extractc();

		// if its a character, transmit it.
		if( ch != EOF )
			sendChar( ch );
		}

	// Handle: RS232 lines changed status.
	else if( cause == MODEMSTAT )
		inportb( port+MSR );

	// Handle: transmitter/receiver status/error.
	else if( cause == LINEFAIL )
		errFlags |= ( inportb(port+LSR) & (OR|PE|FE|BI) );

	// Signal EOI to the PIC.
	outportb( 0x20, 0x20 );
	}


//-------------------------------------------------------------
// Class COMBUF		method		constructor
//-------------------------------------------------------------

combuf::combuf( CommPort *p ) : streambuf()
	{
	// set buffer sizes to 0.
	inSize = outSize = 0;

	// clear errors.
	errFlags = 0;

	// set default timeout to 3 seconds.
	setTimeout( 300L );

	// save the CommPort pointer.
	server = p;
	}

//-------------------------------------------------------------
// Class COMBUF		method		constructor
//-------------------------------------------------------------

combuf::combuf( CommPort *p, char *b, int i, int o ) : streambuf(b, i+o)
	{
	// save the CommPort pointer.
	server = p;

	// set the buffer sizes.
	inSize = i;
	outSize = o;

	// clear errors.
	errFlags = 0;

	// set default timeout to 3 seconds.
	setTimeout( 300L );
	}

//-------------------------------------------------------------
// Class COMBUF		method		destructor
//-------------------------------------------------------------

combuf::~combuf()
	{
	// close.
	if( server )	server->close();
	}

//-------------------------------------------------------------
// Class COMBUF		method		setbuf( signed char *, int )
//-------------------------------------------------------------

streambuf *combuf::setbuf( signed char *b, int len )
	{
	// set the buffer.
	streambuf::setbuf( b, len );

	// save the input/output buffer sizes.
	inSize = len/2;
	outSize = len-inSize;

	return this;
	}

//-------------------------------------------------------------
// Class COMBUF		method		setbuf( char *, int, int )
//-------------------------------------------------------------

combuf *combuf::setbuf( char *b, int i, int o )
	{
	// set the buffer.
	streambuf( b, i+o );

	// save the input/output buffer sizes.
	inSize = i;
	outSize = o;
	return this;
	}

//-------------------------------------------------------------
// Class COMBUF		method		open()
//-------------------------------------------------------------

int combuf::open()
	{
	// Fail if invalid CommPort.
	if( server == 0 ) 	return 0;

	// Initialize the buffer pointers.
	setg( base(), base(), base() );
	setp(0,0);

	// Return result from opening the CommPort.
	return server->open( this );
	}

//-------------------------------------------------------------
// Class COMBUF		method		close()
//-------------------------------------------------------------

int combuf::close()
	{
	// Fail if invalid CommPort.
	if( server == 0 )	return 0;

	// Return result from closing the CommPort
	return server->close();
	}

//-------------------------------------------------------------
// Class COMBUF		method		underflow()
//-------------------------------------------------------------

int combuf::underflow()
	{
	StopWatch sw;

	// Fail if invalid CommPort.
	if( server == 0 )	return EOF;

	// If no characters in the input buffer...
	if( in_avail() == 0 )
		{
		// Wait the timeout period for a character to be received.
		while( sw.elapsed() < timeout_ && in_avail() == 0 );

		// Fail (indicate TIMEOUT error) if still haven't received a char.
		if( in_avail() == 0 )
			{
			errFlags |= TIMEOUT;
			return EOF;
			}
		}

	// Return the next character in the input buffer.
	return (unsigned char) *(gptr());
	}

//-------------------------------------------------------------
// Class COMBUF		method		do_sputn( const char *, int )
//-------------------------------------------------------------

int combuf::do_sputn( const char *b, int len )
	{
	StopWatch sw;
	int i;

	// Fail if invalid CommPort or no chars to send.
	if( server == 0 || len <= 0 )	return 0;

	// Note: this algorythm MAY work with an empty output buffer,
	// but I wouldn't try it.
	// If the output buffer pointer is NULL...
	if( pptr() == 0 )
		{

		// If there's only one byte...
		if( len == 1 )
			{
			// Try to send the character.
			while( server->sendChar( *b ) == 0 )

				// But if we timeout, set error flag and fail.
				if( sw.elapsed() >= timeout_ )
					{
					errFlags |= TIMEOUT;
					return 0;
					}
			return 1;
			}

		// Come here if theres 2+ chars to send.
		// Wait the timeout period for the CommPort to be ready
		// to send the first byte in the buffer.
		while( server->okToSend() == 0 )
			if( sw.elapsed() >= timeout_ )
				{
				errFlags |= TIMEOUT;
				return 0;
				}

		// Initialize the output buffer (if there is any).
		setp( base()+inSize, ebuf() );

		// If the output buffer can hold all the characters to be sent...
		if( outSize >= len-1 )
			{
			// copy b+1 to the output buffer.
			memcpy( pptr(), b+1, len-1 );

			// initialize the pointers to show they are there.
			pbump(len-1);

			// prime the pump with the first character.
			server->sendChar( *b );
			return len;
			}

		// Not enough room for all - do piecemeal.
		// Put as many bytes as we can in the buffer.
		memcpy( pptr(), b+1, outSize );

		// and initialize the pointers to show they are there.
		pbump(outSize);

		// prime the pump with the first character.
		server->sendChar(*b);

		// and send the uncopied bytes one at a time.
		b += (outSize+1);
		for( i = outSize+1; i < len && sputc(*b++) != EOF; i++ );
		return i;
		}

	// The buffer has some characters (its full) - call the streambuf
	// method do_sputn() to send the characters one at a time.
	else
		return streambuf::do_sputn( b, len );
	}

//-------------------------------------------------------------
// Class COMBUF		method		setTimeout( long )
//-------------------------------------------------------------

void combuf::setTimeout( long hSec )
	{
	// set member timeout_ to the number of timeticks that represent
	// hSec (in 1/100ths second).
	timeout_ = timeToTicks( hSec );
	}

//-------------------------------------------------------------
// Class COMBUF		method		extractc()
//-------------------------------------------------------------

int combuf::extractc()
	{
	int ch;

	// If there's a character in the buffer, extract it.
	if( out_waiting() != 0 )
		{
		// get the next character.
		ch = *(pptr()-1);

		// update the buffer pointer.
		pbump(-1);

		// if that was the last character, init the pointers to NULL.
		if( out_waiting() == 0 )
			setp(0,0);
		}

	// otherwise, clear the output buffer pointers and return EOF.
	else
		{
		setp(0,0);
		ch = EOF;
		}
	return ch;
	}

//-------------------------------------------------------------
// Class COMBUF		method		insertc( int )
//-------------------------------------------------------------

void combuf::insertc( int ch )
	{
	// set g to the buffer location to store the next char.
	char *g = egptr();

	// if there is a buffer...
	if( g != 0 )
		{
		// Store it and update the pointers if there's room before the
		// end of the input buffer.
		if( g < base()+inSize )
			{
			*g = ch;
			setg( base(), gptr(), g+1 );
			}

		// Shift the buffer contents upward, store it, and update the
		// pointers if the front of the input buffer has room.
		else if( gptr() > base() )
			{
			int len = (int)(g - gptr());
			memmove( base(), gptr(), len );
			*(base()+len) = ch;
			setg( base(), base(), base()+len+1 );
			}

		// Otherwise there's no room - clear the pointer to indicate it.
		else
			g = 0;
		}

	// If there's no room in the input buffer, set the overrun (OR) error
	// flag.
	if( g == 0 )
		errFlags |= OR;
	}

//-------------------------------------------------------------
// Class COMBUF		method		timeToTicks( long )
//-------------------------------------------------------------

unsigned long combuf::timeToTicks( long l )
	{
	// Convert 1/100ths second argument to 1/18.2 seconds timeticks.
	return l*182L/1000L;
	}

