/**************************************************************************
   MODULE		:	CMfDB.h
   Description	:	This is the header file for the C++ class wrapper for
					MF database DLL.  This version supports MF v1.03
**************************************************************************/

#if !defined(CMfDB_H)

	#define CMfDB_H
	
#include <mf.h>

#define	MAX_INDEXES	9
 
// This class manages the DB FileName.  It deals with the re-opening, and
//  controlled closing of the file between accesses.  The class encapsulates
//  calls to mfOpen() and mfClose().  It also keeps track of multiple calls
//  to the Open and Close class members in order to close the DB only when
//  the last Close call is performed.
class	CMfDBFile
{
	// Member variables
	private:
		BOOL	m_bKeepOpen;		// TRUE if DB file(s) are to remain open forever.  FALSE if
									//  the file(s) are to be opened and closed before and after
									//  each access.
		
		char	*m_FileName;		// The filename for this DB.
		WORD	m_iUseCount;		// The file can only change state when this == 0.  This allows
									//  for the stacking of Open() and Close() when internal class
									//  functions call each other.
		hMF_TASK	m_hTask;		// Task identifier for access to the DLL.  This comes from the
									//  class creating this class.
									
	// Member functions
	protected:
		hMF_DB		m_hDataBase;	// handle to an open database hMF_DB, -1 = not initialized or
									//  not opened, -2 = temporary closed, -3 = last close failed
									
	public:
		// Function used to create a DBFile Manager Object
		static CMfDBFile *NewFileObject(hMF_TASK hDBTask = -1, const char *szFileName = NULL);
		
		// If you have a DBFile Manager, use this function to change the
		// DBFileName ATTACHED to the object.
		void Attach(const char *szFileName = NULL);
		
		// Use this function carefully!  It will overwrite the handle
		// used to access the db files(s).  At this time the ONLY place 
		// this is needed is after a mfReIndex() command which causes a
		// new hDB to be generated!
		void AttachDB(hMF_DB hDB) { m_hDataBase = hDB; }
		
		BOOL SetDBOpenFlag(BOOL bNewOpenFlag = FALSE);
		BOOL GetDBOpenFlag() const { return m_bKeepOpen; }
		
		hMF_DB GetHandle() const { return m_hDataBase; }
		char *GetFileName() const { return m_FileName; }
		
		hMF_DB Open();
		int Close();
		
		virtual ~CMfDBFile();
		
	protected:
		// The constructor is protected so a class cannot be created using 'new' or on the
		//  stack.  You have to use the 'NewFileObject' member function !
		CMfDBFile(const char *szFileName = NULL, hMF_TASK hDBTask = -1);
};
	
	
class	CMfDB
{
	// Class encapsulated defines
	public:
		#define		ILLEGAL_FILENAME		-100
		#define		DATABASE_CLOSED			-101	/* Start here leaving room for MF.DLL's errors */
		#define		NO_BUFFER_ALLOCATED		-102
		#define		ILLEGAL_INDEX_NUMBER	-103
		#define		ILLEGAL_RECORD_NUMBER	-104
		#define		ILLEGAL_BUFFER_TYPE		-105
		#define		ILLEGAL_FILE_STATE		-106
		#define		RECORD_ALREADY_LOCKED	-107
		#define		RECORD_NOT_LOCKED		-108
		#define		NO_MATCH_FOUND			-109
		#define		ILLEGAL_ARRAY_SIZE		-110
		#define		START_SEARCH_NOT_DONE	-111
		#define		DATABASE_NOT_CLOSED		-112
		#define		ILLEGAL_INDEX_SIZE		-113
		#define		FAILED_TO_INIT_MF_DLL	-114
		#define		FILE_MANAGER_FAILED		-115
		#define		ATTEMPT_TO_INIT_TWICE	-116
		#define		PROGRAM_INDEX_ERROR		-117
		
	// member variables
	private:
	    CMfDBFile	*m_pCMfDBFile;			// Ptr to file management class
		
		int			m_iRecordSize;			// The size of a database entry record size
		int			m_iNumberOfIndexes;		// Number of index files for this database
		int			m_iKeySize[MAX_INDEXES];// The size of each index
		long		m_lTotalIndexSize;		// The cumulative size of all indexes
		
		int			m_iActiveIndex;			// The index being used to transvers the database
		RPTR		m_lActiveRecord;		// identifies the most recent record number accessed
		
		char		*m_LastSearch;			// Buffer from last search
		int			m_LastSearchLength;		// Match length of last search
		long		m_LastSearchRecord;		// last record searched
		
		BOOL		m_ClassOkToUse;			// TRUE is class successfully constructed
		
		// this member variable is static because it is needed only once per application !
		static		hMF_TASK	m_hTask;		// task identifier for access to the DLL
		static		int			m_iUseCount;	// number of instances of this class.  This
												// is used to check for the last objects so
												// to close the DLL (mfDeInit())
											
	// member functions
	private:
		void ClearMemberVariables();
		int InitMFDLL(const char *szUDKFileName = NULL, int iType = MFCOMP_UDK);
		CMfDB(const CMfDB&);					// Don't allow copies !
		CMfDB& operator =(const CMfDB&);		// Don't allow copies !
		
	protected:
		// This is protected so an object of this type can't be created on the heap
		// Classes which derive this class should check for proper construction using
		//  the GetValid() member function which returns TRUE of class is OK to use
		CMfDB(const char *szUDKFileName = NULL, int iType = MFCOMP_UDK);
		
	public:
		// I chose to do it with a function instead of throwing an exception since a
		// lot of compilers don't support the 'throw' keyword
		BOOL GetValid() const { return m_ClassOkToUse; }
		
	public:
		// This static function is used to init the DLL and return an object for use.  The default
		//  constructor is protected which will block the user from creating an object of type
		//  CMfDB.  At the same time an object of type CMfDBFile is created which manages the
		//  open state of any database file(s).  It s possible to create the object and open the
		//  database, assigning an optional user defined Key (.DLL) in this one call.  It should be
		//  noted that if ANY part of the process failes a NULL will be returned.
		static CMfDB* NewDBObject(const char *szFileName = NULL, const char *szUDKFileName = NULL, int iType = MFCOMP_UDK);
		
		// These next functions are used when this class is derived from.  If this happens then
		//  A NULL value passed instead of a buffer will cause one of these functions to be called.
		//  If you overload these functions within your derived class then you have the chance to
		//  return a pointer to a buffer which will be filled by the command.  This will allow you
		//  to totaly encapsulate the database functions and data in your own class.
		
		virtual char *GetDataBuffer() const { return NULL; }
		virtual char *GetIndexBuffer() const { return NULL; }
		virtual char *GetBothBuffers() const { return NULL; }
		         
		// These 2 functions are used to set the m_bKeepOpen flag associated with set of database
		//  files.  This flag is FALSE by default which means that the database file(s) are openned
		//  and closed for each access.  This happens very quickly so changing this flag will have
		//  very little affect on overall performance.
		BOOL SetDBOpenFlag(BOOL NewOpenFlag) { return m_pCMfDBFile->SetDBOpenFlag(NewOpenFlag); }
		BOOL GetDBOpenFlag() const { return m_pCMfDBFile->GetDBOpenFlag(); }
		
		// These functions manipulate the index being used for database tranversing functions
		//  (GetFirst, GetNext, GetPrev, GetLast etc).  The index number is a zero-based value
		//  referencing the index used when the database was created.  By default index number 0
		//  is used.
		int SetActiveIndex(int NewIndex = 0);
		int GetActiveIndex() const { return m_iActiveIndex; }
		
		
		// This function returns the active record number within the database.
		RPTR GetRecordNumber() const { return m_lActiveRecord; }
		
		// These functions return information about this database.
		//  GetIndexCount() return the number of indexes within this database.
		//  GetIndexSize(int) returns the size in bytes of the requested index.
		//  GetRecordSize() returns the size in bytes of a datarecord for this database.
		//  GetRecordCount() returns the total number of records (deleted and active).
		//  GetActiveCount() returns the number of active (nondeleted) records.
		int GetIndexCount() const { return m_iNumberOfIndexes; }
		int GetIndexSize(int iIndexNumber = 0) const { return m_iKeySize[iIndexNumber]; }
		int GetRecordSize() const { return m_iRecordSize; }
		long GetRecordCount() const;
		long GetActiveCount() const;
		
		// This function is used to create an empty database file set.
		int CreateDB(const char *szFileName, int iDBRecSize, int iNumIndexes, const int iKeySize[], const int iKeyType[], BOOL bOpenFlag = TRUE);
		
		// Open a database file set
		int OpenDB(const char *szFileName = NULL);
		
		// Close a database fileset.  This is done automatically in the destructor as well.
		void CloseDB();
		              
		// These functions are for transvering the DB in index order.  If a buffer is supplied the
		//  resulting record will be placed in the user's buffer upon return.
		int GetFirst(int iBufferType = MFRW_ALL, char *lpBuffer = NULL);
		int GetNext(int iBufferType = MFRW_ALL, char *lpBuffer = NULL);
		int GetPrev(int iBufferType = MFRW_ALL, char *lpBuffer = NULL);
		int GetLast(int iBufferType = MFRW_ALL, char *lpBuffer = NULL);
		int Skip(long lSkipCount = 0, int iBufferType = MFRW_ALL, char *lpBuffer = NULL);
		
		// General read functions.  The 1st function will read the active record number.  This is
		//  propably the one used most since all operations which access the database set the
		//  m_lActiveRecord variable for you.  You can pass one of 3 pointers to buffers as defined
		//  by the last parameter.  MFRF_DATA (just the data part of the record), MFRW_KEY (just the
		//  index key part) or MFRW_ALL (both index and data parts).  The size of these buffers are
		//  verified to try and minimize errors.
		int Read(int iBufferType = MFRW_ALL, char *lpBuffer = NULL) { return Read(m_lActiveRecord, iBufferType, lpBuffer); }
		
		// This version of the read only differs in that the user can specify a specific record 
		//  number.  If this record happens to be deleted an error is returned.
		//  The internally kept last active record is not modified.
		int Read(RPTR lRecordNumber = -1, int iBufferType = MFRW_ALL, char *lpBuffer = NULL);
		
		// Database modification services
		
		// This function adds a record to the database.  If a data buffer of type MFRW_DATA is passed
		//  the record is written unindexed.  A data buffer of type MFRW_ALL will write the record
		//  in the database as well as update the indexes.
		int AddRecord(int iBufferType = MFRW_ALL, char *lpBuffer = NULL);
		
		// This function updates a previously read database record by writing the buffer, and optional
		//  key data to the same location as the current active record number returned from a previous
		//  read.  If you are using keys you should update key data as well !
		int UpdateRecord(int iBufferType = MFRW_ALL, char *lpBuffer = NULL);
		
		// This version of the UpdateRecord only differs in that the user can specify a
		//  specific record number.  If this record happens to be deleted an error is
		//  returned.  The internally kept last active record is not modified.
		int UpdateRecord(RPTR lRecordNumber = -1, int iBufferType = MFRW_ALL, char *lpBuffer = NULL);
		
		// The following 3 functions will perform their task on the user specified record, or if
		//  no record is specified (default) the function will be performed on the active record.
		int DeleteRecord(RPTR lRecordNumber = -1);
		int LockRecord(RPTR lRecordNumber = -1);
		int UnlockRecord(RPTR lRecordNumber = -1);
		int IsLocked(RPTR lRecordNumber = -1);
		
		// Search an index for an item
		
		// Searches an indexed database for the 1st record which matches the specified search item.
		//  If a buffer is supplied it will be filled with the record upon return.
		RPTR Find(int iBufferSize = 0, const char *StringToFind = NULL, int iSearchType = MFSEEK_PARTIAL_MATCH, int iBufferType = MFRW_ALL, char *lpBuffer = NULL);
		
		// Return a list of records matching the search item.  If no array of longs is supplied
		//  then a count of matching items is returned and the active record point to the 1st match.
		//  If an array of longs are supplied then the return value is the number of matches found
		//  and their record numbers placed in the passed array elements.  In either case the active
		//  record pointed within the class points to the 1st match.
		long FindAll(int iBufferSize = 0, const char *StringToFind = NULL, int iMatchLength = -1, RPTR *lRecordArray = NULL, long lArraySize = -1, RPTR lRecordNumber = 0);
		long FindMore(RPTR *lRecordArray = NULL, long lArraySize = 0);
		
		// This function will recreate and reindex the specified index item of a database file set.
		//  An optional hWnd parameter may be specified to identify a window which is to receive
		//  WM_SETTEXT messages containing the percentage complete as described in the MF.TXT
		//  documentation.
		int ReCreateIndexDB(int iIndex = -1, int iIndexType = -1, int iIndexSize = -1, HWND hWnd = NULL);
		
		virtual ~CMfDB();
		
		// **** Added Feb, 1994 ****
		
		// The functions return 0 if FALSE, 1 if TRUE and < 0 if an error
		int IsFirst() const;
		int IsLast() const;
};

#endif
