// CmTime.cpp
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Time class implementation.
// -----------------------------------------------------------------

#include <time.h>
#include <iostream.h>
#include <iomanip.h>
#include <cm/include/cmtime.h>
#include <stdio.h>

// Static constant definitions.
static const CmDate        refDate((unsigned) 0, (unsigned) 0);
static const unsigned long SECONDS_IN_DAY  = 86400L;
static const unsigned long SECONDS_IN_HOUR = 3600L;
static const unsigned long SECONDS_IN_MIN  = 60L;

#if defined(hpux)          // For unix machines, use "gettimeofday"
  class CmInitTime {       // function for time zones.
  public:
    CmInitTime();
    long time_zone;
    int  dst_observed;
  };

  CmInitTime ___cm_global_init_time;
  CmInitTime::CmInitTime()
  {
    struct timeval  tval;
    struct timezone tz;
    gettimeofday(&tval,&tz);
    time_zone    = (long)(SECONDS_IN_MIN * (unsigned long) tz.tz_minuteswest);
    dst_observed = tz.tz_dsttime;
  }

# define TIME_ZONE    ___cm_global_init_time.time_zone
# define DST_OBSERVED ___cm_global_init_time.dst_observed
#else
# define TIME_ZONE    timezone
# define DST_OBSERVED daylight
#endif

// Initialize display style.
int CmTime::_displayStyle = CmTime::DATE12;


// "CmTime" is the default time constructor.
//
CmTime::CmTime()
{
  time_t ltime = time(0);
  struct tm *t = localtime(&ltime);
  CmDate today((unsigned)t->tm_mday, (unsigned)(t->tm_mon + 1),
               (unsigned)t->tm_year);
  _sec = localTime(today, (unsigned)t->tm_hour, (unsigned)t->tm_min,
                  (unsigned)t->tm_sec)._sec;
  if (isDST()) _sec -= SECONDS_IN_HOUR;
}


// "CmTime" constructs a time from the specified hour, minute,
// and second.
//
CmTime::CmTime(unsigned h, unsigned m, unsigned s)
{
  _sec = CmTime(CmDate(), h, m, s)._sec;
}


// "CmTime" constructs a time from the specified date, hour, minute,
// and second.
//
CmTime::CmTime(const CmDate& d, unsigned h, unsigned m, unsigned s)
{
  _sec = localTime(d, h, m, s)._sec;
  if (isDST()) _sec -= SECONDS_IN_HOUR;
}


// "isDST" returns TRUE if this time is daylight savings time.
//
Bool CmTime::isDST() const
{
  if (!DST_OBSERVED) return FALSE;

  unsigned daycount = (unsigned)(_sec / SECONDS_IN_DAY);
  unsigned year     = CmDate((unsigned) daycount, (unsigned) 0).year();

  return (*this >= beginDST(year) && *this < endDST(year));
}


// "localTime" returns the local time.
//
CmTime CmTime::localTime() const
{
  CmTime local_time(_sec - TIME_ZONE);
  if (local_time.isDST()) local_time._sec += SECONDS_IN_HOUR;
  return local_time;
}


// "hour" returns the hour number.
//
unsigned CmTime::hour() const
{
  return localTime().hourGMT();
}


// "hourGMT" returns the hour number (Greenwich mean time).
//
unsigned CmTime::hourGMT() const
{
  return (_sec % SECONDS_IN_DAY) / SECONDS_IN_HOUR;
}


// "minute" returns the minute number.
//
unsigned CmTime::minute() const
{
  return localTime().minuteGMT();
}


// "minuteGMT" returns the minute number (Greenwich mean time).
//
unsigned CmTime::minuteGMT() const
{
  return ((_sec % SECONDS_IN_DAY) % SECONDS_IN_HOUR) / SECONDS_IN_MIN;
}


// "second" returns the second number.
//
unsigned CmTime::second() const
{
  return ((_sec % SECONDS_IN_DAY) % SECONDS_IN_HOUR) % SECONDS_IN_MIN;
}


// "makeCurrent" makes this time the current time.
//
void CmTime::makeCurrent()
{
  time_t ltime = time(0);
  struct tm *t = localtime(&ltime);
  CmDate today((unsigned)t->tm_mday, (unsigned)(t->tm_mon + 1),
               (unsigned)t->tm_year);
  _sec = localTime(today, (unsigned)t->tm_hour, (unsigned)t->tm_min,
                  (unsigned)t->tm_sec)._sec;
  if (isDST()) _sec -= SECONDS_IN_HOUR;
}


// "CmDate" operator returns a date object based on this time.
//
CmTime::operator CmDate() const
{
  unsigned long ls = _sec;
  if (isDST()) ls += SECONDS_IN_HOUR;
  unsigned long jn = ((unsigned long) 2415386L) + (ls / SECONDS_IN_DAY);
  return CmDate(jn);
}


// "isEqual" compares this time with the specified time.
//
Bool CmTime::isEqual(CmObject* pObj) const
{
  if (!pObj->isA("CmTime")) return CmObject::isEqual(pObj);
  return (((CmTime*) pObj)->_sec == _sec);
}


// "compare" compares this time with the specified time.
//
int CmTime::compare(CmObject* pObj) const
{
  if (!pObj->isA("CmTime")) return CmObject::compare(pObj);
  unsigned long sc = ((CmTime*) pObj)->_sec;
  return (_sec == sc ? 0 : (_sec > sc ? 1 : -1));
}


// "hash" hashes this time returning the hash value.
//
unsigned CmTime::hash(unsigned m) const
{
  return (unsigned) (_sec %  (unsigned long) m);
}


// "printOn" prints this time to the specified stream.
//
void CmTime::printOn(ostream& os) const
{
  if (CmTime::_displayStyle == DATE24 || CmTime::_displayStyle == DATE12)
    os << CmDate(*this) << ' ';

  char buff[40];
  if (CmTime::_displayStyle == DATE24 || CmTime::_displayStyle == NODATE24)
  {
    sprintf(buff, "%u:%02u:%02u ", hour(), minute(), second());
    os << buff;
  }
  else
  {
    unsigned hh24 = hour();
    unsigned hh12 = (hh24 > 12) ? hh24 - 12 : hh24;
    sprintf(buff, "%u:%02u:%02u ", hh12, minute(), second());
    os << buff << (hh24 < 12 ? "am" : "pm");
  }
}


// "write" writes the time to the specified reserve binary file.
//
Bool CmTime::write(CmReserveFile& file) const
{
  return file.write(_sec);
}


// "read" reads the time from the specified reserve binary file.
//
Bool CmTime::read(CmReserveFile& file)
{
  return file.read(_sec);
}


// "beginDST" returns the time daylight savings time begins in the
// specified year.
//
CmTime CmTime::beginDST(unsigned year)
{
  CmTime DSTtime(localTime(CmDate(31,"March",year).previous("Sunday")+7,2));

  if (year <= 1986)
  {
    DSTtime = localTime(CmDate(30,"April",year).previous("Sunday"),2);
    if (year==1974) DSTtime = localTime(CmDate(6,"January",1974),2);
    if (year==1975) DSTtime = localTime(CmDate(23,"February",1975),2);
  }
  return DSTtime;
}


// "endDST" returns the time daylight savings time ends in the
// specified year.
//
CmTime CmTime::endDST(unsigned year)
{
  CmTime STDtime(localTime(CmDate(31,"October",year).previous("Sunday"),1));

  return STDtime;
}


// "localTime" returns the local time based on the date, hour, minute,
// and second.
//
CmTime CmTime::localTime(const CmDate& d, unsigned h, unsigned m, unsigned s)
{
  unsigned long secs = SECONDS_IN_DAY  * (d - refDate).julNum() +
                       SECONDS_IN_HOUR * (unsigned long) h      +
                       SECONDS_IN_MIN  * (unsigned long) m      +
                       (unsigned long) s + (unsigned long) TIME_ZONE;
  return CmTime(secs);
}
