// CmHDict.cpp
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Hash dictionary implementation.
// -----------------------------------------------------------------

#include <cm/include/cmhdict.h>


// "CmHashDictionary" is the default dictionary constructor.
//
CmHashDictionary::CmHashDictionary(unsigned sz)
                 : CmHashTable(sz)
{}


// "CmHashDictionary" is the dictionary copy constructor.
//
CmHashDictionary::CmHashDictionary(const CmHashDictionary& D)
             : CmHashTable(D)
{}


// "~CmHashDictionary" is the dictionary destructor.
//
CmHashDictionary::~CmHashDictionary()
{
  removeAll();
}


// "=" assignment operator copies the specified dictionary into
// this dictionary.
//
CmHashDictionary& CmHashDictionary::operator=(const CmHashDictionary& D)
{
  return (CmHashDictionary&) CmHashTable::operator=(D);
}


// "add" adds an association to the dictionary.
//
Bool CmHashDictionary::add(CmObject* pObj)
{
  if (pObj->isA("CmAssociation"))
    if (!CmHashTable::contains(pObj))
      return CmHashTable::add(pObj);
  return FALSE;
}


// "add" adds a key/object pair to the dictionary.
//
Bool CmHashDictionary::add(CmObject* pKey, CmObject* pObj)
{
  CmAssociation *pAss = new CmAssociation(pKey, pObj);
  if (!pAss) return FALSE;

  if (CmHashTable::contains(pAss))
  {
    delete pAss;
    return FALSE;
  }
  else
    return CmHashTable::add(pAss);
}


// "lookup" looks for the first occurrence of an association with
// a key that isEqual to the specified key and returns that association's
// key value.
//
CmObject* CmHashDictionary::lookup(CmObject* pKey) const
{
  CmAssociation *pAss = lookupAssoc(pKey);
  return (pAss) ? pAss->key() : NULL;
}


// "lookupObject" looks for the first occurrence of an association with
// a key that isEqual to the specified key and returns that association's
// object value.
//
CmObject* CmHashDictionary::lookupObject(CmObject* pKey) const
{
  CmAssociation *pAss = lookupAssoc(pKey);
  return (pAss) ? pAss->object() : NULL;
}


// "lookupObject" looks for the first occurrence of an association with
// a key that isEqual to the specified key and returns that association.
//
CmAssociation* CmHashDictionary::lookupAssoc(CmObject* pKey) const
{
  CmAssociation assoc(pKey, NULL);
  return (CmAssociation*) CmHashTable::lookup(&assoc);
}


// "replace" looks for the first occurrence of an association with
// a key that isEqual to the specified key and replaces it's object
// value with the specified object.
//
Bool CmHashDictionary::replace(CmObject* pKey, CmObject* pObj)
{
  CmAssociation *pAss = lookupAssoc(pKey);
  if (!pAss) return FALSE;

  if (ownsObjects()) delete pAss->object();
  pAss->object(pObj);
  return TRUE;
}


// "remove" looks for the first occurrence of an association with
// a key that isEqual to the specified key and removes it.
//
Bool CmHashDictionary::remove(CmObject* pKey)
{
  CmAssociation *pAss = lookupAssoc(pKey);
  if (pAss)
  {
    CmObject* pKey    = pAss->key   ();
    CmObject* pObject = pAss->object();
    Bool success = CmHashTable::remove(pAss);
    if (success && ownsObjects())
    {
      delete pKey;
      delete pObject;
    }
    return success;
  }
  return FALSE;
}


// "contains" checks to see is an association exists with a key which
// isEqual to the specified key.
//
Bool CmHashDictionary::contains(CmObject* pKey) const
{
  return (lookupAssoc(pKey) == NULL) ? FALSE : TRUE;
}


// "occurrences" returns the number of associations with a key which
// isEqual to the specified key.  (This should always be 1).
//
unsigned CmHashDictionary::occurrences(CmObject* pKey) const
{
  CmAssociation assoc(pKey, NULL);
  return CmHashTable::occurrences(&assoc);
}


// "removeAll" removes all of the associations from the dictionary.
//
void CmHashDictionary::removeAll()
{
  if (ownsObjects())
  {
    CmHashTableIterator iterator(*this);
    while (!iterator.done())
    {
      CmAssociation *pAss = (CmAssociation*) iterator.next();
      delete pAss->key   ();
      delete pAss->object();
    }
  }
  CmHashTable::removeAll();
}
