// CmBDict.cpp
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// BTree dictionary implementation.
// -----------------------------------------------------------------

#include <cm/include/cmbdict.h>


// "CmBTreeDictionary" is the default dictionary constructor.
//
CmBTreeDictionary::CmBTreeDictionary(unsigned sz)
                  : CmBTree(sz)
{}


// "CmBTreeDictionary" is the dictionary copy constructor.
//
CmBTreeDictionary::CmBTreeDictionary(const CmBTreeDictionary& D)
                  : CmBTree(D)
{}


// "~CmBTreeDictionary" is the dictionary destructor.
//
CmBTreeDictionary::~CmBTreeDictionary()
{
  removeAll();
}


// "=" assignment operator copies the specified dictionary into
// this dictionary.
//
CmBTreeDictionary& CmBTreeDictionary::operator=(const CmBTreeDictionary& D)
{
  return (CmBTreeDictionary&) CmBTree::operator=(D);
}


// "add" adds an association to the dictionary.
//
Bool CmBTreeDictionary::add(CmObject* pObj)
{
  if (pObj->isA("CmAssociation"))
    if (!CmBTree::contains(pObj))
      return CmBTree::add(pObj);
  return FALSE;
}


// "add" adds a key/object pair to the dictionary.
//
Bool CmBTreeDictionary::add(CmObject* pKey, CmObject* pObj)
{
  CmAssociation *pAss = new CmAssociation(pKey, pObj);
  if (!pAss) return FALSE;

  if (CmBTree::contains(pAss))
  {
    delete pAss;
    return FALSE;
  }
  else
    return CmBTree::add(pAss);
}


// "lookup" looks for the first occurrence of an association with
// a key that isEqual to the specified key and returns that association's
// key value.
//
CmObject* CmBTreeDictionary::lookup(CmObject* pKey) const
{
  CmAssociation *pAss = lookupAssoc(pKey);
  return (pAss) ? pAss->key() : NULL;
}


// "lookupObject" looks for the first occurrence of an association with
// a key that isEqual to the specified key and returns that association's
// object value.
//
CmObject* CmBTreeDictionary::lookupObject(CmObject* pKey) const
{
  CmAssociation *pAss = lookupAssoc(pKey);
  return (pAss) ? pAss->object() : NULL;
}


// "lookupObject" looks for the first occurrence of an association with
// a key that isEqual to the specified key and returns that association.
//
CmAssociation* CmBTreeDictionary::lookupAssoc(CmObject* pKey) const
{
  CmAssociation assoc(pKey, NULL);
  return (CmAssociation*) CmBTree::lookup(&assoc);
}


// "replace" looks for the first occurrence of an association with
// a key that isEqual to the specified key and replaces it's object
// value with the specified object.
//
Bool CmBTreeDictionary::replace(CmObject* pKey, CmObject* pObj)
{
  CmAssociation *pAss = lookupAssoc(pKey);
  if (!pAss) return FALSE;

  if (ownsObjects()) delete pAss->object();
  pAss->object(pObj);
  return TRUE;
}


// "remove" looks for the first occurrence of an association with
// a key that isEqual to the specified key and removes it.
//
Bool CmBTreeDictionary::remove(CmObject* pKey)
{
  CmAssociation *pAss = lookupAssoc(pKey);
  if (pAss)
  {
    CmObject* pKey    = pAss->key   ();
    CmObject* pObject = pAss->object();
    Bool success = CmBTree::remove(pAss);
    if (success && ownsObjects())
    {
      delete pKey;
      delete pObject;
    }
    return success;
  }
  return FALSE;
}


// "contains" checks to see is an association exists with a key which
// isEqual to the specified key.
//
Bool CmBTreeDictionary::contains(CmObject* pKey) const
{
  return (lookupAssoc(pKey) == NULL) ? FALSE : TRUE;
}


// "occurrences" returns the number of associations with a key which
// isEqual to the specified key.  (This should always be 1).
//
unsigned CmBTreeDictionary::occurrences(CmObject* pKey) const
{
  CmAssociation assoc(pKey, NULL);
  return CmBTree::occurrences(&assoc);
}


// "removeAll" removes all of the associations from the dictionary.
//
void CmBTreeDictionary::removeAll()
{
  if (ownsObjects())
  {
    CmBTreeIterator iterator(*this);
    while (!iterator.done())
    {
      CmAssociation *pAss = (CmAssociation*) iterator.next();
      delete pAss->key   ();
      delete pAss->object();
    }
  }
  CmBTree::removeAll();
}
