// CmTime.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Time class definition.
// -----------------------------------------------------------------

#ifndef _CMTIME_H
#define _CMTIME_H

#include <cm/include/cmdate.h>

class CmTime : public CmObject {              // Time class definition.
public:
  CmTime();                                   // Default time constructor.
  CmTime(unsigned long s) : _sec(s) {}        // Construct from seconds.
  CmTime(unsigned, unsigned, unsigned = 0);   // Construct from h, m, s.
  CmTime(const CmDate&, unsigned = 0, unsigned = 0, unsigned = 0);

  Bool     isDST      () const;               // Check if daylight savings.
  CmTime   localTime  () const;               // Get local time.
  unsigned hour       () const;               // Get hour number.
  unsigned hourGMT    () const;               // Get hour number (Greenwich).
  unsigned minute     () const;               // Get minute number.
  unsigned minuteGMT  () const;               // Get minute num  (Greenwich).
  unsigned second     () const;               // Get second number.
  void     makeCurrent();                     // Make this current time.

  unsigned long seconds() const;              // Returns seconds.

  operator CmDate() const;                    // Make date from time.

  Bool operator< (const CmTime&) const;       // Check if time <  this.
  Bool operator<=(const CmTime&) const;       // Check if time <= this.
  Bool operator> (const CmTime&) const;       // Check if time >  this.
  Bool operator>=(const CmTime&) const;       // Check if time >= this.
  Bool operator==(const CmTime&) const;       // Check if time == this.
  Bool operator!=(const CmTime&) const;       // Check if time != this.

  Bool     isEqual(CmObject*) const;          // Compare times.
  int      compare(CmObject*) const;          // Compare times.
  unsigned hash   (unsigned)  const;          // Hash time.
  void     printOn(ostream&)  const;          // Print time to stream.
  Bool     write  (CmReserveFile&) const;     // Write time to file.
  Bool     read   (CmReserveFile&);           // Read time from file.

  CMOBJECT_DEFINE(CmTime, CmObject)           // Define object funcs.

  static CmTime beginDST (unsigned);          // Get start of DST for year.
  static CmTime endDST   (unsigned);          // Get end of DST for year.
  static CmTime localTime(const CmDate&, unsigned = 0,  // Get local time.
                           unsigned = 0, unsigned = 0);

  static void displayStyle(int);              // Set display style for all.
  static int  displayStyle();                 // Get display style.

  enum display_style { DATE12, DATE24, NODATE12, NODATE24 };  // Styles.

private:
  unsigned long _sec;                         // Seconds.
  static   int  _displayStyle;                // Display style.
};

// "seconds" returns the seconds value.
inline unsigned long CmTime::seconds() const
{ return _sec; }

// "<" checks if the input is less than this time.
inline Bool CmTime::operator<(const CmTime& T) const
{ return (_sec < T._sec); }

// "<=" checks if the input is less than or equal to this time.
inline Bool CmTime::operator<=(const CmTime& T) const
{ return (_sec <= T._sec); }

// ">" checks if the input is greater than this time.
inline Bool CmTime::operator>(const CmTime& T) const
{ return (_sec > T._sec); }

// ">=" checks if the input is greater than or equal to this time.
inline Bool CmTime::operator>=(const CmTime& T) const
{ return (_sec >= T._sec); }

// "==" checks if the input is equal to this time.
inline Bool CmTime::operator==(const CmTime& T) const
{ return (_sec == T._sec); }

// "!=" checks if the input is not equal to this time.
inline Bool CmTime::operator!=(const CmTime& T) const
{ return (_sec != T._sec); }

// "displayStyle" sets the display style for all times.
inline void CmTime::displayStyle(int ds)
{ CmTime::_displayStyle = ds; }

// "displayStyle returns the current display style.
inline int CmTime::displayStyle()
{ return CmTime::_displayStyle; }

#endif
