// CmTHash.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Hash table template definition.
// -----------------------------------------------------------------

#ifndef _CMTHASH_H
#define _CMTHASH_H

#include <cm/include/cmtcont.h>

template <class T> class CmTHashBucket;         // Bucket class stub.
template <class T> class CmTHashNode;           // Node class stub.
template <class T> class CmTHashTableIterator;  // Iterator class stub.

template <class T> class CmTHashTable : public CmTContainer<T> {
public:
  CmTHashTable(unsigned = 10);                  // Default table constructor.
  CmTHashTable(const CmTHashTable<T>&);         // Table copy constructor.
 ~CmTHashTable();                               // Table destructor.

  CmTHashTable<T>& operator=(const CmTHashTable<T>&);  // Assignment.

  void setHashFunction(unsigned (*pFunc)(const T&, unsigned));  // Set hasher.

  int      total      () const;                 // Return number of items.
  Bool     add        (const T&);               // Add item to table.
  Bool     remove     (const T&);               // Remove equal item from table.
  const T& lookup     (const T&) const;         // Return equal item from table.
  Bool     contains   (const T&) const;         // See if equal item in table.
  unsigned occurrences(const T&) const;         // How many occurrences of item.
  void     removeAll  ();                       // Remove all items.
  Bool     resize     (unsigned);               // Resize the table.
  Bool     isEmpty    () const;                 // Is table empty.

  CmTIterator<T>* newIterator() const;          // Create and return iterator.

protected:
  unsigned (*_hFunc)(const T&, unsigned);       // Hash function pointer.

  int               _total;                     // Number of items in table.
  Bool              _funcSet;                   // Is hash function set.
  CmTHashBucket<T> *_buckets;                   // Hash bucket list.
  friend            CmTHashTableIterator<T>;    // Iterator can access.
};

template <class T> class CmTHashBucket {        // Bucket definition.
protected:
  CmTHashBucket() : _size(0), _first(NULL), _last(NULL) {}  // Constructor.
 ~CmTHashBucket() { removeAll(); }              // Bucket destructor.

  unsigned size() const { return _size; }       // Return bucket size.

  Bool append   (const T&);                     // Append item to bucket.
  Bool contains (const T&) const;               // Is item in bucket?
  Bool remove   (const T&);                     // Remove item from bucket.
  T*   lookup   (const T&) const;               // Return equal item.
  void removeAll();                             // Remove all items.

  unsigned        _size;                        // Bucket size.
  CmTHashNode<T> *_first;                       // First node in bucket.
  CmTHashNode<T> *_last;                        // Last node in bucket.
  friend          CmTHashTable<T>;              // Table can access.
  friend          CmTHashTableIterator<T>;      // Iterator can access.
};

template <class T> class CmTHashNode {          // Node definition.
protected:
  CmTHashNode(const T& O) : _next(NULL), _data(O) {}  // Constructor.

  CmTHashNode<T> *_next;                        // Next node in bucket.
  T               _data;                        // Item.
  friend          CmTHashBucket<T>;             // Bucket can access.
  friend          CmTHashTableIterator<T>;      // Iterator can access.
};

template <class T> class CmTHashTableIterator : public CmTIterator<T> {
public:
  CmTHashTableIterator(const CmTHashTable<T>&); // Iterator constructor.

  Bool     done    () const;                    // Check if done iterating.
  const T& next    ();                          // Return and advance.
  const T& previous();                          // Return and backup.
  const T& current () const;                    // Return current item.
  void     first   ();                          // Move to first item.
  void     last    ();                          // Move to last item.

protected:
  const CmTHashTable<T>& _table;                // Table being iterated.
  int                    _bucket;               // Current hash bucket.
  CmTHashNode<T>        *_node;                 // Current bucket node.
  friend                 CmTHashTable<T>;       // Table can access.
};

#if defined(__TURBOC__) || defined(__xlC__)
#include <cm/include/cmthash.cc>
#endif

#endif
