// CmRing.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Ring (circular linked list) definition.
// -----------------------------------------------------------------

#ifndef _CMRING_H
#define _CMRING_H

#include <cm/include/cmcont.h>

class CmRingIterator;                        // Ring iterator class stub.
class CmRingNode;                            // Ring node class stub.

class CmRing : public CmContainer {          // Ring definition.
public:
  CmRing() : _top(NULL), _last(NULL) {}      // Default ring constructor.
  CmRing(const CmRing&);                     // Ring copy constructor.
 ~CmRing();                                  // Ring destructor.

  CmRing& operator=(const CmRing&);          // Assignment operator.

  Bool        add        (CmObject*);        // Add object to ring top.
  Bool        remove     (CmObject*);        // Remove equal object.
  Bool        removeTop  ();                 // Remove top object.
  CmObject*   lookup     (CmObject*) const;  // Return equal object.
  Bool        contains   (CmObject*) const;  // See if ring contains equal.
  unsigned    occurrences(CmObject*) const;  // Count number of equal objects.
  void        removeAll  ();                 // Remove all objects from ring.
  CmObject*   top        () const;           // Get pointer to top object.
  CmIterator* newIterator() const;           // Create and return iterator.

  CMOBJECT_DEFINE(CmRing, CmContainer)       // Define object funcs.

protected:
  CmRingNode *_top;                          // Top node in ring.
  CmRingNode *_last;                         // Last node in ring.
  friend      CmRingIterator;                // Iterator can access.
};

class CmRingIterator : public CmIterator {   // Iterator definition.
public:
  CmRingIterator(const CmRing& R)            // Iterator constructor.
                : _ring(R), _node(R._top) {}

  Bool      done    () const;                // See if done iterating (FALSE).
  CmObject* next    ();                      // Get next object in ring.
  CmObject* previous();                      // Return and backup.
  CmObject* current () const;                // Get current ring object.
  void      first   ();                      // Move to first object.
  void      last    ();                      // Move to last object.

  CMOBJECT_DEFINE(CmRingIterator, CmIterator)  // Define object funcs.

protected:
  CmRingNode   *_node;                       // Current ring node.
  const CmRing &_ring;                       // Ring being iterated.
  friend         CmRing;                     // Ring class can access.
};

class CmRingNode {                           // Ring node definition.
protected:
  CmRingNode(CmObject *O)                    // Ring node constructor.
            : _next(NULL), _data(O) {}

  CmRingNode *_next;                         // Next node in ring.
  CmObject   *_data;                         // Pointer to object.
  friend      CmRing;                        // Ring class can access.
  friend      CmRingIterator;                // Ring iterator can access.
};

// "top" returns a pointer to the top object in the ring.
inline CmObject* CmRing::top() const
{ return (_top) ? _top->_data : NULL; }

// "done" checks to see if done iterating.  (Always FALSE for a ring.)
inline Bool CmRingIterator::done() const
{ return FALSE; }

// "current" returns the current object in the ring.
inline CmObject* CmRingIterator::current() const
{ return (_node) ? _node->_data : NULL; }

// "first" resets the iterator to the ring top.
inline void CmRingIterator::first()
{ _node = _ring._top; }

// "last" moves the iterator to the last object.
inline void CmRingIterator::last()
{ _node = _ring._last; }

#endif
