// CmResFil.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Reserve binary file interface definition.
// -----------------------------------------------------------------

#ifndef _CMRESFIL_H
#define _CMRESFIL_H

#include <stdio.h>
#include <cm/include/cmdefs.h>

class CmReserveFile {                         // File class definition.
public:
  CmReserveFile();                            // Default constructor.
  CmReserveFile(const char*);                 // File constructor.
  CmReserveFile(const CmReserveFile&);        // Copy constructor.
 ~CmReserveFile();                            // File destructor.

  CmReserveFile& operator=(const CmReserveFile&);  // Assignment.

  Bool open (const char*);                    // Open named file.
  Bool close();                               // Close the file.

  operator int() const;                       // Check if file is open.
  Bool isValid() const;                       // Check if file is open.

  const char* name() const;                   // Return file name.

  Bool remove();                              // Close and remove file.
  Bool rename(const char*);                   // Rename the file.

  size_t read (void*, size_t, size_t);        // Read buffers of size.
  Bool   read (void*, size_t);                // Read one buffer of size.
  size_t write(void*, size_t, size_t);        // Write buffers of size.
  Bool   write(void*, size_t);                // Write one buffer of size.

  Bool read(char&);                           // Read one char.
  Bool read(short&);                          // Read one short.
  Bool read(int&);                            // Read one integer.
  Bool read(long&);                           // Read one long.
  Bool read(unsigned char&);                  // Read one unsigned char.
  Bool read(unsigned short&);                 // Read one unsigned short.
  Bool read(unsigned int&);                   // Read one unsigned int.
  Bool read(unsigned long&);                  // Read one unsigned long.
  Bool read(float&);                          // Read one float.
  Bool read(double&);                         // Read one double.
  Bool read(char*,   unsigned);               // Read a char array.
  Bool read(short*,  unsigned);               // Read a short array.
  Bool read(int*,    unsigned);               // Read an int array.
  Bool read(long*,   unsigned);               // Read a long array.
  Bool read(float*,  unsigned);               // Read a float array.
  Bool read(double*, unsigned);               // Read a double array.

  Bool write(char);                           // Write one char.
  Bool write(short);                          // Write one short.
  Bool write(int);                            // Write one integer.
  Bool write(long);                           // Write one long.
  Bool write(unsigned char);                  // Write one unsigned char.
  Bool write(unsigned short);                 // Write one unsigned short.
  Bool write(unsigned int);                   // Write one unsigned int.
  Bool write(unsigned long);                  // Write one unsigned long.
  Bool write(float);                          // Write one float.
  Bool write(double);                         // Write one double.
  Bool write(char*,   unsigned);              // Write a char array.
  Bool write(short*,  unsigned);              // Write a short array.
  Bool write(int*,    unsigned);              // Write an int array.
  Bool write(long*,   unsigned);              // Write a long array.
  Bool write(float*,  unsigned);              // Write a float array.
  Bool write(double*, unsigned);              // Write a double array.

  Bool  write(const char*);                   // Write strlen and string.
  char* read ();                              // Read string written by above.

  int  seek  (long, int);                     // Set file position for stream.
  long tell  ();                              // Return current file position.
  void rewind();                              // Set position to beginning.
  int  getPos(fpos_t*);                       // Get current position in ptr.
  int  setPos(const fpos_t*);                 // Set position to ptr.

  void clearError();                          // Clear all error indicators.
  Bool eof       ();                          // Check for end of file.
  Bool error     ();                          // Check for error.
  void printError(const char*);               // Print error message.

  static Bool  exists  (const char*);         // See if named file exists.
  static Bool  remove  (const char*);         // Remove named file from disk.
  static char* tempName(char*);               // Create non-existing file name.

  CmReserveFile& operator>>(char&);           // Read one char.
  CmReserveFile& operator>>(short&);          // Read one short.
  CmReserveFile& operator>>(int&);            // Read one integer.
  CmReserveFile& operator>>(long&);           // Read one long.
  CmReserveFile& operator>>(unsigned char&);  // Read one unsigned char.
  CmReserveFile& operator>>(unsigned short&); // Read one unsigned short.
  CmReserveFile& operator>>(unsigned int&);   // Read one unsigned int.
  CmReserveFile& operator>>(unsigned long&);  // Read one unsigned long.
  CmReserveFile& operator>>(float&);          // Read one float.
  CmReserveFile& operator>>(double&);         // Read one double.
  CmReserveFile& operator>>(char*&);          // Read strlen and string.

  CmReserveFile& operator<<(char);            // Write one char.
  CmReserveFile& operator<<(short);           // Write one short.
  CmReserveFile& operator<<(int);             // Write one integer.
  CmReserveFile& operator<<(long);            // Write one long.
  CmReserveFile& operator<<(unsigned char);   // Write one unsigned char.
  CmReserveFile& operator<<(unsigned short);  // Write one unsigned short.
  CmReserveFile& operator<<(unsigned int);    // Write one unsigned int.
  CmReserveFile& operator<<(unsigned long);   // Write one unsigned long.
  CmReserveFile& operator<<(float);           // Write one float.
  CmReserveFile& operator<<(double);          // Write one double.
  CmReserveFile& operator<<(const char*);     // Write strlen and string.

protected:
  char *_name;                                // File name.
  FILE *_file;                                // File stream pointer.
};

// "isValid" checks if the file was successfully opened.
inline Bool CmReserveFile::isValid() const
{ return (_file) ? TRUE : FALSE; }

// "int" operator checks if the file is currently open.
inline CmReserveFile::operator int() const
{ return isValid(); }

// "name" returns the file name.
inline const char* CmReserveFile::name() const
{ return _name; }

// ">>" operator reads one character.
inline CmReserveFile& CmReserveFile::operator>>(char& c)
{ read(c); return *this; }

// ">>" operator reads one short.
inline CmReserveFile& CmReserveFile::operator>>(short& s)
{ read(s); return *this; }

// ">>" operator reads one integer.
inline CmReserveFile& CmReserveFile::operator>>(int& i)
{ read(i); return *this; }

// ">>" operator reads one long.
inline CmReserveFile& CmReserveFile::operator>>(long& l)
{ read(l); return *this; }

// ">>" operator reads one unsigned character.
inline CmReserveFile& CmReserveFile::operator>>(unsigned char& uc)
{ read(uc); return *this; }

// ">>" operator reads one unsigned short.
inline CmReserveFile& CmReserveFile::operator>>(unsigned short& us)
{ read(us); return *this; }

// ">>" operator reads one unsigned integer.
inline CmReserveFile& CmReserveFile::operator>>(unsigned int& ui)
{ read(ui); return *this; }

// ">>" operator reads one unsigned long.
inline CmReserveFile& CmReserveFile::operator>>(unsigned long& ul)
{ read(ul); return *this; }

// ">>" operator reads one float.
inline CmReserveFile& CmReserveFile::operator>>(float& f)
{ read(f); return *this; }

// ">>" operator reads one long.
inline CmReserveFile& CmReserveFile::operator>>(double& d)
{ read(d); return *this; }

// ">>" operator reads string length and string text.
inline CmReserveFile& CmReserveFile::operator>>(char*& s)
{ s = read(); return *this; }

// "<<" operator writes one character.
inline CmReserveFile& CmReserveFile::operator<<(char c)
{ write(c); return *this; }

// "<<" operator writes one short.
inline CmReserveFile& CmReserveFile::operator<<(short s)
{ write(s); return *this; }

// "<<" operator writes one integer.
inline CmReserveFile& CmReserveFile::operator<<(int i)
{ write(i); return *this; }

// "<<" operator writes one long.
inline CmReserveFile& CmReserveFile::operator<<(long l)
{ write(l); return *this; }

// "<<" operator writes one unsigned character.
inline CmReserveFile& CmReserveFile::operator<<(unsigned char uc)
{ write(uc); return *this; }

// "<<" operator writes one unsigned short.
inline CmReserveFile& CmReserveFile::operator<<(unsigned short us)
{ write(us); return *this; }

// "<<" operator writes one unsigned integer.
inline CmReserveFile& CmReserveFile::operator<<(unsigned int ui)
{ write(ui); return *this; }

// "<<" operator writes one unsigned long.
inline CmReserveFile& CmReserveFile::operator<<(unsigned long ul)
{ write(ul); return *this; }

// "<<" operator writes one unsigned float.
inline CmReserveFile& CmReserveFile::operator<<(float f)
{ write(f); return *this; }

// "<<" operator writes one unsigned double.
inline CmReserveFile& CmReserveFile::operator<<(double d)
{ write(d); return *this; }

// "<<" outputs the string length and string text.
inline CmReserveFile& CmReserveFile::operator<<(const char* s)
{ write(s); return *this; }

#endif
