// CmReserv.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// In-memory data base definition.
// -----------------------------------------------------------------

#ifndef _CMRESERV_H
#define _CMRESERV_H

#include <cm/include/cmbdict.h>
#include <cm/include/cmlist.h>
#include <cm/include/cmbtree.h>
#include <cm/include/cmstring.h>

class CmReserve {                               // Reserve definition.
public:
  CmReserve(const char*);                       // Reserve constructor.
  CmReserve(const CmReserve&);                  // Reserve copy constructor.
 ~CmReserve();                                  // Reserve destructor.

  CmReserve& operator=(const CmReserve&);       // Assignment operator.

  Bool exists() const;                          // Check file status.

  void        name(const char*);                // Set new reserve name.
  const char* name() const;                     // Get reserve name.

  void        identifier(const char*);          // Set reserve identifier.
  const char* identifier() const;               // Get reserve identifier.

  Bool         addRoot     (const char*, CmContainer*); // Add root to reserve.
  CmContainer* createRoot  (const char*);       // Create a new root.
  int          totalRoots  () const;            // Get number of roots.
  Bool         containsRoot(const char*) const; // See if root is in reserve.
  CmContainer* getRoot     (const char*) const; // Get a root by index.
  Bool         removeRoot  (const char*);       // Remove root by index.
  void         removeAll   ();                  // Remove all roots.

  CmContainer* operator[](const char*) const;   // Get a root by name.

  virtual Bool write() const;                       // Write reserve to file.
  virtual Bool read ();                             // Read reserve from file.

  // Call the specified function for every object in the reserve
  // until the end is reached or the function returns FALSE.
  void forEach(CmQueryFunc,    void* = NULL);
  void forEach(CmQueryMemFunc, void* = NULL);

  // Return the first object to satisfy the conditional test performed
  // in the specified function.
  CmObject* firstThat(CmQueryFunc,    void* = NULL);
  CmObject* firstThat(CmQueryMemFunc, void* = NULL);

  // Call the specified function for every object in the reserve
  // adding all objects for which the function returns TRUE to an
  // output container.
  CmContainer* query(CmQueryFunc,    void* = NULL); // Create subset.
  CmContainer* query(CmQueryMemFunc, void* = NULL); // Create subset.

  static void      initialize      ();              // Register Cm classes.
  static char*     fileIdentifier  (const char*);   // Get an identifier.
  static Bool      registerClass   (CmObject*);     // Register a class.
  static CmObject* getClassRegister(const char*);   // Get class register.
  static void      resetTable      ();              // Reset reference table.

  static void maintainReferences(Bool);             // Set reference flag.
  static Bool maintainReferences();                 // Get reference flag.

  enum ref_flag { OBJECT=0, REFERENCE=1 };          // Reference flag.

private:
  static Bool      addToTable  (CmObject*,int); // Add object to table.
  static CmObject* getFromTable(int);           // Get object at index.
  static int       getFromTable(CmObject*);     // Get object at index.

  CmString                   _name;             // Reserve name.
  CmString                   _identifier;       // Reserve identifier.
  CmBTreeDictionary         *_roots;            // Root list.
  static CmLinkedList       *_classes;          // Registered class list.
  static Bool                _references;       // References allowed flag.
  friend                     CmObject;          // Object can access.

  static CmBTree *_referenceTable;              // I/O reference table.
};

// "name" sets a new reserve name.
inline void CmReserve::name(const char* name)
{ _name = name; }

// "name" returns the current reserve name.
inline const char* CmReserve::name() const
{ return _name; }

// "identifier" sets a new reserve identifier.
inline void CmReserve::identifier(const char* ident)
{ _identifier = ident; }

// "identifier" returns the current reserve identifier.
inline const char* CmReserve::identifier() const
{ return _identifier; }

// "[]" indexing operator returns a pointer to the root by name.
inline CmContainer* CmReserve::operator[](const char* name) const
{ return getRoot(name); }

#endif
