// CmObject.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Abstract object definition.
// -----------------------------------------------------------------

#ifndef _CMOBJECT_H
#define _CMOBJECT_H

#include <string.h>
#include <iostream.h>
#include <cm/include/cmdefs.h>
#include <cm/include/cmresfil.h>

class CmObject {                                       // Object class definition.
public:
  CmObject() {}                                        // Default constructor.
 
  virtual ~CmObject() {}                               // Virtual destructor.

  virtual const char* isA     ()            const = 0; // Return class name.
  virtual Bool        isA     (const char*) const = 0; // Compare class names.
  virtual Bool        isTypeOf(const char*) const;     // Check inheritance.
  virtual CmObject*   newCopy ()            const = 0; // Virtual copy ctor.

  virtual Bool     isEqual (CmObject*) const;          // Compare objects.
  virtual int      compare (CmObject*) const;          // Compare objects.
  virtual unsigned hash    (unsigned)  const;          // Hash function.
  virtual void     printOn (ostream&)  const {}        // Print to stream.
  virtual void     readFrom(istream&) {}               // Read from stream.
  virtual Bool     write   (CmReserveFile&) const;     // Write to reserve.
  virtual Bool     read    (CmReserveFile&);           // Read from reserve.

  static CmObject* readObject (CmReserveFile&);        // Read an object.
  Bool             writeObject(CmReserveFile&);        // Write an object.
};

// "isTypeOf" checks if this class is the same type as the specified name.
inline Bool CmObject::isTypeOf(const char* s) const
{ return (!strcmp(s, "CmObject")); }

// "isEqual" function used by containers for object lookup.
inline Bool CmObject::isEqual(CmObject* O) const
{ return (O == this); }

// "compare" used by containers for object compare (binary tree, etc.).
inline int CmObject::compare(CmObject* O) const
{ return ((O == this) ? 0 : ((this > O) ? 1 : -1)); }

// "hash" hashing function used by hash table classes.
inline unsigned CmObject::hash(unsigned m) const
{ return (unsigned) this % m; }

// "write" writes objects to the specified reserve file.
inline Bool CmObject::write(CmReserveFile&) const
{ return TRUE; }

// "read" reads objects from the specified reserve file.
inline Bool CmObject::read(CmReserveFile&)
{ return TRUE; }

// "<<" global output stream operator calls the object's print function.
inline ostream& operator<<(ostream& strm, const CmObject& O)
{ O.printOn(strm); return strm; }

// ">>" global input stream operator calls the object's read function.
inline istream& operator>>(istream& strm, CmObject& O)
{ O.readFrom(strm); return strm; }

// "<<" global output operator calls the object's write function.
inline CmReserveFile& operator<<(CmReserveFile& F, CmObject& O)
{ O.write(F); return F; }

// "<<" global input operator calls the object's read function.
inline CmReserveFile& operator>>(CmReserveFile& F, CmObject& O)
{ O.read(F); return F; }

// Shorthand macro for defining the pure virtual functions contained within
// the "CmObject" definition.  The classname and parent classname are passed
// as parameters to the macro.
//
// NOTE: This macro will not work with derived class that still contain
//       thier own pure virtual functions.  See the macro definition
//       below.  Also, for classes deriving from more than one "CmObject"
//       classes, these macros cannot be used meaning that the programmer
//       needs to put in the definitions by hand.
//
#define CMOBJECT_DEFINE(name, parent)                                 \
  const char* isA() const                                             \
        { return #name; }                                             \
  Bool  isA(const char* s) const                                      \
        { return (!strcmp(s, #name)); }                               \
  Bool  isTypeOf(const char* s) const                                 \
        { return (!strcmp(s, #name)) ? TRUE : parent::isTypeOf(s); }  \
  CmObject* newCopy() const { return (CmObject*) new name(*this); }

// Shorthand macro for defining the pure virtual functions contained within
// the "CmObject" class.  This macro is to be used when deriving classes
// which still contain their own pure virtual functions.
//
#define CMOBJECT_ABSTRACT(name, parent)                               \
  const char* isA() const                                             \
        { return #name; }                                             \
  Bool  isA(const char* s) const                                      \
        { return (!strcmp(s, #name)); }                               \
  Bool  isTypeOf(const char* s) const                                 \
        { return (!strcmp(s, #name)) ? TRUE : parent::isTypeOf(s); }  \
  virtual CmObject* newCopy() const = 0;

#endif
