// CmDate.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Date class definition.
// -----------------------------------------------------------------

#ifndef _CMDATE_H
#define _CMDATE_H

#include <cm/include/cmobject.h>

class CmDate : public CmObject {                  // Date class definition.
public:
  CmDate();                                       // Default date constructor.
  CmDate(unsigned, unsigned);                     // Construct from d, y.
  CmDate(unsigned, const char*, unsigned);        // Construct from d, m, y.
  CmDate(unsigned, unsigned, unsigned);           // Construct from d, m, y.
  CmDate(unsigned long j) : _julnum(j) {}         // Construct from jul num.

  unsigned    day            () const;            // Get day of year number.
  unsigned    dayOfMonth     () const;            // Get day of month number.
  unsigned    firstDayOfMonth() const;            // Get day num first of month.
  unsigned    firstDayOfMonth(unsigned) const;    // Get day num first of month.
  unsigned    month          () const;            // Get date's month number.
  const char* nameOfDay      () const;            // Get date's day name.
  const char* nameOfMonth    () const;            // Get date's month name.
  unsigned    weekDay        () const;            // Get date's weekday number.
  unsigned    year           () const;            // Get date's year number.
  Bool        leap           () const;            // Check if leap year.
  CmDate      previous       (const char*) const; // Get previous named day.
  CmDate      previous       (unsigned) const;    // Get previous numbered day.
  void        makeCurrent    ();                  // Make this date current.

  Bool     isEqual(CmObject*) const;              // Compare dates.
  int      compare(CmObject*) const;              // Compare dates.
  unsigned hash   (unsigned)  const;              // Hash date.
  void     printOn(ostream&)  const;              // Print date to stream.
  Bool     write  (CmReserveFile&) const;         // Write date to file.
  Bool     read   (CmReserveFile&);               // Read date from file.

  CMOBJECT_DEFINE(CmDate, CmObject)               // Define object funcs.

  unsigned long julNum() const;                   // Return julian number.

  Bool operator< (const CmDate&) const;           // Check if date <  this.
  Bool operator<=(const CmDate&) const;           // Check if date <= this.
  Bool operator> (const CmDate&) const;           // Check if date >  this.
  Bool operator>=(const CmDate&) const;           // Check if date >= this.
  Bool operator==(const CmDate&) const;           // Check if date == this.
  Bool operator!=(const CmDate&) const;           // Check if date != this.

  CmDate operator+(const CmDate&) const;          // Add dates.
  CmDate operator+(int)           const;          // Add julnum to date.
  CmDate operator-(const CmDate&) const;          // Subtract dates.
  CmDate operator-(int)           const;          // Sub julnum from date.

  static unsigned long julDay   (unsigned, unsigned, unsigned); // Get julnum.
  static unsigned dayOfWeek     (const char*);    // Return day of week num.
  static Bool     dayWithinMonth(unsigned, unsigned, unsigned); // Get day num.
  static Bool     leapYear      (unsigned);       // Check if leap year.
  static unsigned indexOfMonth  (const char*);    // Get month number.

  static const char* dayName  (unsigned);         // Get day name from number.
  static const char* monthName(unsigned);         // Get month name from num.

  static void  displayStyle(int);                 // Set display style for all.
  static int   displayStyle();                    // Get display style.

  enum display_style { NORMAL=0, NUMBERS=1 };     // Display styles.

private:
  void mdy(unsigned&, unsigned&, unsigned&) const; // Get date from julnum.

  unsigned long _julnum;                          // Julian day number.
  static   int  _displayStyle;                    // Date display style.
};

// "julNum" returns the julian day number.
inline unsigned long CmDate::julNum() const
{ return _julnum; }

// "<" checks if the specified date is less than this.
inline Bool CmDate::operator<(const CmDate& D) const
{ return (_julnum < D._julnum); }

// "<=" checks if the specified date is less than or equal to this.
inline Bool CmDate::operator<=(const CmDate& D) const
{ return (_julnum <= D._julnum); }

// ">" checks if the specified date is greater than this.
inline Bool CmDate::operator>(const CmDate& D) const
{ return (_julnum > D._julnum); }

// ">=" checks if the specified date is greater than or equal to this.
inline Bool CmDate::operator>=(const CmDate& D) const
{ return (_julnum >= D._julnum); }

// "==" checks if the specified date is equal to this.
inline Bool CmDate::operator==(const CmDate& D) const
{ return (_julnum == D._julnum); }

// "!=" checks if the specified date is not equal to this.
inline Bool CmDate::operator!=(const CmDate& D) const
{ return (_julnum != D._julnum); }

// "+" adds two dates returning the resulting date.
inline CmDate CmDate::operator+(const CmDate& D) const
{ return CmDate(_julnum + D._julnum); }

// "+" adds a julian day number to this date returning the new date.
inline CmDate CmDate::operator+(int d) const
{ return CmDate(_julnum + d); }

// "-" subtracts the input date from this returning the resulting date.
inline CmDate CmDate::operator-(const CmDate& D) const
{ return CmDate(_julnum - D._julnum); }

// "-" subtracts a julian day number from this returning the new date.
inline CmDate CmDate::operator-(int d) const
{ return CmDate(_julnum - d); }

// "displayStyle" sets a new display style for all dates.
inline void CmDate::displayStyle(int ds)
{ CmDate::_displayStyle = ds; }

// "displayStyle" returns the current display style.
inline int CmDate::displayStyle()
{ return CmDate::_displayStyle; }

#endif
