// CmCont.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Abstract container definition.
// -----------------------------------------------------------------

#ifndef _CMCONT_H
#define _CMCONT_H

#include <cm/include/cmiter.h>

typedef Bool (*CmQueryFunc)(CmObject*, void*);          // Query function type.
typedef Bool (CmObject::*CmQueryMemFunc)(void*);        // Query member function.

class CmContainer : public CmObject {                   // Container definition.
public:
  CmContainer() : _size(0), _ownsObjects(TRUE) {}       // Constructor.

  virtual ~CmContainer() {}                             // Virtual destructor.

  virtual CmObject*   operator[] (int) const;           // Get object by index.
  virtual unsigned    size       () const;              // Return size.
  virtual Bool        add        (CmObject*) = 0;       // Add object.
  virtual Bool        remove     (CmObject*) = 0;       // Remove equal object.
  virtual CmObject*   lookup     (CmObject*) const = 0; // Find equal object.
  virtual Bool        contains   (CmObject*) const = 0; // Container has object?
  virtual unsigned    occurrences(CmObject*) const = 0; // Count occurrences.
  virtual void        removeAll  () = 0;                // Remove all objects.
  virtual Bool        isEmpty    () const;              // Is container empty?
  virtual CmIterator* newIterator() const = 0;          // Get new iterator.

  CmContainer* lookupAll (CmObject*) const;             // Find all equal objs.
  Bool         addAllFrom(CmContainer*);                // Combine containers.

  void ownsObjects(Bool);                               // Set ownership policy.
  Bool ownsObjects() const;                             // Get ownership policy.

  void printOn(ostream&) const;                         // Print to stream.
  Bool write  (CmReserveFile&) const;                   // Write to reserve.
  Bool read   (CmReserveFile&);                         // Read from reserve.

  CMOBJECT_ABSTRACT(CmContainer, CmObject)              // Define object funcs.

  // Call the specified function for every object in the container
  // until the end is reached or the function returns FALSE.
  void forEach(CmQueryFunc,    void* = NULL);
  void forEach(CmQueryMemFunc, void* = NULL);

  // Return the first object to satisfy the conditional test performed
  // in the specified function.
  CmObject* firstThat(CmQueryFunc,    void* = NULL);
  CmObject* firstThat(CmQueryMemFunc, void* = NULL);

  // Call the specified function for every object in the container
  // adding all objects for which the function returns TRUE to an
  // output container.
  CmContainer* query(CmQueryFunc,    void* = NULL);
  CmContainer* query(CmQueryMemFunc, void* = NULL);

protected:
  virtual void copy(const CmContainer&);                // Copy method.

  Bool     writeBase(CmReserveFile&) const;             // Write base container.
  unsigned readBase (CmReserveFile&);                   // Read base container.

  unsigned _size;                                       // Current size.
  Bool     _ownsObjects;                                // Ownership flag.
};

// "size" returns the container size.
inline unsigned CmContainer::size() const
{ return _size; }

// "isEmpty" checks if the container is empty or not.
inline Bool CmContainer::isEmpty() const
{ return (_size == 0); }

// "ownsObjects" sets a new ownership policy for this container.
inline void CmContainer::ownsObjects(Bool O)
{ _ownsObjects = O; }

// "ownsObjects" returns the current ownership policy for this container.
inline Bool CmContainer::ownsObjects() const
{ return _ownsObjects; }

// "CmForEach" and "CmEndFor" define a foreach loop where the object
// pointer and collection are passed in and the object pointer will
// point to the next object in the collection for each loop iteration.
//
#define CmForEach(obj, cltn) {                           \
        CmIterator *___cm_iterator = cltn.newIterator(); \
        while (!___cm_iterator->done()) {                \
          obj = ___cm_iterator->next();
#define CmEndFor } delete ___cm_iterator; }

// "CmRepeat" and "CmUntil" define a pascal style repeat/until loop
// where the object pointer and collection are passed in and the object
// pointer will point to the next object in the collection for each loop
// iteration.
//
#define CmRepeat(obj, cltn) {                            \
        CmIterator *___cm_iterator = cltn.newIterator(); \
        do {                                             \
          obj = ___cm_iterator->next();
#define CmUntil(condition) } while (!(condition)); }

// "CmDo" and "CmWhile" define a typical do/while loop where the object
// pointer and collection are passed in and the object pointer will point
// to the next object in the collection for each loop iteration.
//
#define CmDo(obj, cltn) {                                \
        CmIterator *___cm_iterator = cltn.newIterator(); \
        do {                                             \
          obj = ___cm_iterator->next();
#define CmWhile(condition) } while (condition); }

#endif
