// CmBinTr.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Binary tree definition.
// -----------------------------------------------------------------

#ifndef _CMBINTR_H
#define _CMBINTR_H

#include <cm/include/cmcont.h>
#include <cm/include/cmtstack.h>
#include <cm/include/cmtqueue.h>

class CmBinaryTreeNode;                            // Tree node class stub.
class CmBinaryTreeIterator;                        // Iterator class stub.

class CmBinaryTree : public CmContainer {          // Tree class definition.
public:
  CmBinaryTree() : _root(NULL) {}                  // Default tree constructor.
  CmBinaryTree(const CmBinaryTree&);               // Tree copy constructor.
 ~CmBinaryTree();                                  // Tree destructor.

  CmBinaryTree& operator=(const CmBinaryTree&);    // Assignment operator.

  Bool        add        (CmObject*);              // Add object to tree.
  Bool        remove     (CmObject*);              // Remove equal object.
  CmObject*   lookup     (CmObject*) const;        // Locate equal object.
  Bool        contains   (CmObject*) const;        // Tree has object?
  unsigned    occurrences(CmObject*) const;        // Count occurrences of.
  void        removeAll  ();                       // Clear tree contents.
  CmObject*   root       () const;                 // Return root object.
  void        balance    ();                       // Balance the tree.
  CmIterator* newIterator() const;                 // Get tree iterator.
  Bool        write      (CmReserveFile&) const;   // Write to reserve file.
  Bool        read       (CmReserveFile&);         // Read from reserve file.

  CMOBJECT_DEFINE(CmBinaryTree, CmContainer)       // Define object funcs.

protected:
  void balanceHalf(CmTQueue<CmBinaryTreeNode*>*, int, int);  // Balance tree.

  static void killNode(CmBinaryTreeNode*, Bool);   // Destroy sub-nodes.

  CmBinaryTreeNode *_root;                         // Tree root node.
  friend            CmBinaryTreeIterator;          // Iterator can access.
};

class CmBinaryTreeIterator : public CmIterator {   // Iterator definition.
public:
  CmBinaryTreeIterator(const CmBinaryTree &Tr);    // Constructor.
 ~CmBinaryTreeIterator();                          // Destructor.

  Bool      done    () const { return (!_node); }  // Check if at end.
  CmObject* next    ();                            // Return and advance.
  CmObject* previous();                            // Return and backup.
  CmObject* current () const;                      // Get current object.
  void      first   ();                            // Move to first object.
  void      last    ();                            // Move to last object.

  CMOBJECT_DEFINE(CmBinaryTreeIterator, CmIterator)  // Define object funcs.

protected:
  const CmBinaryTree          &_tree;              // Tree being iterated.
  CmBinaryTreeNode            *_node;              // Current tree node.
  CmTStack<CmBinaryTreeNode*> *_stack;             // Node stack.
  friend                       CmBinaryTree;       // Tree class can access.
};

class CmBinaryTreeNode {                           // Tree node class.
protected:
  CmBinaryTreeNode *_left;                         // Left subtree pointer.
  CmObject         *_data;                         // Object pointer.
  CmBinaryTreeNode *_right;                        // Right subtree pointer.
  friend            CmBinaryTree;                  // Tree can access.
  friend            CmBinaryTreeIterator;          // Iterator can access.
 
  CmBinaryTreeNode(CmObject *O)                    // Node constructor.
                  : _left(NULL), _data(O), _right(NULL) {}
};

// "root" returns the root object of the tree.
inline CmObject* CmBinaryTree::root() const
{ return (_root) ? _root->_data : NULL; }

// "current" returns the current tree object.
inline CmObject* CmBinaryTreeIterator::current() const
{ return (_node) ? _node->_data : NULL; }

#endif
