// CmTokens.cpp
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// String tokenizer implementation.
// -----------------------------------------------------------------

#include <cm/include/cmtokens.h>


// "CmTokens" is the tokenizer constructor.
//
CmTokens::CmTokens(const char* str)
{
  _string = str;
  _place  = _string.text();
  _idx    = 0;
}


// "CmTokens" is the tokenizer copy constructor.
//
CmTokens::CmTokens(const CmTokens& tk)
{
  _string = tk._string;
  _place  = _string.text();
  _idx    = 0;
}


// "=" assignment operator copies the specified tokenizer into this
// tokenizer.
//
CmTokens& CmTokens::operator=(const CmTokens& tk)
{
  if (&tk != this)
  {
    _string = tk._string;
    _place  = _string.text();
    _idx    = 0;
  }
  return *this;
}


// "next" searches for and returns the next token in the string using
// the input char string as the delimiter.
//
const char* CmTokens::next(const char *del)
{
  if (!*_place) return CmString("");

  _idx   += strspn(_place, del);
  _place += strspn(_place, del);

  if (!*_place) return CmString("");

  const char* s = strpbrk(_place, del);
  unsigned extent = s ? s - _place : strlen(_place);

  int start = _idx;
  int stop  = _idx + extent - 1;

  _place += extent;
  _idx   += extent;
  return (const char*) CmString(_string(start, stop));
}


// "reset" resets the tokenizer to the beginning of the string.
//
void CmTokens::reset()
{
  _place = _string.text();
  _idx   = 0;
}


// "isEqual" compares the string with the input tokenizer string.
//
Bool CmTokens::isEqual(CmObject* pObj) const
{
  if (!pObj->isA("CmTokens")) return CmObject::isEqual(pObj);
  CmTokens& tk = (CmTokens&) *pObj;
  return (_string.isEqual(&(tk._string)));
}


// "compare" compares the string with the input tokenizer string.
//
int CmTokens::compare(CmObject* pObj) const
{
  if (!pObj->isA("CmTokens")) return CmObject::compare(pObj);
  CmTokens& tk = (CmTokens&) *pObj;
  return (_string.compare(&(tk._string)));
}


// "hash" hashes on the string.
//
unsigned CmTokens::hash(unsigned m) const
{
  return _string.hash(m);
}


// "printOn" prints the string to the specified output stream.
//
void CmTokens::printOn(ostream& os) const
{
  _string.printOn(os);
}
