// CmResFil.cpp
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Reserve binary file interface implementation.
// -----------------------------------------------------------------

#include <string.h>
#include <cm/include/cmresfil.h>


// Define a file buffer size of 16k.
//
#define CM_BUF_SIZE 16384


// "CmReserveFile" is the default file constructor.
//
CmReserveFile::CmReserveFile()
{
  _name = NULL;
  _file = NULL;
}


// "CmReserveFile" is the file constructor.  The file is opened/created at
// construct time and the error parameter signals success or failure.
//
CmReserveFile::CmReserveFile(const char* filename)
{
  _file = NULL;
  _name = (strlen(filename) > 0) ? new char[strlen(filename)+1] : NULL;
  if (_name)
  {
    strcpy(_name, filename);
    _file = fopen(_name, "rb+");
    if (!_file) _file = fopen(_name, "wb+");
    if (_file) setvbuf(_file, NULL, _IOFBF, CM_BUF_SIZE);
  }
}


// "CmReserveFile" is the file copy constructor.  An invalid file
// is created.
//
CmReserveFile::CmReserveFile(const CmReserveFile& F)
{
  _name = (strlen(F._name) > 0) ? new char[strlen(F._name)+1] : NULL;
  strcpy(_name, F._name);
  _file = NULL;
}


// "~CmReserveFile" is the file destructor.  The file is closed at
// destruct time.
//
CmReserveFile::~CmReserveFile()
{
  delete[] _name;
  if (_file) fclose(_file);
}


// "=" assignment operator copies the file name from the specified
// file.  However, an invalid file is created.
//
CmReserveFile& CmReserveFile::operator=(const CmReserveFile& F)
{
  if (&F != this)
  {
    if (_file) fclose(_file);
    delete[] _name;
    _name = (strlen(F._name) > 0) ? new char[strlen(F._name)+1] : NULL;
    strcpy(_name, F._name);
    _file = NULL;
  }
  return *this;
}


// "open" opens the named file.
//
Bool CmReserveFile::open(const char* filename)
{
  if (_file) return FALSE;

  if (_name) delete[] _name;
  _name = (strlen(filename) > 0) ? new char[strlen(filename)+1] : NULL;

  if (_name)
  {
    strcpy(_name, filename);
    _file = fopen(_name, "rb+");
    if (!_file) _file = fopen(_name, "wb+");
    if (_file) setvbuf(_file, NULL, _IOFBF, CM_BUF_SIZE);
  }
  return (_file) ? TRUE : FALSE;
}


// "close" closes the file.
//
Bool CmReserveFile::close()
{
  int out = 1;
  if (_file)
  {
    out = fclose(_file);
    _file = NULL;
  }
  return (out == 0) ? TRUE : FALSE;
}


// "remove" closes and removes this file from the disk.
//
Bool CmReserveFile::remove()
{
  if (!_file) return FALSE;
  close();
  return (::remove(_name) == 0) ? TRUE : FALSE;
}


// "rename" renames this file.
//
Bool CmReserveFile::rename(const char* newname)
{
  if (::rename(_name, newname) != 0) return FALSE;
  delete[] _name;
  _name = (strlen(newname) > 0) ? new char[strlen(newname)+1] : NULL;
  if (_name) strcpy(_name, newname);
  return (_name) ? TRUE : FALSE;
}


// "read" is the direct input function for this file.
//
size_t CmReserveFile::read(void* ptr, size_t size, size_t nobj)
{
  return (_file) ? fread(ptr, size, nobj, _file) : 0;
}


// "read" is a direct input function reading only one object.
//
Bool CmReserveFile::read(void* ptr, size_t size)
{
  return (!_file) ? FALSE : ((fread(ptr, size, 1, _file) == 1) ? TRUE : FALSE);
}


// "write" is the direct output function for this file.
//
size_t CmReserveFile::write(void* ptr, size_t size, size_t nobj)
{
  return (_file) ? fwrite(ptr, size, nobj, _file) : 0;
}


// "write" is a direct output function writing only one object.
//
Bool CmReserveFile::write(void* ptr, size_t size)
{
  return (!_file) ? FALSE : ((fwrite(ptr, size, 1, _file) == 1) ? TRUE : FALSE);
}


// "read" reads one character from the file.
//
Bool CmReserveFile::read(char& c)
{
  return read((void*) &c, sizeof(char));
}


// "read" reads one short from the file.
//
Bool CmReserveFile::read(short& s)
{
  return read((void*) &s, sizeof(short));
}


// "read" reads one integer from the file.
//
Bool CmReserveFile::read(int& i)
{
  return read((void*) &i, sizeof(int));
}


// "read" reads one unsigned character from the file.
//
Bool CmReserveFile::read(long& l)
{
  return read((void*) &l, sizeof(long));
}


// "read" reads one unsigned character from the file.
//
Bool CmReserveFile::read(unsigned char& uc)
{
  return read((void*) &uc, sizeof(unsigned char));
}


// "read" reads one unsigned character short the file.
//
Bool CmReserveFile::read(unsigned short& us)
{
  return read((void*) &us, sizeof(unsigned short));
}


// "read" reads one unsigned int from the file.
//
Bool CmReserveFile::read(unsigned int& ui)
{
  return read((void*) &ui, sizeof(unsigned int));
}


// "read" reads one unsigned long from the file.
//
Bool CmReserveFile::read(unsigned long& ul)
{
  return read((void*) &ul, sizeof(unsigned long));
}


// "read" reads one float from the file.
//
Bool CmReserveFile::read(float& f)
{
  return read((void*) &f, sizeof(float));
}


// "read" reads one double from the file.
//
Bool CmReserveFile::read(double& d)
{
  return read((void*) &d, sizeof(double));
}


// "read" reads an array of characters from the file.
//
Bool CmReserveFile::read(char* c, unsigned u)
{
  return (read(c, sizeof(char), u) == u);
}


// "read" reads an array of shorts from the file.
//
Bool CmReserveFile::read(short* s, unsigned u)
{
  return (read(s, sizeof(short), u) == u);
}


// "read" reads an array of integers from the file.
//
Bool CmReserveFile::read(int* i, unsigned u)
{
  return (read(i, sizeof(int), u) == u);
}


// "read" reads an array of longs from the file.
//
Bool CmReserveFile::read(long* l, unsigned u)
{
  return (read(l, sizeof(long), u) == u);
}


// "read" reads an array of floats from the file.
//
Bool CmReserveFile::read(float* f, unsigned u)
{
  return (read(f, sizeof(float), u) == u);
}


// "read" reads an array of doubles from the file.
//
Bool CmReserveFile::read(double* d, unsigned u)
{
  return (read(d, sizeof(double), u) == u);
}


// "write" writes one character to the file.
//
Bool CmReserveFile::write(char c)
{
  return write((void*) &c, sizeof(char));
}


// "write" writes one short to the file.
//
Bool CmReserveFile::write(short s)
{
  return write((void*) &s, sizeof(short));
}


// "write" writes one integer to the file.
//
Bool CmReserveFile::write(int i)
{
  return write((void*) &i, sizeof(int));
}


// "write" writes one long to the file.
//
Bool CmReserveFile::write(long l)
{
  return write((void*) &l, sizeof(long));
}


// "write" writes one unsigned character to the file.
//
Bool CmReserveFile::write(unsigned char uc)
{
  return write((void*) &uc, sizeof(unsigned char));
}


// "write" writes one unsigned short to the file.
//
Bool CmReserveFile::write(unsigned short us)
{
  return write((void*) &us, sizeof(unsigned short));
}


// "write" writes one unsigned integer to the file.
//
Bool CmReserveFile::write(unsigned int ui)
{
  return write((void*) &ui, sizeof(unsigned int));
}


// "write" writes one unsigned long to the file.
//
Bool CmReserveFile::write(unsigned long ul)
{
  return write((void*) &ul, sizeof(unsigned long));
}


// "write" writes one float to the file.
//
Bool CmReserveFile::write(float f)
{
  return write((void*) &f, sizeof(float));
}


// "write" writes one double to the file.
//
Bool CmReserveFile::write(double d)
{
  return write((void*) &d, sizeof(double));
}


// "write" writes an array of characters to the file.
//
Bool CmReserveFile::write(char* c, unsigned u)
{
  return (write(c, sizeof(char), u) == u);
}


// "write" writes an array of shorts to the file.
//
Bool CmReserveFile::write(short* s, unsigned u)
{
  return (write(s, sizeof(short), u) == u);
}


// "write" writes an array of integers to the file.
//
Bool CmReserveFile::write(int* i, unsigned u)
{
  return (write(i, sizeof(int), u) == u);
}


// "write" writes an array of longs to the file.
//
Bool CmReserveFile::write(long* l, unsigned u)
{
  return (write(l, sizeof(long), u) == u);
}


// "write" writes an array of floats to the file.
//
Bool CmReserveFile::write(float* f, unsigned u)
{
  return (write(f, sizeof(float), u) == u);
}


// "write" writes an array of doubles to the file.
//
Bool CmReserveFile::write(double* d, unsigned u)
{
  return (write(d, sizeof(double), u) == u);
}


// "write" writes the specified character string length and the
// string text to the file.
//
Bool CmReserveFile::write(const char* s)
{
  int lnth = (s) ? strlen(s) + 1 : 0;
  if (!write(lnth)) return FALSE;
  return (lnth>0) ? write((char*) s, (unsigned) lnth) : TRUE;
}


// "read" reads a string length and string text from the file.  The
// output string must be deleted when use is finished.
//
char* CmReserveFile::read()
{
  int lnth;
  if (!read(lnth)) return FALSE;
  char* s = (lnth > 0) ? new char[lnth] : NULL;
  if (s) read(s, (unsigned) lnth);
  return s;
}


// "seek" sets the file position for this file.
//
int CmReserveFile::seek(long offset, int origin)
{
  return (_file) ? fseek(_file, offset, origin) : -1;
}


// "tell" returns the current file position for this file.
//
long CmReserveFile::tell()
{
  return (_file) ? ftell(_file) : -1L;
}


// "rewind" rewinds the file pointer to the beginning of the file.
//
void CmReserveFile::rewind()
{
  if (_file) ::rewind(_file);
}


// "getPos" records the current file position in ptr.
//
int CmReserveFile::getPos(fpos_t* ptr)
{
  return (_file) ? fgetpos(_file, ptr) : -1;
}


// "setPos" sets the file position to the position recorded by "getPos"
// in ptr.
//
int CmReserveFile::setPos(const fpos_t* ptr)
{
  return (_file) ? fsetpos(_file, ptr) : -1;
}


// "clearError" clears all error indicators for this file.
//
void CmReserveFile::clearError()
{
  if (_file) clearerr(_file);
}


// "eof" checks if the end of file indicator is set for this file.
//
Bool CmReserveFile::eof()
{
  return (!_file) ? -1 : ((feof(_file) != 0) ? TRUE : FALSE);
}


// "error" checks if any error indicators are set for this file.
//
Bool CmReserveFile::error()
{
  return (!_file) ? -1 : ((ferror(_file) != 0) ? TRUE : FALSE);
}


// "printError" prints the error message associated with the error
// indicator in the file and also the specified message.
//
void CmReserveFile::printError(const char* s)
{
  perror(s);
}


// "exists" checks if the specified file exists on the disk.
//
Bool CmReserveFile::exists(const char* filename)
{
  FILE *fp = fopen(filename, "r");
  if (!fp) return FALSE;
  fclose(fp);
  return TRUE;
}


// "remove" removes the named file from the disk.
//
Bool CmReserveFile::remove(const char* name)
{
  return (::remove(name) == 0) ? TRUE : FALSE;
}


// "tempName" returns a file name which does not currently exist on the
// disk for use as a temporary file name.
//
char* CmReserveFile::tempName(char* buff)
{
  return tmpnam(buff);
}
