// CmCont.cpp
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Abstract container implementation.
// -----------------------------------------------------------------

#include <cm/include/cmcont.h>
#include <cm/include/cmlist.h>


// "[]" indexing operator allows objects to be retrieved by index.
// This operator cannot be used as an lvalue.
//
CmObject* CmContainer::operator[](int idx) const
{
  if (idx < 0 || idx >= size()) return NULL;
  CmIterator *iterator = newIterator();
  CmObject   *out      = NULL;
  int         ii       = 0;
  while (!iterator->done() && ii <= idx)
  {
    if (ii++ == idx) out = iterator->current();
    iterator->next();
  }
  delete iterator;
  return out;
}


// "lookupAll" allocates and returns a container with pointers to
// all objects in this container equal to the specified object.
//
CmContainer* CmContainer::lookupAll(CmObject* pObj) const
{
  CmLinkedList *pList    = new CmLinkedList;
  CmIterator   *iterator = newIterator();

  pList->ownsObjects(FALSE);
  while (!iterator->done())
  {
    CmObject* temp = iterator->next();
    if (temp->isEqual(pObj)) pList->add(temp);
  }

  delete iterator;
  return pList;
}


// "addAllFrom" adds all the objects from the specified container
// to this container.
//
Bool CmContainer::addAllFrom(CmContainer* container)
{
  CmIterator *iterator = container->newIterator();
  Bool        success  = TRUE;
  while (!iterator->done() && success)
  {
    if (ownsObjects()) success = add(iterator->next()->newCopy());
    else               success = add(iterator->next());
  }
  delete iterator;
  return success;
}


// "printOn" prints the type of container and the container contents
// on the specified output stream.
//
void CmContainer::printOn(ostream& os) const
{
  os << "Type " << isA() << " contains -" << endl;
  CmIterator *iterator = newIterator();
  while (!iterator->done())
    os << *(iterator->next()) << endl;
  delete iterator;
}


// "write" writes the container contents to the specified file.
//
Bool CmContainer::write(CmReserveFile& file) const
{
  if (!writeBase(file)) return FALSE;              // Write base container.

  Bool        success  = TRUE;
  CmIterator *iterator = newIterator();
  while (!iterator->done() && success)             // Cycle through objects.
    success = iterator->next()->writeObject(file); // Write the object.
  delete iterator;
  return success;
}


// "read" reads objects from the specified file into this container.
//
Bool CmContainer::read(CmReserveFile& file)
{
  unsigned size = readBase(file);                 // Read base container.

  removeAll();                                    // Empty the container.

  int  ii      = 0;
  Bool success = TRUE;
  while (ii < size && success)                    // For each object.
  {
    CmObject *pObj = CmObject::readObject(file);  // Read object.
    if (pObj) success = add(pObj); ii++;          // Add to container.
  }
  return success;
}


// "forEach" iterates through all the objects in the container calling
// the specified function for each object until there are no more objects
// or the function returns FALSE.
//
void CmContainer::forEach(CmQueryFunc pFunc, void* pData)
{
  Bool        cont     = TRUE;
  CmIterator* iterator = newIterator();
  while (!iterator->done() && cont)
    cont = (*pFunc)(iterator->next(), pData);
  delete iterator;
}


// "forEach" iterates through all the objects in the container calling
// the specified object member function for each object until there are
// no more objects or the function returns FALSE.
//
void CmContainer::forEach(CmQueryMemFunc pFunc, void* pData)
{
  Bool        cont     = TRUE;
  CmIterator* iterator = newIterator();
  while (!iterator->done() && cont)
  {
    CmObject *pObj = iterator->next();
    cont = (pObj->*pFunc)(pData);
  }
  delete iterator;
}


// "firstThat" returns the first object in the container that is
// passed into the specified function and returns TRUE.
//
CmObject* CmContainer::firstThat(CmQueryFunc pFunc, void* pData)
{
  CmObject   *outObj   = NULL;
  CmIterator *iterator = newIterator();
  while (!iterator->done() && !outObj)
  {
    CmObject *pObj = iterator->next();
    if ((*pFunc)(pObj, pData)) outObj = pObj;
  }
  return outObj;
}


// "firstThat" returns the first object in the container who's specified
// member function returns TRUE.
//
CmObject* CmContainer::firstThat(CmQueryMemFunc pFunc, void* pData)
{
  CmObject   *outObj   = NULL;
  CmIterator *iterator = newIterator();
  while (!iterator->done() && !outObj)
  {
    CmObject *pObj = iterator->next();
    if ((pObj->*pFunc)(pData)) outObj = pObj;
  }
  return outObj;
}


// "query" cycles through all the objects in the container calling the
// specified function for each.  Every object in which the function returns
// returns TRUE gets added to a subset container which is then returned
// from the function.
//
CmContainer* CmContainer::query(CmQueryFunc pFunc, void* pData)
{
  CmLinkedList *pList    = new CmLinkedList;
  CmIterator   *iterator = newIterator();

  pList->ownsObjects(FALSE);
  while (!iterator->done())
  {
    CmObject* pObj = iterator->next();
    if ((*pFunc)(pObj, pData)) pList->add(pObj);
  }

  delete iterator;
  return pList;
}


// "query" cycles through all the objects in the container calling the
// specified object member function for each.  Every object in which the
// function returns returns TRUE gets added to a subset container which
// is then returned from the function.
//
CmContainer* CmContainer::query(CmQueryMemFunc pFunc, void* pData)
{
  CmLinkedList *pList    = new CmLinkedList;
  CmIterator   *iterator = newIterator();

  pList->ownsObjects(FALSE);
  while (!iterator->done())
  {
    CmObject* pObj = iterator->next();
    if ((pObj->*pFunc)(pData)) pList->add(pObj);
  }

  delete iterator;
  return pList;
}


// "copy" copies the contents from the specified container into this
// container.
//
void CmContainer::copy(const CmContainer& C)
{
  _ownsObjects = C._ownsObjects;
  CmIterator *iterator = C.newIterator();
  while (!iterator->done())
  {
    if (ownsObjects()) add(iterator->next()->newCopy());
    else               add(iterator->next());
  }
  delete iterator;
}


// "writeBase" writes the base container members to the specified file.
//
Bool CmContainer::writeBase(CmReserveFile& file) const
{
  Bool success = file.write(_ownsObjects);
  if (success) success = file.write(_size);
  return success;
}


// "readBase" reads the base container members from the specified file
// and returns the container size (the _size member is not set here).
//
unsigned CmContainer::readBase(CmReserveFile& file)
{
  unsigned size;
  Bool success = file.read(_ownsObjects);
  if (success) success = file.read(size);
  return (success) ? size : (unsigned) 0;
}
