// CmTCont.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Abstract container template definition.
// -----------------------------------------------------------------

#ifndef _CMTCONT_H
#define _CMTCONT_H

#include <cm/include/cmtiter.h>

template <class T> class CmTContainer {             // Container definition.
public:
  CmTContainer() : _error(* new T), _size(0) {}     // Default constructor.
  CmTContainer(const CmTContainer<T>&);             // Copy constructor.

  virtual ~CmTContainer();                          // Virtual destructor.

  virtual const T& operator[] (int) const;          // Get object by index.
  virtual int      size       () const;             // Return container size.
  virtual Bool     add        (const T&) = 0;       // Add object to container.
  virtual Bool     remove     (const T&) = 0;       // Remove object.
  virtual const T& lookup     (const T&) const = 0; // Find equal object.
  virtual Bool     contains   (const T&) const = 0; // Container has object?
  virtual unsigned occurrences(const T&) const = 0; // Count occurrences.
  virtual void     removeAll  ()         = 0;       // Remove all objects.
  virtual Bool     isEmpty    () const;             // Is container empty?

  virtual CmTIterator<T>* newIterator() const = 0;  // Get container iterator.

protected:
  void copy(const CmTContainer<T>&);                // Internal copy method.

  T&  _error;                                       // Error object.
  int _size;                                        // Container size.
};

#if defined(__TURBOC__) || defined(__xlC__)
#include <cm/include/cmtcont.cc>
#endif

// "CmTForEach" and "CmTEndFor" define a foreach loop where the item,
// type and collection are passed in and the item will be set equal to
// the next item in the collection for each loop iteration.
//
#define CmTForEach(obj, type, cltn) {                           \
        CmTIterator<type> *___cm_iterator = cltn.newIterator(); \
        while (!___cm_iterator->done()) {                       \
          obj = ___cm_iterator->next();
#define CmTEndFor } delete ___cm_iterator; }

// "CmTRepeat" and "CmTUntil" define a pascal style repeat/until loop
// where the item, type, and collection are passed in and the item will
// be set equal to next item in the collection for each loop iteration.
//
#define CmTRepeat(obj, type, cltn) {                            \
        CmTIterator<type> *___cm_iterator = cltn.newIterator(); \
        do {                                                    \
          obj = ___cm_iterator->next();
#define CmTUntil(condition) } while (!(condition)); }

// "CmTDo" and "CmTWhile" define a typical do/while loop where the item,
// type, and collection are passed in and the item will be set equal to
// the next item in the collection for each loop iteration.
//
#define CmTDo(obj, type, cltn) {                                \
        CmTIterator<type> *___cm_iterator = cltn.newIterator(); \
        do {                                                    \
          obj = ___cm_iterator->next();
#define CmTWhile(condition) } while (condition); }

#endif
