// CmTBTree.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// BTree template definition.
// -----------------------------------------------------------------

#ifndef _CMTBTREE_H
#define _CMTBTREE_H

#include <cm/include/cmtcont.h>

template <class T> class CmTBTreeNode;          // Tree node class stub.
template <class T> class CmTBTreeIterator;      // Tree iterator class stub.

template <class T> class CmTBTree : public CmTContainer<T> {
public:
  CmTBTree(unsigned = 3);                       // Default tree constructor.
  CmTBTree(const CmTBTree<T>&);                 // Tree copy constructor.
 ~CmTBTree();                                   // Tree destructor.

  CmTBTree<T>& operator=(const CmTBTree<T>&);   // Assignment operator.

  unsigned order () const;                      // Return tree order.
  unsigned height() const;                      // Return tree height.

  Bool     add        (const T&);               // Add item to tree.
  Bool     remove     (const T&);               // Remove item from tree.
  const T& lookup     (const T&) const;         // Find equal item in tree.
  Bool     contains   (const T&) const;         // See if tree contains equal.
  unsigned occurrences(const T&) const;         // Count occurrences of equal.
  void     removeAll  ();                       // Remove all items.

  CmTIterator<T>* newIterator() const;          // Create tree iterator.

protected:
  CmTBTreeNode<T>* nodeSearch(const T&) const; // Search for insertion node.
  CmTBTreeNode<T>* lookupNode(const T&) const; // Get node where item is.

  static void removeNodes(CmTBTreeNode<T>*);   // Remove node children.
  static unsigned minKeys(unsigned);           // Get minimum keys allowed.

  unsigned         _order;                     // Tree order.
  CmTBTreeNode<T> *_root;                      // Root tree node.
  friend           CmTBTreeIterator<T>;        // Iterator class can access.
};

template <class T> class CmTBTreeNode {        // Tree node definition.
protected:
  CmTBTreeNode(CmTBTreeNode<T>*, unsigned);    // Node constructor.
 ~CmTBTreeNode();                              // Node destructor.

  int  shouldGo(const T&);                     // Where should item insert.
  int  index   (const T&);                     // Index where item is located.
  int  index   (CmTBTreeNode<T>*);             // Index where child is located.
  Bool addAt   (int, const T&, CmTBTreeNode<T>*, int); // Add item at index.
  Bool removeAt(int);                          // Remove item at index.
  void clear   (int);                          // Clear arrays.

  CmTBTreeNode<T>* split(unsigned);            // Split the node.

  CmTBTreeNode<T>  *_parent;                   // Parent node.
  T                *_data;                     // (order  ) data items.
  CmTBTreeNode<T> **_children;                 // (order+1) child nodes.
  int               _total;                    // Number of items in node.
  friend            CmTBTree<T>;               // Tree class can access.
  friend            CmTBTreeIterator<T>;       // Iterator class can access.
};

template <class T> class CmTBTreeIterator : public CmTIterator<T> {
public:
  CmTBTreeIterator(const CmTBTree<T>&);        // Iterator constructor.

  Bool     done    () const;                   // Check if at end.
  const T& next    ();                         // Return and advance.
  const T& previous();                         // Return and backup.
  const T& current () const;                   // Get current item.
  void     first   ();                         // Move to first item.
  void     last    ();                         // Move to last item.

protected:
  const CmTBTree<T>      &_tree;               // Tree being iterated.
  CmTBTreeNode<T>        *_node;               // Current tree node.
  int                     _index;              // Current node object.
  friend                  CmTBTree<T>;         // Tree class can access.
};

#if defined(__TURBOC__) || defined(__xlC__)
#include <cm/include/cmtbtree.cc>
#endif

#endif
