// CmTArray.cc
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Array template implementation.
// -----------------------------------------------------------------


// "CmTArray" is the default array constructor.
//
template <class T> CmTArray<T>::CmTArray(unsigned sz, unsigned dt)
{
  _delta   = dt;
  _total   = 0;
  _entries = (sz > 0) ? new T[sz] : NULL;
  _size    = (_entries != NULL) ? sz : 0;
}


// "CmTArray" is the array copy constructor.
//
template <class T> CmTArray<T>::CmTArray(const CmTArray<T>& A)
                              : CmTContainer<T>(A)
{
  _total   = 0;
  _entries = NULL;
  copy(A);
}


// "~CmTArray" is the array destructor.
//
template <class T> CmTArray<T>::~CmTArray()
{
  delete[] _entries;
}


// "=" assignment operator copies the specified array into this array.
//
template <class T> CmTArray<T>& CmTArray<T>::operator=(const CmTArray<T>& A)
{
  if (&A != this) copy(A);
  return *this;
}


// "delta" sets a new delta value for automatic growing.
//
template <class T> void CmTArray<T>::delta(unsigned dt)
{
  _delta = dt;
}


// "delta" returns the current delta value.
//
template <class T> unsigned CmTArray<T>::delta() const
{
  return _delta;
}


// "total" returns the number of items in this array.
//
template <class T> int CmTArray<T>::total() const
{
  return _total;
}


// "at" returns the item at the specified index.
//
template <class T> const T& CmTArray<T>::at(int idx) const
{
  return _entries[idx];
}


// "[]" returns the item at the specified index.
//
template <class T> const T& CmTArray<T>::operator[](int idx) const
{
  return at(idx);
}


// "add" appends the specified item to this array.
//
template <class T> Bool CmTArray<T>::add(const T& rObj)
{
  if (_total >= _size)
    if (_delta == 0 || !resize(_size+_delta))
      return FALSE;
  _entries[_total++] = rObj;
  return TRUE;
}


// "insertAt" inserts the specified item at the specified index
// shifting the other items to make room.
//
template <class T> Bool CmTArray<T>::insertAt(int idx, const T& rObj)
{
  if (idx < 0 || idx >= _total) return FALSE;
  if (_total >= _size)
    if (_delta == 0 || !resize(_size+_delta))
      return FALSE;

  for (int ii = _total; ii > idx; ii--)
    _entries[ii] = _entries[ii-1];
  _entries[idx] = rObj;
  _total++;
  return TRUE;
}


// "replaceAt" replaces the item at the specified index with the
// specified item.
//
template <class T> Bool CmTArray<T>::replaceAt(int idx, const T& rObj)
{
  if (idx < 0 || idx >= _total) return FALSE;
  _entries[idx] = rObj;
  return TRUE;
}


// "remove" removes the first occurrence of an item equal to the specified
// item in the array.
//
template <class T> Bool CmTArray<T>::remove(const T& rObj)
{
  if (_total == 0) return FALSE;
  int  ii    = 0;
  Bool found = FALSE;
  while (ii < _total && !found)
    if (_entries[ii++] == rObj) found = TRUE;
  return (found) ? removeAt(ii-1) : FALSE;
}


// "removeAt" removes the item at the specified index.
//
template <class T> Bool CmTArray<T>::removeAt(int idx)
{
  if (idx < 0 || idx >= _total) return FALSE;

  for (int ii = idx; ii < _total-1; ii++)
    _entries[ii] = _entries[ii+1];
  _total--;
  return TRUE;
}


// "index" returns the index of the first occurrence of an item equal
// to the specified item.
//
template <class T> int CmTArray<T>::index(const T& rObj) const
{
  int  ii  = 0;
  int  idx = -1;
  while (ii < _total && idx == -1)
    if (_entries[ii++] == rObj) idx = ii-1;
  return idx;
}


// "lookup" returns the first occurrence of the item equal to the
// specified item in the array.
//
template <class T> const T& CmTArray<T>::lookup(const T& rObj) const
{
  int idx = index(rObj);
  return (idx > -1) ? _entries[idx] : _error;
}


// "contains" checks the array for an item equal to the input.
//
template <class T> Bool CmTArray<T>::contains(const T& rObj) const
{
  return (index(rObj) > -1) ? TRUE : FALSE;
}


// "occurrences" counts the number of occurrences of items equal to the
// specified item.
//
template <class T> unsigned CmTArray<T>::occurrences(const T& rObj) const
{
  int      ii  = 0;
  unsigned num = 0;
  while (ii < _total)
    if (_entries[ii++] == rObj) num++;
  return num;
}


// "removeAll" sets the number of items in this array to zero.
//
template <class T> void CmTArray<T>::removeAll()
{
  _total = 0;
}


// "resize" resizes the storage allocated for this array.
//
template <class T> Bool CmTArray<T>::resize(unsigned newSize)
{
  if (newSize == 0)
  {
    delete[] _entries;
    _entries = NULL;
    _size    = 0;
    _total   = 0;
    return TRUE;
  }

  T *newEntries = new T[newSize];
  if (!newEntries) return FALSE;

  if (newSize < _total) _total = newSize;

  for (int ii = 0; ii < _total; ii++)
    newEntries[ii] = _entries[ii];

  delete[] _entries;
  _entries = newEntries;
  _size    = newSize;
  return TRUE;
}


// "isEmpty" returns whether or not this array has any items in it.
//
template <class T> Bool CmTArray<T>::isEmpty() const
{
  return (_total == 0);
}


// "quickSort" uses the standard library to perform a quick sort
// on this array.
//
template <class T> void CmTArray<T>::quickSort()
{
  qsort(_entries, _total, sizeof(T), &CmTArray<T>::cmpObjs);
}


// "newIterator" creates and returns a new array iterator.
//
template <class T> CmTIterator<T>* CmTArray<T>::newIterator() const
{
  return new CmTArrayIterator<T>(*this);
}


// "copy" copies the items from the specified array into this array.
//
template <class T> void CmTArray<T>::copy(const CmTArray<T>& A)
{
  _size  = A._size;
  _total = A._total;
  _delta = A._delta;

  delete[] _entries;
  _entries = (_size > 0) ? new T[_size] : NULL;
  if (!_entries)
  {
    _total = 0;
    _size  = 0;
  }
  else
  {
    for (int ii = 0; ii < _total; ii++)
      _entries[ii] = A._entries[ii];
  }
}


// "cmpObjs" is called by the standard library qsort function for
// sorting the array into ascending order.
//
template <class T> int CmTArray<T>::cmpObjs(const void* obj1, const void* obj2)
{
  T* t1 = (T*) obj1;
  T* t2 = (T*) obj2;
  return ((*t1 == *t2) ? 0 : ((*t1 > *t2) ? 1 : -1));
}


// "CmTArrayIterator" is the iterator constructor.
//
template <class T> CmTArrayIterator<T>::CmTArrayIterator(const CmTArray<T>& A)
                                       : _array(A), _index(0)
{ }


// "done" returns whether or not this iterator can advance any further.
//
template <class T> Bool CmTArrayIterator<T>::done() const
{
  return (_index >= _array._total || _index < 0);
}


// "next" returns the current item in the array and advances the
// iterator to the next item.
//
template <class T> const T& CmTArrayIterator<T>::next()
{
  if (_index < _array._total) return _array._entries[_index++];
  else                        return _array._error;
}


// "previous" returns the current item in the array and decrements the
// iterator to the previous item.
//
template <class T> const T& CmTArrayIterator<T>::previous()
{
  if (_index >= 0) return _array._entries[_index--];
  else             return _array._error;
}


// "current" returns the current array item.
//
template <class T> const T& CmTArrayIterator<T>::current() const
{
  if (_index < _array._total) return _array._entries[_index];
  else                        return _array._error;
}


// "first" moves the iterator to the first item.
//
template <class T> void CmTArrayIterator<T>::first()
{
  _index = 0;
}


// "last" moves the iterator to the last item.
//
template <class T> void CmTArrayIterator<T>::last()
{
  _index = _array._total - 1;
}
