// CmString.h
// -----------------------------------------------------------------
// Compendium - C++ Container Class Library
// Copyright (C) 1992-1994, Glenn M. Poorman, All rights reserved
// -----------------------------------------------------------------
// Character string definition.
// -----------------------------------------------------------------

#ifndef _CMSTRING_H
#define _CMSTRING_H

#include <cm/include/cmobject.h>

class CmSubString;                                // Substring class stub.

class CmString : public CmObject {                // String class definition.
public:
  CmString(const char* = NULL);                   // Default constructor.
  CmString(int, char);                            // Construct with character.
  CmString(const CmSubString&);
  CmString(const CmString&);                      // Copy constructor.
 ~CmString();                                     // Destructor.

  CmString& operator=(const CmString&);           // Assignment operator.
  CmString& operator=(const char*);               // Assignment operator.

  char& operator[](int);                          // Return char at position.
  char  operator[](int) const;                    // Return char at position.

        CmSubString operator()(int, int);         // Substring operator.
  const CmSubString operator()(int, int) const;   // SubString operator.

  operator const char*() const;                   // Type conversion op.

  char*    text        () const;                  // Return char pointer.
  int      length      () const;                  // Return string length.
  int      index       (char) const;              // Return first occurrence.
  void     leftJustify ();                        // Left justify string.
  void     rightJustify();                        // Right justify string.
  void     trim        ();                        // Trim lead/trail blanks.
  void     toLower     ();                        // Lower case entire string.
  void     toUpper     ();                        // Upper case entire string.
  Bool     isEqual     (CmObject*) const;         // Compare objects.
  int      compare     (CmObject*) const;         // Compare objects.
  unsigned hash        (unsigned)  const;         // String hash function.
  void     printOn     (ostream&)  const;         // Print string to stream.
  void     readFrom    (istream&);                // Read string from stream.
  Bool     write       (CmReserveFile&) const;    // Write string to file.
  Bool     read        (CmReserveFile&);          // Read string from file.

  CMOBJECT_DEFINE(CmString, CmObject)             // Define object funcs.

  Bool      operator< (const CmString&) const;    // Check if input <  this.
  Bool      operator<=(const CmString&) const;    // Check if input <= this.
  Bool      operator> (const CmString&) const;    // Check if input >  this.
  Bool      operator>=(const CmString&) const;    // Check if input >= this.
  Bool      operator==(const CmString&) const;    // Check if input == this.
  Bool      operator!=(const CmString&) const;    // Check if input != this.
  CmString  operator+ (const CmString&) const;    // Append input to string.
  CmString& operator+=(const CmString&);          // Append input to string.

  Bool      operator< (const char*) const;        // Check if input <  this.
  Bool      operator<=(const char*) const;        // Check if input <= this.
  Bool      operator> (const char*) const;        // Check if input >  this.
  Bool      operator>=(const char*) const;        // Check if input >= this.
  Bool      operator==(const char*) const;        // Check if input == this.
  Bool      operator!=(const char*) const;        // Check if input != this.
  CmString  operator+ (const char*) const;        // Append input to string.
  CmString& operator+=(const char*);              // Append input to string.

  friend Bool     operator< (const char*, const CmString&); // Check if A <  B.
  friend Bool     operator<=(const char*, const CmString&); // Check if A <= B.
  friend Bool     operator> (const char*, const CmString&); // Check if A >  B.
  friend Bool     operator>=(const char*, const CmString&); // Check if A >= B.
  friend Bool     operator==(const char*, const CmString&); // Check if A == B.
  friend Bool     operator!=(const char*, const CmString&); // Check if A != B.
  friend CmString operator+ (const char*, const CmString&); // Append B to A.

  static void caseSensitive(Bool);                // Set case sensitive compare.
  static Bool caseSensitive();                    // Get case sensitive compare.

protected:
  void copy  (const char*);                       // Copy chars into string.
  void append(const CmString &AString);           // Append chars to string.

  char       *_text;                              // Character array.
  static Bool _caseSensitive;                     // Case sensitive compare.
  friend      CmSubString;                        // Substring can access.
};

class CmSubString {                               // Sub-string definition.
public:
  CmString& operator=(const CmString&);           // Assignment from string.
  CmString& operator=(const CmSubString&);        // Assignment from sub-str.
  CmString& operator=(const char*);               // Assignment from char*.

  operator const char*() const;                   // Type conversion.

  Bool operator< (const CmString&) const;         // See if input <  this.
  Bool operator<=(const CmString&) const;         // See if input <= this.
  Bool operator> (const CmString&) const;         // See if input >  this.
  Bool operator>=(const CmString&) const;         // See if input >= this.
  Bool operator==(const CmString&) const;         // See if input == this.
  Bool operator!=(const CmString&) const;         // See if input != this.

  Bool operator< (const CmSubString&) const;      // See if input <  this.
  Bool operator<=(const CmSubString&) const;      // See if input <= this.
  Bool operator> (const CmSubString&) const;      // See if input >  this.
  Bool operator>=(const CmSubString&) const;      // See if input >= this.
  Bool operator==(const CmSubString&) const;      // See if input == this.
  Bool operator!=(const CmSubString&) const;      // See if input != this.

  Bool operator< (const char*) const;             // See if input <  this.
  Bool operator<=(const char*) const;             // See if input <= this.
  Bool operator> (const char*) const;             // See if input >  this.
  Bool operator>=(const char*) const;             // See if input >= this.
  Bool operator==(const char*) const;             // See if input == this.
  Bool operator!=(const char*) const;             // See if input != this.

  friend Bool operator< (const char*, const CmSubString&); // See if A <  B.
  friend Bool operator<=(const char*, const CmSubString&); // See if A <= B.
  friend Bool operator> (const char*, const CmSubString&); // See if A >  B.
  friend Bool operator>=(const char*, const CmSubString&); // See if A >= B.
  friend Bool operator==(const char*, const CmSubString&); // See if A == B.
  friend Bool operator!=(const char*, const CmSubString&); // See if A != B.

  friend ostream& operator<<(ostream&, const CmSubString&);  // Stream output.

private:
  CmSubString(const CmString&, int, int);         // Sub-string constructor.
  CmSubString(const CmSubString&);                // Copy constructor.

  void replace(const char*, int);                 // Replace sub-str contents.

  char     *_sp;                                  // Sub string text.
  int       _sl;                                  // Length of sub string.
  CmString *_st;                                  // String this is sub of.
  friend    CmString;                             // String can access.
};

// "=" operator sets the contents of this string to that of an input string.
inline CmString& CmString::operator=(const CmString &AString)
{ if (&AString != this) copy(AString._text); return *this; }

// "=" operator sets the contents of this string to that of an input string.
inline CmString& CmString::operator=(const char* s)
{ if (s != _text) copy(s); return *this; }

// "[]" operator returns the character at a given index.
inline char& CmString::operator[](int idx)
{ return _text[idx]; }

// "[]" operator returns the character at a given index.
inline char CmString::operator[](int idx) const
{ return _text[idx]; }

// "operator const char*" returns the character pointer for this string.
inline CmString::operator const char*() const
{ return _text; }

// "text" returns the character pointer for this string.
inline char* CmString::text() const
{ return _text; }

// "length" returns the length of the character string.
inline int CmString::length() const
{ return (_text) ? strlen(_text) : 0; }

// "printOn" prints the string to the specified output stream.
inline void CmString::printOn(ostream& os) const
{ os << _text; }

// "operator const char*" returns the text pointer of this sub string.
inline CmSubString::operator const char*() const
{ return _sp; }

#endif
